
/*
 * DREADERD/GROUP.C
 *
 *	Group/Overview routines.  Group control and overview information is
 *	arranged in a two-level directory scheme by hash code under
 *	/news/spool/group.  The first level consists of 256 subdirectories.
 *	The group is stored as a KPDB database file in the second level.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

#define OVHSIZE		64
#define OVHMASK		(OVHSIZE-1)

#define HMAPALIGN	(1024 * 64)
#define HMAPSIZE	(1024 * 1024)

Prototype void NNFeedOverview(Connection *conn);
Prototype OverInfo *GetOverInfo(const char *group);
Prototype void PutOverInfo(OverInfo *ov);
Prototype const char *GetOverRecord(OverInfo *ov, int artno, int *plen);

Prototype int NNTestOverview(Connection *conn);
Prototype const char *NNRetrieveHead(Connection *conn, int *povlen, char **msgid);

void AssignArticleNo(Connection *conn, ArtNumAss **pan, const char *group, const char *xref, int approved, const char *art, int artLen);
void WriteOverview(Connection *conn, ArtNumAss *an, const char *group, const char *xref, const char *art, int artLen);
OverData *MakeOverHFile(OverInfo *ov, int artNo, int create);
void FreeOverInfo(OverInfo *ov);
void FreeOverData(OverData *od);

OverInfo *OvHash[OVHSIZE];
int	  NumOverInfo;

void
NNFeedOverview(Connection *conn)
{
    int artLen;
    int l = 0;
    char *art = MBNormalize(&conn->co_ArtBuf, &artLen);
    char *line;
    int hasXPGPSig = 0;
    int appr = 0;

    char *newsgroups = NULL;
    char *msgid = NULL;
    char *subject = NULL;
    char *date = NULL;
    char *xref = NULL;
    char *control = NULL;

    /*
     * Scan headers, look for Newsgroups: line, Subject:, Date:, From:, and
     * Message-Id:.  If any are missing, the article is bad.  If there is an
     * Xref: line, save that too and use it to calculate line numbers if 
     * Xref operation is enabled.
     */

    for (line = art; line < art + artLen; line += l + 1) {
	for (l = line - art; l < artLen; ++l) {
	    if (art[l] == '\n') {
		if (l + 1 >= artLen || (art[l+1] != ' ' && art[l+1] != '\t'))
		    break;
	    }
	}
	l -= line - art;

	if (l == 1 && line[0] == '\r') {
	    /* out of headers */
	    break;
	} else if (strncasecmp(line, "Newsgroups:", 11) == 0) {
	    newsgroups = zallocStrTrim(&conn->co_MemPool, line + 11, l - 11);
	} else if (strncasecmp(line, "Message-ID:", 11) == 0) {
	    msgid = zallocStrTrim(&conn->co_MemPool, line + 11, l - 11);
	} else if (strncasecmp(line, "Subject:", 8) == 0) {
	    subject = zallocStrTrim(&conn->co_MemPool, line + 8, l - 8);
	} else if (strncasecmp(line, "Date:", 5) == 0) {
	    date = zallocStrTrim(&conn->co_MemPool, line + 5, l - 5);
	} else if (strncasecmp(line, "Xref:", 5) == 0) {
	    xref = zallocStrTrim(&conn->co_MemPool, line + 5, l - 5);
	} else if (strncasecmp(line, "Control:", 8) == 0) {
	    control = zallocStrTrim(&conn->co_MemPool, line + 8, l - 8);
	} else if (strncasecmp(line, "X-PGP-Sig:", 10) == 0) {
	    hasXPGPSig = 1;
	} else if (strncasecmp(line, "Approved:", 9) == 0) {
	    appr = 1;
	}
    }

    if (newsgroups == NULL || msgid == NULL || subject == NULL || 
	date == NULL || strcmp(msgid, "<>") == 0
    ) {
	/*
	 * failure
	 */
	if (conn->co_Flags & COF_IHAVE) {
	    MBPrintf(&conn->co_TMBuf, "437 Rejected, headers missing\r\n");
	} else {
	    MBPrintf(&conn->co_TMBuf, "439 %s\r\n",  conn->co_IHaveMsgId);
	}
    } else {
	/*
	 * write out overview information
	 */
	char *group;
	char *ngroup = NULL;

	if ((conn->co_Flags & COF_WASCONTROL) == 0) {
	    ArtNumAss	*ANABase = NULL;

	    if (DebugOpt)
		printf("Feed overview %s %s\n", msgid, newsgroups);

	    /*
	     * pass 1 - assign article numbers
	     */

	    for (group = newsgroups; *group; group = ngroup) {
		char c;

		for (ngroup = group; *ngroup && *ngroup != ','; ++ngroup)
		   ;
		c = *ngroup;
		*ngroup = 0;

		AssignArticleNo(conn, &ANABase, group, xref, appr, art, artLen);

		*ngroup = c;
		if (*ngroup == ',')
		    ++ngroup;
	    }
	    for (group = newsgroups; *group; group = ngroup) {
		char c;

		for (ngroup = group; *ngroup && *ngroup != ','; ++ngroup)
		   ;
		c = *ngroup;
		*ngroup = 0;

		WriteOverview(conn, ANABase, group, xref, art, artLen);

		*ngroup = c;
		if (*ngroup == ',')
		    ++ngroup;
	    }
	    while (ANABase) {
		ArtNumAss *an = ANABase;
		ANABase = an->an_Next;
		zfree(&conn->co_MemPool, an, sizeof(ArtNumAss));
	    }
	} else {
	    if (DebugOpt)
		printf("Control message: %s %s\n", msgid, newsgroups);
	    ExecuteControl(conn, control, art, artLen, hasXPGPSig);
	}

	if (conn->co_Flags & COF_IHAVE) {
	    MBPrintf(&conn->co_TMBuf, "235\r\n");
	} else {
	    MBPrintf(&conn->co_TMBuf, "239 %s\r\n",  conn->co_IHaveMsgId);
	}
    }

    zfreeStr(&conn->co_MemPool, &newsgroups);
    zfreeStr(&conn->co_MemPool, &msgid);
    zfreeStr(&conn->co_MemPool, &subject);
    zfreeStr(&conn->co_MemPool, &date);
    zfreeStr(&conn->co_MemPool, &xref);
    zfreeStr(&conn->co_MemPool, &control);
    zfreeStr(&conn->co_MemPool, &conn->co_IHaveMsgId);

    MBFree(&conn->co_ArtBuf);
    NNCommand(conn);
}

void
AssignArticleNo(Connection *conn, ArtNumAss **pan, const char *group, const char *xref, int approved, const char *art, int artLen)
{
    const char *rec;
    int recLen;
    int groupLen = strlen(group);
    int activeArtBeg;
    int activeArtEnd;
    int aabegchanged = 0;
    char aabegbuf[16];
    char aaendbuf[16];
    int artNo;

    /*
     * locate group in active file and lock
     */

    if ((rec = KPDBReadRecord(KDBActive, group, KP_LOCK, &recLen)) == NULL)
	return;

    /*
     * silently drop postings to moderated groups that do not have an
     * approved header.
     */

    if (approved == 0) {
	int flagsLen;
	const char *flags = KPDBGetField(rec, recLen, "S", &flagsLen, "y");

	while (flagsLen > 0) {
	    if (*flags == 'm') {
		KPDBUnlock(KDBActive, rec);
		return;
	    }
	    --flagsLen;
	    ++flags;
	}
    }

    /*
     * assign article number.  Locate Xref: line if Xref's are enabled
     */

    activeArtEnd = strtol(KPDBGetField(rec, recLen, "NE", NULL, "-1"), NULL,10);
    activeArtBeg = strtol(KPDBGetField(rec, recLen, "NB", NULL, "-1"), NULL,10);
    artNo = activeArtEnd + 1;

    if (xref) {
	const char *test;
	for (test = strchr(xref, ' '); test; test = strchr(test, ' ')) {
	    ++test;
	    if (strncmp(test, group, groupLen) == 0 && test[groupLen] == ':') {
		artNo = strtol(test + groupLen + 1, NULL, 10);
		break;
	    }
	}
    }
    if (artNo < 1)
	artNo = 1;

    if (activeArtEnd < artNo) {
	activeArtEnd = artNo;
	if (activeArtBeg > activeArtEnd) {
	    activeArtBeg = activeArtEnd;
	    aabegchanged = 1;
	}
    } else if (activeArtBeg > artNo) {
	activeArtBeg = artNo;
	aabegchanged = 1;
	if (activeArtEnd < activeArtBeg)
	    activeArtEnd = activeArtBeg;
    } else if (activeArtBeg > activeArtEnd) {
	activeArtBeg = activeArtEnd = artNo;
	aabegchanged = 1;
    }

    sprintf(aabegbuf, "%010d", activeArtBeg);
    sprintf(aaendbuf, "%010d", activeArtEnd);

    if (aabegchanged)
	KPDBWrite(KDBActive, group, "NB", aabegbuf, KP_LOCK_CONTINUE);	/* continuing lock */
    KPDBWrite(KDBActive, group, "NE", aaendbuf, KP_UNLOCK);		/* and unlock 	   */

    {
	ArtNumAss *an = zalloc(&conn->co_MemPool, sizeof(ArtNumAss));
	an->an_Next = *pan;
	*pan = an;
	an->an_GroupName = group;
	an->an_GroupLen = strlen(group);
	an->an_ArtNo = artNo;
    }
}

void
WriteOverview(Connection *conn, ArtNumAss *an, const char *group, const char *xref, const char *art, int artLen)
{
    int artNo = 0;
    const char *body;
    char *xtmp = NULL;
    int xtmpLen = 16 + strlen(ReportedHostName);

    /*
     * Locate article number assignment
     */
    {
	ArtNumAss *scan;

	for (scan = an; scan; scan = scan->an_Next) {
	    if (scan->an_GroupName == group) {
		artNo = scan->an_ArtNo;
	    }
	    xtmpLen += scan->an_GroupLen + 15;
	}
    }

    if (artNo == 0)
	return;

    /*
     * Locate start of body (we may have to append our own Xref: line)
     */

    {
	int l;
	int lnl = 1;

	for (l = 0; l < artLen; ++l) {
	    /*
	     * blank line terminates headers
	     */
	    if (art[l] == '\r' && (l + 1 >= artLen || art[l+1] == '\n')) {
		if (lnl)
		    break;
	    }
	    lnl = 0;
	    if (art[l] == '\n')
		lnl = 1;
	}
	body = art + l;
    }

    /*
     * Write overview record.
     */

    {
	off_t pos;
	OverInfo *ov;
	int actLen = 0;
	int iovLen = 0;
	struct iovec iov[3];

	if ((ov = GetOverInfo(group)) == NULL)
	    return;

	if (MakeOverHFile(ov, artNo, 1) == NULL)
	    return;

	hflock(ov->ov_HCache->od_HFd, 0, XLOCK_EX);
	pos = lseek(ov->ov_HCache->od_HFd, 0L, 2);

	errno = 0;

	if (xref) {
	    iov[0].iov_base = (void *)art;
	    iov[0].iov_len = artLen + 1;
	    iovLen = 1;
	} else {
	    ArtNumAss *scan;
	    int soff;
	    
	    xtmp = zalloc(&conn->co_MemPool, xtmpLen);
	    sprintf(xtmp, "Xref: %s", ReportedHostName);
	    soff = strlen(xtmp);
		
	    for (scan = an; scan; scan = scan->an_Next) {
		xtmp[soff++] = ' ';
		memcpy(xtmp + soff, scan->an_GroupName, scan->an_GroupLen);
		soff += scan->an_GroupLen;
		sprintf(xtmp + soff, ":%d", scan->an_ArtNo);
		soff += strlen(xtmp + soff);
	    }
	    sprintf(xtmp + soff, "\r\n");
	    soff += 2;
	    iov[0].iov_base = (void *)art;
	    iov[0].iov_len = body - art;
	    iov[1].iov_base = xtmp;
	    iov[1].iov_len = soff;
	    iov[2].iov_base = (void *)body;
	    iov[2].iov_len = (art + artLen + 1) - body;
	    iovLen = 3;
	}
	{
	    int i;
	    for (i = 0; i < iovLen; ++i)
		actLen += iov[i].iov_len;
	}
	if (writev(ov->ov_HCache->od_HFd, iov, iovLen) == actLen) {
	    OverArt ovart = { 0 };
	    off_t ovpos = ((artNo & 0x7FFFFFFF) % ov->ov_MaxArts) * sizeof(OverArt);
	    ovart.oa_ArtNo = artNo;
	    ovart.oa_SeekPos = pos;
	    ovart.oa_Bytes = actLen - 1;	/* do not include \0 */

	    lseek(ov->ov_OFd, ovpos, 0);
	    write(ov->ov_OFd, &ovart, sizeof(ovart));
	} else {
	    ftruncate(ov->ov_HCache->od_HFd, pos);
	    logit(LOG_ERR, "error writing overview data file for %s", group);
	}
	hflock(ov->ov_HCache->od_HFd, 0, XLOCK_UN);
	PutOverInfo(ov);
    }
    if (xtmp)
	zfree(&conn->co_MemPool, xtmp, xtmpLen);
}

OverInfo *
GetOverInfo(const char *group)
{
    OverInfo **pov = &OvHash[shash(group) & OVHMASK];
    OverInfo *ov;

    while ((ov = *pov) != NULL) {
	if (strcmp(group, ov->ov_Group) == 0)
	    break;
	pov = &ov->ov_Next;
    }
    if (ov == NULL) {
	struct stat st;
	char *path;

	bzero(&st, sizeof(st));

	/*
	 * If our cache has grown too large, attempt to free up
	 * a bunch of overview structures.  Depending on the load,
	 * we may not be able to.
	 */

	if (NumOverInfo >= OV_CACHE_MAX) {
	    int i;
	    int freeup = OV_CACHE_MAX / 2;
	    static int OI = 0;

	    for (i = 0; i < OVHSIZE && freeup; ++i) {
		int ai = OI;
		OI = (ai + 1) & OVHMASK;

		pov = &OvHash[ai];
		while ((ov = *pov) != NULL) {
		    if (ov->ov_Refs == 0) {
			*pov = ov->ov_Next;
			FreeOverInfo(ov);
			--NumOverInfo;
			--freeup;
		    } else {
			pov = &ov->ov_Next;
		    }
		}
	    }
	}
	ov = zalloc(&SysMemPool, sizeof(OverInfo));
	ov->ov_Group = zallocStr(&SysMemPool, group);

	path = zalloc(&SysMemPool, strlen(OverviewHome) + 32);
	{
	    const char *gfname = GFName(group, "over", 0);
	    const char *slash  = strchr(gfname, '/');

	    sprintf(path, "%s/%*.*s", 
		OverviewHome,
		slash - gfname, slash - gfname, gfname
	    );
	    if (stat(path, &st) < 0)
		mkdir(path, 0755);
	    errno = 0;
	    ov->ov_OFd = xopen(O_RDWR|O_CREAT, 0644, "%s%s", path, slash);
	}
	if (ov->ov_OFd < 0 || fstat(ov->ov_OFd, &st) < 0) {
	    logit(LOG_ERR, "Error on overview open/create for group %s", group);
	    FreeOverInfo(ov);
	    ov = NULL;
	} else {
	    /*
	     * check if new overview file and size accordingly
	     */
	    if (st.st_size == 0) {
		hflock(ov->ov_OFd, 0, XLOCK_EX);
		fstat(ov->ov_OFd, &st);
		if (st.st_size == 0) {
		    st.st_size = sizeof(OverArt) * DEFARTSINGROUP;
		    ftruncate(ov->ov_OFd, st.st_size);
		}
		hflock(ov->ov_OFd, 0, XLOCK_UN);
	    }
	    ov->ov_Size = st.st_size;
	    ov->ov_MaxArts = st.st_size / sizeof(OverArt);
	    ++NumOverInfo;
	    pov = &OvHash[shash(group) & OVHMASK];
	    ov->ov_Next = *pov;
	    *pov = ov;
	}
	zfree(&SysMemPool, path, strlen(OverviewHome) + 32);
    }
    if (ov)
	++ov->ov_Refs;
    return(ov);
}

const char *
GetOverRecord(OverInfo *ov, int artno, int *plen)
{
    int ovpos = ((artno & 0x7FFFFFFF) % ov->ov_MaxArts) * sizeof(OverArt);
    int hvpos;
    OverArt *oa;
    OverData *od;

    if (DebugOpt)
	printf("MaxArts %d\n", ov->ov_MaxArts);

    /*
     * memory map the overview array
     */

    if (
	ov->ov_OMapBase == NULL || 
	ovpos < ov->ov_OMapPos || 
	ovpos + sizeof(OverArt) > ov->ov_OMapPos + ov->ov_OMapBytes
    ) {
	if (ov->ov_OMapBase)
	    xunmap((void *)ov->ov_OMapBase, ov->ov_OMapBytes);
	ov->ov_OMapBytes = ov->ov_Size;
	ov->ov_OMapPos = 0;
	ov->ov_OMapBase = xmap(NULL, ov->ov_OMapBytes, PROT_READ, MAP_SHARED, ov->ov_OFd, ov->ov_OMapPos);
	if (ov->ov_OMapBase == NULL) {
	    logit(LOG_CRIT, "mmap() failed A %s", strerror(errno));
	    exit(1);
	}
    }

    /*
     * memory map the overview data.  Check overview record to
     * see if we actually have the requested information.
     */

    oa = (OverArt *)(ov->ov_OMapBase + ovpos - ov->ov_OMapPos);

    if (DebugOpt)
	printf("OA %08lx %d,%d\n", (long)oa, oa->oa_ArtNo, artno);

    if (oa->oa_ArtNo != artno) {
	if (plen)
	    *plen = 0;
	return(NULL);
    }

    if (plen == NULL) {
	return((const char *)1);
    }

    if ((od = MakeOverHFile(ov, artno, 0)) == NULL)
	return(NULL);

    hvpos = oa->oa_SeekPos;

    if (
	od->od_HMapBase == NULL || 
	hvpos < od->od_HMapPos || 
	hvpos + oa->oa_Bytes > od->od_HMapPos + od->od_HMapBytes
    ) {
	struct stat st;

	if (od->od_HMapBase) {
	    xunmap((void *)od->od_HMapBase, od->od_HMapBytes);
	    od->od_HMapBase = NULL;
	    od->od_HMapBytes = 0;
	    od->od_HMapPos = 0;
	}

	st.st_size = 0;
	fstat(od->od_HFd, &st);

	/*
	 * Make sure the file is big enough to map requested header.  It
	 * is possible for it to not be.
	 */

	if (hvpos + oa->oa_Bytes > st.st_size)
	    return(NULL);

	if (hvpos > HMAPSIZE/2)
	    od->od_HMapPos = (hvpos - HMAPSIZE/2) & ~(HMAPALIGN-1);
	else
	    od->od_HMapPos = hvpos & ~(HMAPALIGN-1);
	od->od_HMapBytes = HMAPSIZE;
	if (od->od_HMapBytes + od->od_HMapPos > st.st_size)
	    od->od_HMapBytes = st.st_size - od->od_HMapPos;

	od->od_HMapBase = xmap(NULL, od->od_HMapBytes, PROT_READ, MAP_SHARED, od->od_HFd, od->od_HMapPos);
	if (od->od_HMapBase == NULL) {
	    logit(LOG_CRIT, "mmap() failed B %s", strerror(errno));
	    exit(1);
	}
    }
    *plen = oa->oa_Bytes;
    return(od->od_HMapBase + hvpos - od->od_HMapPos);
}

OverData *
MakeOverHFile(OverInfo *ov, int artNo, int create)
{
    int artBase = artNo & ~OD_HMASK;
    OverData **pod;
    OverData *od;
    int count = 0;

    if (create)
	create = O_CREAT;

    if (ov->ov_HCache && artBase == ov->ov_HCache->od_ArtBase)
	return(ov->ov_HCache);
    for (pod = &ov->ov_HData; (od = *pod) != NULL; pod = &od->od_Next) {
	if (artBase == od->od_ArtBase)
	    break;
	++count;
    }
    if (od == NULL) {
	const char *gfname = GFName(ov->ov_Group, "data", artBase);

	*pod = od = zalloc(&SysMemPool, sizeof(OverData));
        errno = 0;
	od->od_HFd = xopen(O_RDWR|create, 0644, "%s/%s", OverviewHome, gfname);
	if (od->od_HFd < 0) {
	    if (create) {
		logit(LOG_ERR, "Unable to open/create %s/%s: %s",
		    OverviewHome,
		    gfname,
		    strerror(errno)
		);
	    }
	    FreeOverData(od);
	    *pod = od = NULL;
	} else {
	    od->od_ArtBase = artBase;
	    if (count > MAX_OVERVIEW_CACHE_REGIONS) {
		OverData *t = ov->ov_HData;
		ov->ov_HData = t->od_Next;
		FreeOverData(t);
	    }
	}
    }
    ov->ov_HCache = od;
    return(od);
}

/*
 * PutOverInfo() - release the ref count, but do not immediately unmap or 
 *		   free the data (other routines depend on this).
 */

void
PutOverInfo(OverInfo *ov)
{
    --ov->ov_Refs;
}

void
FreeOverInfo(OverInfo *ov)
{
    OverData *od;

    while ((od = ov->ov_HData) != NULL) {
	ov->ov_HData = od->od_Next;
	FreeOverData(od);
    }
    if (ov->ov_OMapBase)
	xunmap((void *)ov->ov_OMapBase, ov->ov_OMapBytes);
    if (ov->ov_OFd >= 0)
	close(ov->ov_OFd);
    zfreeStr(&SysMemPool, &ov->ov_Group);
    bzero(ov, sizeof(OverInfo));
    zfree(&SysMemPool, ov, sizeof(OverInfo));
}

void
FreeOverData(OverData *od)
{
    if (od->od_HMapBase) {
	xunmap((void *)od->od_HMapBase, od->od_HMapBytes);
	od->od_HMapBase = NULL;
    }
    if (od->od_HFd >= 0) {
	close(od->od_HFd);
	od->od_HFd = -1;
    }
    zfree(&SysMemPool, od, sizeof(OverData));
}

int
NNTestOverview(Connection *conn)
{
    OverInfo *ov;
    int r = -1;

    if ((ov = GetOverInfo(conn->co_GroupName)) != NULL) {
	if (GetOverRecord(ov, conn->co_ArtNo, NULL) != NULL)
	    r = 0;
	PutOverInfo(ov);
    }
    return(r);
}

const char *
NNRetrieveHead(Connection *conn, int *povlen, char **pmsgid)
{
    OverInfo *ov;
    const char *res = NULL;

    *povlen = 0;
    *pmsgid = "<>";

    if ((ov = GetOverInfo(conn->co_GroupName)) != NULL) {
	if ((res = GetOverRecord(ov, conn->co_ArtNo, povlen)) != NULL) {
	    /*
	     * Locate the Message-ID
	     */
	    const char *scan = res;
	    int scanLen = *povlen;

	    while (scanLen > 0) {
		int i;
		for (i = 0; i < scanLen && scan[i] != '\n'; ++i)
		    ;
		if (strncasecmp(scan, "Message-ID:", 11) == 0) {
		    int b = 11;
		    int e;
		    char buf[MAXMSGIDLEN];

		    while (b < scanLen && (scan[b] == ' ' || scan[b] == '\t'))
			++b;
		    e = b;
		    while (e < scanLen && (scan[e] != '>'))
			++e;
		    if (e < scanLen)
			++e;
		    if (e - b < MAXMSGIDLEN) {
			bcopy(scan + b, buf, e - b);
			buf[e-b] = 0;
			*pmsgid = MsgId(buf);
		    }
		    break;
		}
		if (i == 1 && scan[0] == '\r')
		    break;
		if (i < scanLen)
		    ++i;
		scanLen -= i;
		scan += i;
	    }
	} 
	PutOverInfo(ov);
    }
    if (strcmp(*pmsgid, "<>") == 0)
	res = NULL;
    return(res);
}

