
/*
 * POST.C
 *
 *	Post an article.  The article is posted to one or several of the
 *	upstream servers depending on prioritization and the existence of
 *	the 'P'ost flag.  
 *
 *	The article is loaded into memory prior to posting.  If no servers
 *	are available, the article is written to a local posting queue and
 *	the post is retried later.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void NNPostBuffer(Connection *conn);
Prototype void NNPostCommand1(Connection *conn);

void NNPostResponse1(Connection *conn);
void NNPostArticle1(Connection *conn);
void NNPostResponse2(Connection *conn);

/*
 * NNPostBuffer() - called when client connection post buffer contains the
 *		    entire article.  We need to check the article headers
 *		    and post it as appropriate.  If we are unable to post it
 *		    immediately, we queue it.
 *
 * STEPS:
 *	(1) remove headers otherwise generated internally (for security)
 *	(2) add internally generated headers
 *	
 * Internally generated headers:
 *	Path:			(removed, generated)
 *	Date:			(removed, generated)
 *	Lines:			(removed, generated)
 *	Message-ID:		(checked, generated)
 *	NNTP-Posting-Host:	(removed, generated)
 *	X-Trace:		(removed, generated)
 *	Xref:			(removed, RIP)
 *
 * Errors:
 *	441 Article has no body -- just headers
 *	441 Required "XXX" header is missing (Subject, From, Newsgroups)
 *	441 435 Bad Message-ID
 */

void
NNPostBuffer(Connection *conn)
{
    int artLen;
    int haveMsgId = 0;
    int haveSubject = 0;
    int haveFrom = 0;
    int haveDate = 0;
    int haveNewsgroups = 0;
    const char *errMsg = NULL;
    char *base = MBNormalize(&conn->co_ArtBuf, &artLen);
    char *msgid = NULL;
    int	msgidAllocLen = 0;
    MBufHead	tmbuf;

    MBInit(&tmbuf, -1, &conn->co_MemPool, &conn->co_BufPool);

    /*
     * Output Path: line
     */

    MBPrintf(&tmbuf, "Path: %s%snot-for-mail\r\n",
	PathHost,
	(PathHost[0] ? "!" : "")
    );

    /*
     * Scan Headers, check for required headers
     */

    while (artLen > 1 && base[0] != '\r' && base[1] != '\n') {
	int l;
	int keep = 1;

	/*
	 * extract header.  Headers can be multi-line so deal with that.
	 */

	for (l = 0; l < artLen; ++l) {
	    if (base[l] == '\r' && 
		base[l+1] == '\n' &&
		(l + 2 >= artLen || (base[l+2] != ' ' && base[l+2] != '\t'))
	    ) {
		l += 2;
		break;
	    }
	}

	/*
	 * check against specials
	 */
	if (strncasecmp(base, "Subject:", 8) == 0) {
	    haveSubject = 1;
	} else if (strncasecmp(base, "From:", 5) == 0) {
	    haveFrom = 1;
	} else if (strncasecmp(base, "Newsgroups:", 11) == 0) {
	    haveNewsgroups = 1;
	} else if (strncasecmp(base, "Date:", 5) == 0) {
	    haveDate = 1;
	} else if (strncasecmp(base, "Path:", 5) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Date:", 5) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Lines:", 6) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Message-ID:", 11) == 0) {
	    int save = base[l-2];

	    base[l-2] = 0;
	    if (strcmp((msgid = MsgId(base + 11)), "<>") == 0 ||
		strchr(base, '@') == NULL
	    ) {
		keep = 0;	/* delete */
		errMsg = "441 435 Bad Message-ID\r\n";
		msgid = NULL;
	    } else {
		keep = 1;
		haveMsgId = 1;
		msgidAllocLen = strlen(msgid) + 1;
		msgid = strcpy(zalloc(&conn->co_MemPool, msgidAllocLen), msgid);
	    }
	    base[l-2] = save;
	} else if (strncasecmp(base, "NNTP-Posting-Host:", 18) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "X-Trace:", 8) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Xref:", 5) == 0) {
	    keep = 0;	/* delete */
	}
	if (keep)
	    MBWrite(&tmbuf, base, l);
	base += l;
	artLen -= l;
    }

    /*
     * Check errors
     */

    if (haveNewsgroups == 0)
	errMsg = "441 Required \"Newsgroups\" header is missing\r\n";
    if (haveFrom == 0)
	errMsg = "441 Required \"From\" header is missing\r\n";
    if (haveSubject == 0)
	errMsg = "441 Required \"Subject\" header is missing\r\n";
    if (artLen <= 2)
	errMsg = "441 Article has no body -- just headers\r\n";

    /*
     * If no error, add our own headers
     */
    if (errMsg == NULL) {
	time_t t = time(NULL);

	/*
	 * Add Date:
	 */
	if (haveDate == 0) {
	    struct tm *tp = gmtime(&t);
	    char buf[64];

	    strftime(buf, sizeof(buf), "%d %b %Y %H:%M:%S GMT", tp);
	    MBPrintf(&tmbuf, "Date: %s\r\n", buf);
	}
	/*
	 * Add Lines:
	 */
	{
	    int lines = 0;
	    int i;

	    for (i = 2; i < artLen; ++i) {
		if (base[i] == '\n')
		    ++lines;
	    }
	    if (base[i-1] != '\n')	/* last line not terminated? */
		++lines;
	    MBPrintf(&tmbuf, "Lines: %d\r\n", lines);
	}
	/*
	 * Add Message-ID:
	 */
	if (haveMsgId == 0) {
	    static time_t LastT;
	    static int MsgIdCounter;

	    if (LastT != t) {
		LastT = t;
		MsgIdCounter = 0;
	    }
	    msgidAllocLen = strlen(ReportedHostName + 40);
	    msgid = zalloc(&conn->co_MemPool, msgidAllocLen);

	    sprintf(msgid, "<%08lx$%d$%d@%s>",
		(long)t,
		MsgIdCounter,
		(int)getpid(),
		ReportedHostName
	    );
	    MBPrintf(&tmbuf, "Message-ID: %s\r\n", msgid);
	    ++MsgIdCounter;
	}
	/*
	 * Add NNTP-Posting-Host:
	 */
	MBPrintf(&tmbuf, "NNTP-Posting-Host: %s\r\n", conn->co_Auth.dr_Host);

	/*
	 * Add X-Trace:
	 */
	MBPrintf(&tmbuf, "X-Trace: %lu %d %s %s\r\n",
	    (unsigned long)t,
	    (int)getpid(),		/* should probably be slot id */
	    (conn->co_Auth.dr_User[0] ? conn->co_Auth.dr_User : "(none)"),
	    inet_ntoa(conn->co_Auth.dr_Addr)
	);
    }

    /*
     * blank line and article body
     */

    if (errMsg == NULL) {
	MBWrite(&tmbuf, base, artLen);
    }
    MBFree(&conn->co_ArtBuf);

    /*
     * If no error, pump out to ready server or queue for transmission, 
     * else write the error out and throw away.
     */

    if (errMsg == NULL) {
	MBCopy(&tmbuf, &conn->co_ArtBuf);
	MBFree(&tmbuf);
	NNServerRequest(conn, NULL, msgid, NULL, SREQ_POST);
    } else {
	MBWrite(&conn->co_TMBuf, errMsg, strlen(errMsg));
	MBFree(&tmbuf);
	NNCommand(conn);
    }
    if (msgid)
	zfree(&conn->co_MemPool, msgid, msgidAllocLen);
}

/*
 * NNPostCommand1() - called when article has been queued to a server 
 *		      connection, in the server's context, to post the
 *		      article.
 *
 * SEQUENCE:
 */

void
NNPostCommand1(Connection *conn)
{
    ServReq *sreq = conn->co_SReq;

    MBPrintf(&conn->co_TMBuf, "ihave %s\r\n", sreq->sr_MsgId);
    NNPostResponse1(conn);
}

void
NNPostResponse1(Connection *conn)
{
    /*ServReq *sreq = conn->co_SReq;*/
    char *buf;
    int len;

    conn->co_Func = NNPostResponse1;
    conn->co_State = "pores1";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) > 0) {
	if (strtol(buf, NULL, 10) == 335) {
	    NNPostArticle1(conn);
	} else {
	    char errMsg[256];

	    MBFree(&conn->co_ArtBuf);

	    if (len >= 1 && buf[len-1] == 0)
		--len;
	    if (len >= 1 && buf[len-1] == '\r')
		--len;
	    if (len > sizeof(errMsg) - 32)
		len = sizeof(errMsg) - 32;
	    strcpy(errMsg, "441 ");
	    bcopy(buf, errMsg + 4, len);
	    errMsg[4+len] = '\r';
	    errMsg[5+len] = '\n';
	    errMsg[6+len] = 0;
	    NNFinishSReq(conn, errMsg);
	}
    } else if (len < 0) {
	NNServerTerminate(conn);
    } /* else we haven't gotten the reponse yet */
}

void
NNPostArticle1(Connection *conn)
{
    ServReq *sreq = conn->co_SReq;

    conn->co_Func = NNPostArticle1;
    conn->co_State = "poart1";

    while (conn->co_TMBuf.mh_Bytes < 800) {
	char *buf;
	int len;

	if ((len = MBReadLine(&sreq->sr_CConn->co_ArtBuf, &buf)) > 0) {
	    buf[len-1] = '\n';
	    MBWrite(&conn->co_TMBuf, buf, len);
	} else if (len <= 0) {
	    MBPrintf(&conn->co_TMBuf, ".\r\n");
	    NNPostResponse2(conn);
	    return;
	}
    }
}

void
NNPostResponse2(Connection *conn)
{
    /*ServReq *sreq = conn->co_SReq;*/
    char *buf;
    int len;

    conn->co_Func = NNPostResponse2;
    conn->co_State = "pores2";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) > 0) {
	if (strtol(buf, NULL, 10) == 235) {
	    NNFinishSReq(conn, "240 Article posted\r\n");
	} else {
	    char errMsg[256];

	    MBFree(&conn->co_ArtBuf);

	    if (len >= 1 && buf[len-1] == 0)
		--len;
	    if (len >= 1 && buf[len-1] == '\r')
		--len;
	    if (len > sizeof(errMsg) - 32)
		len = sizeof(errMsg) - 32;
	    strcpy(errMsg, "441 ");
	    bcopy(buf, errMsg + 4, len);
	    errMsg[4+len] = '\r';
	    errMsg[5+len] = '\n';
	    errMsg[6+len] = 0;
	    NNFinishSReq(conn, errMsg);
	}
    } else if (len < 0) {
	NNServerTerminate(conn);
    } /* else we haven't gotten the reponse yet */
}

