#!/usr/local/bin/perl

################################################################################
# A contribution to nntpcache (ftp://ftp.nntpcache.org/pub/nntpcache/nntpcache.tgz)
#
# Make an extended protocol based on the NNTP Cache logfiles.
# Mail the protocol to the news admin.
#
# (C) Frank Schnekenbuehl frank@comsys.dofn.de
#
# $Id: p.pl,v 1.1.1.1 1998/07/29 15:14:31 proff Exp $
#
# External programs: Perl 4.x, Mail, gzip
#
################################################################################

################################################################################
# Default and init's
################################################################################
$accesslog      = 0;
$cachelog       = 0;
$turnlog        = 0;
$outstdout      = 0;
$servername     = 'localhost';
$mailaddr       = "proff";
$logfilename    = "/usr/log/syslog";
$archivedir     = "/tmp";
$tmpfilename    = "/tmp/nnlog.$$";
$syslogpid      = "/etc/syslog.pid";
$mailer         = "Mail";
$compress       = "gzip -f9";

################################################################################
# Usage:
#  - display the parameters
################################################################################
sub Usage
{
    print "Usage: $0 -[acto] [-m address] [-d savedir] [-s servername] [-l logfile]\n";
    print "     : -a            Generate access protocoll\n";
    print "     : -c            Generate cache protocoll\n";
    print "     : -t            Turn logfile (move to savedir)\n";
    print "     : -o            Output to stdout only\n";
    print "     : -s servername hostname of nntp cache server\n";
    print "     : -m adress     Mail protocoll to adress\n";
    print "     : -d savedir    Save Old Logs in savedir\n";
    print "     : -l logfile    Use logfile as input\n";
}

################################################################################
#         
################################################################################
sub AccessLog
{
    local($logfilename, $outfile) = @_;
    local($process)  = '/nntpcached\[([0-9]+)\]: (.*)';
    local($connect)  = 'connect from ([a-zA-Z0-9._+-@]+)';
    local($refuse)   = 'refused connect from ([a-zA-Z0-9._+-@]+)';
    local($groupnow) = '^Current newsgroup now (.+)';
    local($notperm)  = '^(.*) not permitted access to ([a-zA-Z0-9.-]+)';
    local($article)  = '^article [0-9]+';
    
    open (LOG, "<$logfilename") || print $outfile "AccessLog: cannot open input $logfilename: $!\n";
    while (<LOG>)
    {
        if (/$process/oi)
        {
            $pid = $1;
            $text = $2;

            if ($text =~ /$refuse/oi)
            {
                $host = $1;
                $pidToHost{$pid} = $host;
                $refusedFromHosts{$host}++;
            }
            elsif ($text =~ /$connect/oi)
            {
                $host = $1;
                $pidToHost{$pid} = $host;
                $connectFromHosts{$host}++;
            }
            elsif ($text =~ /$groupnow/oi)
            {
                $group = $1;
                $pidToGroup{$pid} = $group;
                $groups{$group}++;

                $hostGroups{join (':',($pidToHost{$pid},$group))}++;
            }
            elsif ($text =~ /$notperm/oi)
            {
                $user = $1;
                $group = $2;
                $forbiddenGroup{$user} .= ($group . "%");
            }
            elsif ($text =~ /$article/oi)
            {
                $articles{$pidToGroup{$pid}}++;
                $hostArticles{$pidToHost{$pid}}++;
                $hostGroupArticles{join (':',($pidToHost{$pid},$pidToGroup{$pid}))}++;
            }
            else
            {
                #print "$text"
            }
        }
    }
    close (LOG);

    #-------------------------------------------------------------------------
    # print it
    #-------------------------------------------------------------------------
    select ($outfile);
    $-=0; $~="HOSTS"; $^="HOSTS_TOP";
    foreach $key (sort(keys %connectFromHosts))
    {
        write;
    }
    
    select ($outfile);
    $-=0; $~="RHOSTS"; $^="RHOSTS_TOP";
    foreach $key (sort(keys %refusedFromHosts))
    {
        write;
    }
    
    select ($outfile);
    $-=0; $~="FUSER"; $^="FUSER_TOP";
    foreach $key (sort(keys %forbiddenGroup))
    {
        $host = $key;
        foreach $group (split (/%/,$forbiddenGroup{$key}))
        {
            write;
            $host = "";
        }
    }
    
    $-=0; $~="GROUPS"; $^="GROUPS_TOP";
    foreach $key (sort(keys %groups))
    {
        write;
    }
    
    $hostold = "";
    $-=0; $~="HGROUPS"; $^="HGROUPS_TOP";
    foreach $key (sort(keys %hostGroups))
    {
        ($host,$group) = split (/:/,$key);
        if (($host eq $hostold) && $- != $^)
        {
            $host = "";
        }
        else
        {
            $hostold = $host;
        }
        write;
    }

    print "\n\n";
}

################################################################################
# Save the logfile
#  - generate new filename (use day of year and pid to uniq the name)
#  - move the file to the savedir
#  - generate a new logfile
#  - send syslog a HUP
#  - compress the old log
################################################################################
sub TurnLog
{
    local($logfilename, $archivedir) = @_;

    if (-s $logfilename && -d $archivedir)
    {
        $tmp = substr($logfilename,rindex($logfilename,"/")+1);
        $newfilename = $archivedir . '/' . $tmp . '.' . (localtime(time))[7];
        while (-e $newfilename)
        {
            $newfilename .= $$;
        }

        if (rename($logfilename,$newfilename))
        {
            open (NEWLOG,">$logfilename") || print $outfile "TurnLog: Cannot create Logfile $logfilename: $!\n";
            close(NEWLOG);
            open (PIDFILE,"<$syslogpid") || print $outfile "TurnLog: Cannot open syslog pidfile $syslogpid: $!\n";

            chop($pid = <PIDFILE>);
            close(PIDFILE);
            kill ('HUP',$pid);

            `$compress $newfilename`;
        }
        else
        {
            print $outfile "TurnLog: rename ($logfilename,$newfilename) failed: $!\n";
        }
    }
    else
    {
        if (! -s $logfilename)
        {
            print $outfile "TurnLog: Logfile $logfilename not found or zero size\n"
        }
        if (! -d $archivedir )
        {
            print $outfile "TurnLog: Savedirectory $archivedir not found: $!\n"
        }
    }
}

################################################################################
# Get the statistics from the NNTPCACHE Server
#  - Connect to server
#  - send a group nntpcache.stats
#  - send a article 1
#  - save the incomming text
################################################################################
sub CacheLog
{
    local($servername, $outfile) = @_;

    $pat = 'S n C4 x8';
    $inet = 2;
    $nntp = 119;

    $this = pack($pat,$inet,0,    0  ,0,0,0);
    $that = pack($pat,$inet,$nntp,127,0,0,1);

    if (!socket(LOG,2,1,6))  { print $outfile "Cachelog: socket failed: $!\n"; return }
    if (!bind(LOG,$this))    { print $outfile "Cachelog: bind failed: $!\n"; return }
    if (!connect(LOG,$that)) { print $outfile "Cachelog: connect failed: $!\n"; return }

    select (LOG); $|=1;
    select ($outfile); $-=0; $~="CACHELOG"; $^="CACHELOG_TOP";

    $_= <LOG>;

    print LOG "group nntpcache.stats\r\n";
    $_= <LOG>;
    print LOG "article 1\r\n";
    while (<LOG>)
    {
        last if (/^\.\r\n$/);
        $line = $_;
        write;
    }
    print LOG "quit\r\n";
    close (LOG);

    print "\n\n";
}


################################################################################
# Output formats
################################################################################
format HOSTS_TOP =
            Connected Hosts                                           Page @##
$%
Host                                Connects  Articles
                                              read
----------------------------------- --------  --------
. 
format HOSTS =
@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< @#######  @#######
$key,$connectFromHosts{$key},$hostArticles{$key}
. 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

format RHOSTS_TOP =
            Refused Hosts                                             Page @##
$%
Host                                Connects  
----------------------------------- --------
. 
format RHOSTS =
@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< @#######
$key,$refusedFromHosts{$key}
. 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

format FUSER_TOP =
               Access to not permitted newsgroups                     Page @##
$%
Accessing User                      Not permitted newsgroup
----------------------------------- --------------------------------
. 
format FUSER =
@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  @<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$host,$group          
. 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

format GROUPS_TOP =
                Group Statistics                                      Page @##
$%
Newsgroup                           select  Articles
                                    count   read
----------------------------------- ------  --------
. 
format GROUPS =
@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< @#####  @#####
$key,$groups{$key},$articles{$key}
. 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

format HGROUPS_TOP =
                 Host - Group Statistics                              Page @##
$%
Host                    Newsgroup                    select  Articles    
                                                     count   read
---------------------- ----------------------------- ------  --------
. 
format HGROUPS =
@<<<<<<<<<<<<<<<<<<<<< @<<<<<<<<<<<<<<<<<<<<<<<<<<<< @#####  @##### 
$host,$group,$hostGroups{$key},$hostGroupArticles{$key}
. 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

format CACHELOG_TOP =
NNTP Cache Statistics from @<<<<<<<<<<<<<<<<<<<<                      Page @##
$servername,$%
---------------------------------------------------------------------
. 
format CACHELOG =
@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< 
$line 
. 

#------------------------------------------------------------------------------
# Check parameters
#------------------------------------------------------------------------------
while ($_ = $ARGV[0], /^-/)
{
    shift;
    if (/^-a/)    { $accesslog++; }
    elsif (/^-c/) { $cachelog++; }
    elsif (/^-t/) { $turnlog++; }
    elsif (/^-o/) { $outstdout++; }
    elsif (/^-d/) { $archivedir = $ARGV[0];  shift }
    elsif (/^-s/) { $servername = $ARGV[0];  shift }
    elsif (/^-m/) { $mailaddr = $ARGV[0]; shift}
    elsif (/^-l/) { $logfilename = $ARGV[0]; shift}
    else          { &Usage; exit(1)}
}

#------------------------------------------------------------------------------
# Do the different jobs
#------------------------------------------------------------------------------
open (OUTPUT, ">$tmpfilename") || die "cannot open output $tmpfilename: $!";

print OUTPUT "Logfiles for NNTPCACHE ", `date`;

$accesslog && &AccessLog($logfilename,OUTPUT);
$cachelog  && &CacheLog($servername,OUTPUT);
$turnlog   && &TurnLog($logfilename, $archivedir);

print OUTPUT "Done ", `date`;
close (OUTPUT);

#------------------------------------------------------------------------------
# Mail the result to the admin
#------------------------------------------------------------------------------
if ($outstdout)
{
    select (STDOUT); 
    open (LOG,"<$tmpfilename") || die "cannot open input $tmpfilename: $!";
    while (<LOG>)
    {
        print;
    }
    close(LOG);
}
else
{
    `$mailer -s 'NNTPCACHE Log' $mailaddr <$tmpfilename`;
}

unlink ($tmpfilename);
         
################################################################################
# EOF         
################################################################################

