//============================================================================
//
//    SSSS    tt          lll  lll
//   SS  SS   tt           ll   ll
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa
//
//============================================================================

#include <time.h>
#include <iostream.h>

#ifdef MSDOS_OS
  #include <strstrea.h>
#else
  #include <strstream.h>
#endif

#include "Term.hxx"
#include "System.hxx"
#include "M6507.hxx"
#include "M6532.hxx"
#include "TIA.hxx"
#include "Snd.hxx"
#include "Cart.hxx"
#include "Joystick.hxx"
#include "Props.hxx"
#include "Error.hxx"
#include "VCS.hxx"
#include "VCSScrpt.hxx"

#if defined DJGPP
  #include <crt0.h>

  // For now we'll just lock all pages so our interrupt routines don't fail
  // TODO: A better solution is needed for this
  int _crt0_startup_flags = _CRT0_FLAG_LOCK_MEMORY;
#endif

#if defined EXCEPTION_FIX
//============================================================================
// Called instead of throwing an exception if EXCEPTION_FIX is defined
//============================================================================
void ExceptionFix(Error& msg)
{
  cerr << "===========" << endl;
  cerr << "FATAL ERROR" << endl;
  cerr << "===========" << endl << endl;
  cerr << msg << endl << endl;
  exit(1);
}

// Dummy error msg
Error msg;
#endif

//============================================================================
// Display usage message
//============================================================================
void usage()
{
  cerr << "Usage: stella filename" << endl;
  exit(1);
}

//============================================================================
// Evaluates the builtin VCS-Script program then evaluates the
// VCS-Script program in 'stella.vcs' if it exists in the 
// current working directory.
//============================================================================
void handleVCSScript(Properties& properties)
{
  VCSScript script;

  // First we evaluate the "built-in" vcs file
  istrstream builtin(defaultVCSScript());
  script.evaluate(builtin, properties);

  // Now, see if there's a file to evaluate
  const char* filename = "stella.vcs";
  ifstream in;

  #if defined UNIX_OS
    in.open(filename, ios::in | ios::nocreate);
  #elif defined MAC_OS
    in.open(filename, ios::in);
  #else
    in.open(filename, ios::in | ios::nocreate);
  #endif

  if(!in.fail())
  {
    script.evaluate(in, properties);
  }
}

//============================================================================
// Main loop
//============================================================================
main(int argc, char* argv[])
{
  // Make sure we have the correct number of arguments
  if(argc != 2)
  {
    usage();
  }

  Properties* pProperties = 0;
  Cartridge* pCartridge = 0;
  Terminal* pTerminal = 0;
  System* pSystem = 0;

  try
  {
    // Load 2600 cartridge properties from the given file
    pProperties = new Properties(argv[1]);

    // Use properties in 'stella.vcs' file or the default one
    handleVCSScript(*pProperties);

    // Make sure the cartridge ROM image exists before we create the 
    // terminal object to avoid screen flickers
    Cartridge::exists(*pProperties);

    // Create the terminal object
    pTerminal = new Terminal(*pProperties);

    // Create 2600 system
    pSystem = new System(*pTerminal, *pProperties);

    #ifdef SHOW_TIMING
      time_t startingClockValue = clock();
    #endif

    bool paused = false;
    bool examinePause = false;

    for(;;)
    {
      // Handle the "pause" key
      bool pause = pTerminal->eventState(BasicTerminal::Pause);

      if(!paused && examinePause && pause)
      {
        paused = true;
        examinePause = false;

        // Turn off the TIA sound by setting volume to zero
        Sound& sound = pSystem->tia().sound();
        sound.set(Sound::AUDV0, 0);
        sound.set(Sound::AUDV1, 0);
      }
      else if(paused && examinePause && pause)
      {
        paused = false;
        examinePause = false;

        // We have to reset the throttle or things will be strange
        pSystem->throttleReset();
      }
      else if(!pause)
      {
        examinePause = true;
      }

      // Exit if the Q key has been pressed
      if(pTerminal->eventState(BasicTerminal::Quit))
        break;

      if(paused)
        pTerminal->update(pSystem->tia());
      else
        pSystem->update();

      // Throttle the frame rate so games don't run to fast
      pSystem->throttle();
    }

    #ifdef SHOW_TIMING
      time_t executionTime = clock() - startingClockValue;
    #endif

    delete pTerminal;

    #ifdef SHOW_TIMING
      #ifdef CLOCKS_PER_SEC
        double clocks = CLOCKS_PER_SEC;
      #else
        double clocks = 1000000.0;
      #endif

      double seconds = ((double)executionTime / clocks);
      double cyclesPerSecond = pSystem->m6507().cycles() / seconds;
      double efficiency = (cyclesPerSecond / 1190000.0) * 100; 
      double framesPerSecond = pSystem->numberOfFrames() / seconds;

      cout << endl;
      cout << pSystem->numberOfFrames() << " total frames drawn" << endl;
      cout << cyclesPerSecond << " Cycles/Second\n";
      cout << efficiency << "% real speed\n";
      cout << framesPerSecond << " Frames/Second\n";
      cout << endl;
    #endif

    delete pSystem;
    delete pCartridge;
    delete pProperties;
  }
  catch(Error& msg)
  {
    delete pTerminal;

    cerr << "===========" << endl;
    cerr << "FATAL ERROR" << endl;
    cerr << "===========" << endl << endl;
    cerr << msg << endl << endl;
    exit(1);
  }
}

