/********************************************************************************
 * Copyright (c) Erik Kunze   1996 - 1998
 *               Ville Hallik 1996
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Erik Kunze
 *         Ville Hallik (basic AY-3-8912 emulation)
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char ay8912_c[] = "$Id: ay8912.c,v 4.1 1998/01/06 21:18:56 erik Rel $";
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <unistd.h>
#if defined(sun)
#ifdef SVR4
#include <sys/audioio.h>
#include <stropts.h>
#include <sys/conf.h>
#else
#include <sun/audioio.h>
#endif
#endif
#if defined(__linux__)
#include <linux/soundcard.h>
#endif
#if defined(XXX)
#include <machine/soundcard.h>
#endif
#include "config.h"
#include "debug.h"
#include "util.h"
#include "ay8912.h"

#define SND_CMD			0xffU
#define SND_PAUSE		0
#define SND_RESUME		1
#define SND_QUIT		2
#ifdef sun
#define CONV(x)			(ulaw[0xff & ((x) ^ 128)])
#else
#define CONV(x)			((x) ^ 128)
#endif

static void ayQuit(void);
static int ayOpenDevice(void);
static void aySignal(int);
static void ayCommand(int);
static void ayPlay(void);

int AyActive = 0;
int AyOn = 0;

static int ayFd = -1;
static char *ayDevice;
static pid_t ayChild;
static int ayParent = 1;
static int ayPaused = 0;
static int ayPipe[2] = { -1, -1 };
static int aySampleRate = SOUND_RATE;
static unsigned char envelopeForms[16][32] = {
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	  15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	  15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	  15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	  15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 }
};

int
AyInit(char *audioDev, int audioOn)
{
	struct sigaction sigact;
	OnQuit(ayQuit);
	ayDevice = audioDev;
	if (ayOpenDevice() == -1)
	{
		return -1;
	}

	if (pipe(ayPipe) == -1)
	{
		Msg(M_PERR, "couldn't create communication pipe");
		(void)close(ayFd);
		return -1;
	}

	switch (ayChild = fork())
	{
		case -1:
			Msg(M_PERR, "couldn't fork AY-3-8292 daemon");
			(void)close(ayFd);
			(void)close(ayPipe[0]);
			ayPipe[0] = -1;
			(void)close(ayPipe[1]);
			ayPipe[1] = -1;
			return -1;
		case 0:
			ayParent = 0;
			(void)fcntl(ayPipe[0], F_SETFL, O_NONBLOCK);
			(void)close(ayPipe[1]);
			ayPipe[1] = -1;
			ayPlay();
			break;
		default:
			(void)close(ayPipe[0]);
			ayPipe[0] = -1;
			(void)fcntl(ayPipe[1], F_SETFL, O_NONBLOCK);
			(void)close(ayFd);
			break;
	}

	(void)sigemptyset(&sigact.sa_mask);
#ifdef SA_RESTART
	sigact.sa_flags = SA_RESTART;
#else
	sigact.sa_flags = 0;
#endif
	sigact.sa_handler = aySignal;
	(void)sigaction(SIGCHLD, &sigact, NULL);
	(void)sigaction(SIGPIPE, &sigact, NULL);
	AyActive = 1;
	AyOn = 1;
	if (!audioOn)
	{
		AyOnOff();
	}
	return 0;
}

static void
ayQuit(void)
{
	int status;
	if (ayParent)
	{
		if (AyActive)
		{

			(void)signal(SIGCHLD, SIG_IGN);
			AyOutByte((uns8)SND_CMD, (uns8)SND_QUIT);
			(void)wait(&status);
		}
	}
	else
	{
		if (ayFd != -1)
		{
			(void)close(ayFd);
		}
	}
	if (ayPipe[0] != -1)
	{
		(void)close(ayPipe[0]);
	}
	if (ayPipe[1] != -1)
	{
		(void)close(ayPipe[1]);
	}
}

static int
ayOpenDevice(void)
{
#if defined(__linux__)
	int i;
	if ((ayFd = open(ayDevice, O_WRONLY | O_NONBLOCK)) == -1)
	{
		Msg(M_PERR, "couldn't open audio device <%s> for writing", ayDevice);
		return -1;
	}

	i = AFMT_U8;
	if (ioctl(ayFd, SNDCTL_DSP_SETFMT, &i) == -1 || i != AFMT_U8) {
		Msg(M_PERR, "couldn't set sample resolution to 8 bit");
		(void)close(ayFd);
		ayFd = -1;
		return -1;
	}

	i = 0;
	if (ioctl(ayFd, SNDCTL_DSP_STEREO, &i) == -1 || i) {
		Msg(M_PERR, "couldn't set mono output");
		(void)close(ayFd);
		ayFd = -1;
		return -1;
	}

	if (ioctl(ayFd, SNDCTL_DSP_SPEED, &aySampleRate) == -1)
	{
		Msg(M_PERR, "couldn't set sample rate to %d Hz", aySampleRate);
		(void)close(ayFd);
		ayFd = -1;
		return -1;
	}

	i = SOUND_BUFSIZE_BITS | (SOUND_NUM_OF_BUFS << 16);
	if (ioctl(ayFd, SNDCTL_DSP_SETFRAGMENT, &i) == -1)
	{
		Msg(M_PERR, "couldn't set fragment size");
		(void)close(ayFd);
		ayFd = -1;
		return -1;
	}
	return 0;
#elif defined(sun)

#ifdef SVR4
	struct audio_info audioInf;
#endif
	if ((ayFd = open(ayDevice, O_WRONLY | O_NONBLOCK)) == -1)
	{
		Msg(M_PERR, "couldn't open audio device <%s> for writing", ayDevice);
		return -1;
	}

#ifdef SVR4
	if (ioctl(ayFd, AUDIO_GETINFO, &audioInf) == -1)
	{
		Msg(M_PERR, "couldn't get audio settings");
		close(ayFd);
		ayFd = -1;
		return -1;
	}
#ifdef DEBUG
	if (GETCFG(debug) & D_AUDIO)
	{
		Msg(M_DEBUG,
			"audio settings: sample rate = %d", audioInf.play.sample_rate);
		Msg(M_DEBUG,
			"audio settings: channels    = %d", audioInf.play.channels);
		Msg(M_DEBUG,
			"audio settings: precision   = %d", audioInf.play.precision);
		Msg(M_DEBUG,
			"audio settings: encoding    = %d", audioInf.play.encoding);
	}
#endif
	audioInf.play.sample_rate = aySampleRate;
	audioInf.play.channels = 1;
	audioInf.play.precision = 8;
	audioInf.play.encoding = AUDIO_ENCODING_ULAW;
	if (ioctl(ayFd, AUDIO_SETINFO, &audioInf))
	{
		Msg(M_PERR, "couldn't initialise audio device");
		close(ayFd);
		ayFd = -1;
		return -1;
	}
#endif
	return 0;
#else

	return -1;
#endif
}

void
AyOnOff(void)
{
	if (AyActive)
	{
		AyOn = !AyOn;
		AyOutByte((uns8)SND_CMD, AyOn ? (uns8)SND_RESUME : (uns8)SND_PAUSE);
	}
}

static void
aySignal(int signo)
{
	int status;
#ifdef DEBUG
	if (GETCFG(debug) & D_AUDIO)
	{
		Msg(M_DEBUG, "got signal %d", signo);
	}
#endif
	switch(signo)
	{
		case SIGCHLD:
#ifdef WNOHANG
			if (waitpid(-1, &status, WNOHANG) != ayChild)
#else
			if (waitp(&status) != ayChild)
#endif
			{
				break;
			}
			Msg(M_WARN, "AY-3-8912 daemon died");

		case SIGPIPE:
			AyActive = 0;
			(void)close(ayPipe[1]);
			ayPipe[1] = -1;
			break;
	}
}

static void
ayCommand(int cmd)
{
	switch (cmd)
	{
		case SND_PAUSE:
#ifdef sun
			IntFrequency(0);
#endif
			ayPaused = 1;
			(void)close(ayFd);
			ayFd = -1;
			break;
		case SND_RESUME:
			if (ayOpenDevice() != -1)
			{
#ifdef sun
				IntFrequency(1000000 * SOUND_BUFSIZE / aySampleRate);
#endif
				ayPaused = 0;
			}
			break;
		case SND_QUIT:
			(void)close(ayPipe[0]);
			if (ayFd != -1)
			{
				(void)close(ayFd);
			}
			exit(0);
	}
}

void
AyOutByte(uns8 reg, uns8 val)
{
	uns8 buf[2];
	if (AyActive)
	{
		buf[0] = reg;
		buf[1] = val;
		(void)write(ayPipe[1], buf, sizeof(buf));
	}
}


static void
ayPlay(void)
{
	int incr0 = 0, incr1 = 0, incr2 = 0, increnv = 0, incrnoise = 0;
	int statenoise = 0, noisegen = 1;
	int counter0 = 0, counter1 = 0, counter2 = 0, countenv = 0, countnoise = 0;
	int i, r = 0, v = 0;
	int vol0, vol1, vol2, volnoise, envelope = 15;
	int c0, c1, l0, l1, l2;
	unsigned char buf[2];
	unsigned char sndbuf[SOUND_BUFSIZE];
	unsigned char psg[14] = { 0, 0, 0, 0, 0, 0, 0, 255, 8, 8, 8, 0, 0, 0 };
	pid_t parent;
#ifdef sun
	struct sigaction sigact;
#endif
	parent = getppid();
#ifdef sun

	(void)sigemptyset(&sigact.sa_mask);
#ifdef SA_RESTART
	sigact.sa_flags = SA_RESTART;
#else
	sigact.sa_flags = 0;
#endif
	sigact.sa_handler = aySignal;
	(void)sigaction(SIGALRM, &sigact, NULL);
	IntFrequency(1000000 * SOUND_BUFSIZE / aySampleRate);
#endif
	for (;;)
	{
		if (parent != getppid())
		{
			Msg(M_WARN, "AY-3-8912 daemon: parent died");
			Quit(1);
		}

		while ((i = (int)read(ayPipe[0], buf, (size_t)(v < 0 ? 1 : 2))) != 0)
		{
			if (i == -1)
			{
				if (errno != EAGAIN && errno != EWOULDBLOCK)
				{
					Msg(M_PERR, "AY-3-8912 daemon: couldn't read from pipe");
				}
				break;
			}
			if (v == -1)
			{
				v = buf[0];
			}
			else
			{
				r = buf[0];
				if (i == 1)
				{
					v = -1;
					break;
				}
				v = buf[1];
			}
#ifdef DEBUG
			if (GETCFG(debug) & D_AUDIO)
			{
				Msg(M_DEBUG, "AY-3-8912 daemon: reg = %02x val = %02x", r, v);
			}
#endif

			if (r == SND_CMD)
			{
				ayCommand(v);
				break;
			}
			psg[r] = v; ;
			switch (r)
			{
				case 1:
					psg[1] &= 0x0f;
				case 0:
					i = psg[0] + (psg[1] << 8);
					incr0 = i ? 1024 * PSG_CLOCK / aySampleRate * 4 / i : 0;
					break;
				case 3:
					psg[3] &= 0x0f;
				case 2:
					i = psg[2] + (psg[3] << 8);
					incr1 = i ? 1024 * PSG_CLOCK / aySampleRate * 4 / i : 0;
					break;
				case 5:
					psg[5] &= 0x0f;
				case 4:
					i = psg[4] + (psg[5] << 8);
					incr2 = i ? 1024 * PSG_CLOCK / aySampleRate * 4 / i : 0;
					break;
				case 6:
					i = psg[6] & 0x1f;
					incrnoise = 1024 * PSG_CLOCK / aySampleRate / (i ? i : 1);
					break;
				case 8:
				case 9:
				case 10:
					psg[r] &= 0x1f;
					break;
				case 11:
				case 12:
					i = psg[11] + (psg[12] << 8);
					increnv = i ? 1024 * PSG_CLOCK / aySampleRate * 4 / i *
						          SOUND_BUFSIZE : 0;
					countenv=0;
				break;
			}
		}
		envelope = envelopeForms[psg[13]][(countenv >> 16) & 0x1f];
		if ((countenv += increnv) & 0xffe00000)
		{
			switch (psg[13])
			{
				case 8:
				case 10:
				case 12:
				case 14:
					countenv -= 0x200000;
					break;
				default:
					countenv = 0x100000;
					increnv = 0;
			}
		}
		vol0 = psg[ 8] < 16 ? psg[ 8] : envelope;
		vol1 = psg[ 9] < 16 ? psg[ 9] : envelope;
		vol2 = psg[10] < 16 ? psg[10] : envelope;
		volnoise = ((psg[7] & 010 ? 0 : vol0) +
					(psg[7] & 020 ? 0 : vol1) +
					(psg[7] & 040 ? 0 : vol2)) / 2;
		vol0 = psg[7] & 0x01 ? 0 : vol0;
		vol1 = psg[7] & 0x02 ? 0 : vol1;
		vol2 = psg[7] & 0x04 ? 0 : vol2;
		for (i = 0; i < SOUND_BUFSIZE; i++)
		{

			c0 = counter0;
			c1 = counter0 + incr0;
			l0 = c0 & 0x8000 ? -16 : 16;
			if ((c0 ^ c1) & 0x8000)
			{
				l0 = l0 * (0x8000 - (c0 & 0x7fff) - (c1 & 0x7fff)) / incr0;
			}
			counter0 = c1 & 0xffff;
			c0 = counter1;
			c1 = counter1 + incr1;
			l1 = c0 & 0x8000 ? -16 : 16;
			if ((c0 ^ c1) & 0x8000)
			{
				l1 = l1 * (0x8000 - (c0 & 0x7fff) - (c1 & 0x7fff)) / incr1;
			}
			counter1 = c1 & 0xffff;
			c0 = counter2;
			c1 = counter2 + incr2;
			l2 = c0 & 0x8000 ? -16 : 16;
			if ((c0 ^ c1) & 0x8000)
			{
				l2 = l2 * (0x8000 - (c0 & 0x7FFF) - (c1 & 0x7fff)) / incr2;
			}
			counter2 = c1 & 0xffff;
			countnoise &= 0xffff;
			if ((countnoise += incrnoise) & 0xffff0000)
			{

				statenoise = ((noisegen <<= 1) & 0x80000000 ?
							  noisegen ^= 0x00040001 : noisegen ) & 1;
			}
			sndbuf[i] = CONV((l0 * vol0 + l1 * vol1 + l2 * vol2) / 16 +
							 (statenoise ? volnoise : -volnoise));
		}
		if (ayPaused)
		{
			(void)sleep(1);
		}
		else
		{
#if defined(__linux__)

			(void)write(ayFd, sndbuf, SOUND_BUFSIZE);
#elif defined(sun)
			(void)write(ayFd, sndbuf, SOUND_BUFSIZE);
#ifdef NEED_AUDIO_DRAIN

			(void)ioctl(ayFd, AUDIO_DRAIN, 0);
#endif

			sigpause(SIGALRM);
#endif
		}
	}
}

