/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU, DNH
 *******************************************************************************/
#ifndef lint
static char sunaudio_c[] = "$Id: sunaudio.c,v 4.2 1998/01/06 21:18:56 erik Rel $";
#endif

#include <assert.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#ifdef SVR4
#include <sys/audioio.h>
#else
#include <sun/audioio.h>
#endif
#include <stropts.h>
#include <sys/conf.h>
#include "config.h"
#include "debug.h"
#include "mem.h"
#include "io.h"
#include "util.h"
#include "emul.h"
#include "sunaudio.h"


static void speakerQuit(void);
static int speakerOpenDevice(void);

int SpActive = 0;
int SpOn = 0;

static int spFd = -1;
static char *spDevice;
static unsigned char *audioBuf = NULL;
static int lastPos = 0;
static signed char lastVal = -128;
static int soundsInBuffer = 0;

static int tstatesPerSample;
static int samplesPerInterrupt;

int
SpeakerInit(char *audioDev, int audioOn)
{
	OnQuit(speakerQuit);
	spDevice = audioDev;
	if (speakerOpenDevice() == -1)
	{
		return -1;
	}
	SpActive = 1;
	SpOn = 1;
	tstatesPerSample = Machine->tstatesPerFrame * 1000 / GETCFG(ms)
																/ SAMPLE_RATE;
	samplesPerInterrupt = GETCFG(ms) * SAMPLE_RATE / 1000;

	audioBuf = Malloc(samplesPerInterrupt, "SpeakerInit");
	if (!audioOn)
	{
		SpeakerOnOff();
	}
	return 0;
}

static void
speakerQuit(void)
{
	if (spFd != -1)
	{
		(void)close(spFd);
	}
}

static int
speakerOpenDevice(void)
{
#ifdef SVR4
	struct audio_info audioInf;
#endif

	if ((spFd = open(spDevice, O_WRONLY | O_NONBLOCK)) == -1)
	{
		Msg(M_PERR, "couldn't open audio device <%s> for writing", spDevice);
		return -1;
	}

#ifdef SVR4
	if (ioctl(spFd, AUDIO_GETINFO, &audioInf) == -1)
	{
		Msg(M_PERR, "couldn't get audio settings");
		close(spFd);
		spFd = -1;
		return -1;
	}
#ifdef DEBUG
	if (GETCFG(debug) & D_AUDIO)
	{
		Msg(M_DEBUG,
			"audio settings: sample rate = %d", audioInf.play.sample_rate);
		Msg(M_DEBUG,
			"audio settings: channels    = %d", audioInf.play.channels);
		Msg(M_DEBUG,
			"audio settings: precision   = %d", audioInf.play.precision);
		Msg(M_DEBUG,
			"audio settings: encoding    = %d", audioInf.play.encoding);
	}
#endif
	audioInf.play.sample_rate = SAMPLE_RATE;
	audioInf.play.channels = 1;
	audioInf.play.precision = 8;
	audioInf.play.encoding = AUDIO_ENCODING_ULAW;
	if (ioctl(spFd, AUDIO_SETINFO, &audioInf))
	{
		Msg(M_PERR, "couldn't initialize audio device");
		close(spFd);
		spFd = -1;
		return -1;
	}
#endif
	return 0;
}

void
SpeakerReInit(void)
{
	if (SpActive)
	{
		tstatesPerSample = Machine->tstatesPerFrame * 1000 / GETCFG(ms)
																/ SAMPLE_RATE;
		samplesPerInterrupt = GETCFG(ms) * SAMPLE_RATE / 1000;

		assert(audioBuf);
		free(audioBuf);
		audioBuf = Malloc(samplesPerInterrupt, "SpeakerInit");
	}
}

void
SpeakerOnOff(void)
{
	if (SpActive)
	{
		SpOn = !SpOn;
		if (SpOn)
		{
			if (speakerOpenDevice() == -1)
			{

				SpOn = 0;
			}
		}
		else
		{
			(void)close(spFd);
		}
	}
}

void
SpeakerOutByte(uns8 newval)
{
	static uns8 oldval = 0;

	if (SpActive && SpOn && ((newval ^ oldval) & B_SPEAKER))
	{
		int pos = (Vline * Machine->tstatesPerLine + Tstates) / tstatesPerSample;
		for (; lastPos < pos; lastPos++)
		{
			audioBuf[lastPos] = lastVal;
		}
		lastVal = audioBuf[lastPos] =
							ulaw[0xff & (((newval & B_SPEAKER ? 127 : -128) *
										  GETCFG(volume) / 100) ^ 128)];
		soundsInBuffer = 1;
		oldval = newval;
	}
}

void
SpeakerPlayBuffer(void)
{
	if (SpActive && SpOn)
	{
#ifndef NEED_AUDIO_DRAIN
		if (soundsInBuffer)
#endif
		{
			soundsInBuffer = 0;
			for (;lastPos < samplesPerInterrupt; lastPos++)
			{
				audioBuf[lastPos] = lastVal;
			}
#ifdef NEED_AUDIO_DRAIN
			(void)ioctl(spFd, AUDIO_DRAIN, 0);
#endif
			(void)write(spFd, audioBuf, samplesPerInterrupt);
		}
		lastPos = 0;
	}
}

