/********************************************************************************
 * Copyright (c) Razvan Surdulescu 1996
 *               Erik Kunze        1996 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Razvan Surdulescu
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: dialog.c,v 4.2 1998/01/06 21:21:30 erik Rel $";
#endif

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include "config.h"
#include "resource.h"
#include "mem.h"
#include "util.h"
#ifdef AUDIO
#include "audio.h"
#endif
#ifdef LINE_VIDEO
#include "emul.h"
#endif
#include "main.h"
#include "screen.h"
#include "tables.h"
#include "dialog.h"

#define PAGE_SIZE			14
#define MAXFILELEN			256
#define DIR_TYPE			0
#define FILE_TYPE			1
typedef struct _mydirent {
	int type;
	char * name;
} mydirent;

static uns16 xyToAddr (uns8, uns8);
static uns16 xyToAttr (uns8, uns8);
static int direntsCompare(void *, void *);
static void direntDestroy(void *);
static int readDirectory(char *, char *, char *);
static mydirent *getFileOrDir(int);
static void displayFiles(int);
static char *addExtension(char *, char *);
static void displayMenuTitle(char *);

int InDialog = 0;
KeySym KeyDialog;
#if defined(PCSPKR_AUDIO) || defined(SUN_AUDIO)
int OldSp;
#endif
#ifdef AYCHIP_AUDIO
int OldAy;
#endif


static char *menuBottom[] = {
	"   UP/DOWN arrows move cursor   ",
	"   ENT/KEY select               ",
	"   ESC     return               ",
	NULL
};
static uns8 cursorX = 0;
static uns8 cursorY = 0;
static uns8 screenAttr = MENU_TEXT_ATTR;

static uns8 *screenBackup = NULL;
static int saveScreenSelect;
#ifdef LINE_VIDEO
static int saveVline;
#endif
static char *saveDir;
static char *saveFSelDir;
static int numFiles = 0;
static int numDirectories = 0;
static llist *dirDirectoriesList = NULL;

static llist *dirFilesList = NULL;

void
DialogInit(int copyright)
{
	time_t timeStart;
	if (!(saveDir = getcwd(NULL, MAXPATHLEN)))
	{
		Msg(M_FATAL, "getcwd failed");
	}
	screenBackup = (uns8 *)Malloc(PIXEL_LENGTH * ATTR_LENGTH, "DialogInit");
	saveFSelDir = (char*)Malloc(MAXPATHLEN, "DialogInit");
	(void)strcpy(saveFSelDir, saveDir);
	if (copyright)
	{

		ClearScreen();
		SetBorderColor((MENU_TEXT_ATTR & PAPER) >> 3);
		SetCursor((COLS - strlen(Version)) / 2, 5);
		PrintString(Version);
		SetCursor(1, 7);
		PrintString("ZX Spectrum 48/128/+3 emulator");
		SetCursor(1, 11);
		PrintString(" \177 Des Herriott       1993-94");
		SetCursor(1, 13);
		PrintString("   Erik Kunze         1995-98");
		SetCursor(1, 15);
		PrintString("   Razvan Surdulescu  1996   ");

		InDialog = 1;
		timeStart = time(NULL);
		while((time(NULL) - timeStart) < 5)
		{
			ScreenRefresh();
			CheckEvents();
		}
		InDialog = 0;
		SetBorderColor(LastBorderColor);
	}
}

static uns16
xyToAddr (uns8 x, uns8 y)
{
	assert(x < COLS &&  y < ROWS);
	y *= 8;
	return (DFILE +
			(y / 64) * (PIXEL_LENGTH / 3) +
			((y % 64) / 8) * COLS +
			(y % 8) * X_PIXELS +
			x);
}

static uns16
xyToAttr (uns8 x, uns8 y)
{
	assert(x < COLS && y < ROWS);
	return (DFILE + ATTR_OFFSET + y * COLS + x);
}

void
SetAttr (uns8 attr)
{
	screenAttr = attr;
}

uns8
GetAttr(void)
{
    return screenAttr;
}

void
SetCursor (uns8 x, uns8 y)
{
	cursorX = x < COLS ? x : COLS - 1;
	cursorY = y < ROWS ? y : ROWS - 1;
}

void
GetCursor(uns8 *x, uns8 *y)
{
	*x = cursorX;
	*y = cursorY;
}

void
ClearScreen(void)
{
	cursorX = 0;
	cursorY = 0;
	DrawBox(COLS, ROWS);
}

void
DrawBox(uns8 width, uns8 height)
{
	uns8 saveX = cursorX, saveY = cursorY;
	uns8 i;
	for (; height != 0; height--, cursorY++)
	{
		for (cursorX = saveX, i = width; i != 0; i--)
		{
			PrintLetter(' ');
		}
	}
	cursorX = saveX;
	cursorY = saveY;
}

void
PrintLetter(uns8 letter)
{
	if (cursorX < COLS && cursorY < ROWS)
	{
		uns8 i;
		uns16 addr = xyToAddr(cursorX, cursorY);
		uns8 *chr;
		if (letter < ' ' || letter > ' ' + sizeof(CharSet)/sizeof(CharSet[0]))
		{
			letter = '?';
		}
		chr = (uns8 *)CharSet[letter - ' '];
		WR_BYTE(xyToAttr(cursorX, cursorY), screenAttr);
		for (i = 0; i < 8; i++, addr += X_PIXELS, chr++)
		{
			WR_BYTE(addr, *chr);
		}
		cursorX++;
	}
}

void
PrintString(char *string)
{
	while (*string && cursorX < COLS)
	{
		PrintLetter(*string++);
	}
}

KeySym
GetKey(void)
{
	int oldInDialog = InDialog;
	InDialog = 1;
	for (KeyDialog = 0; !KeyDialog;)
	{
		CheckEvents();
	}
	InDialog = oldInDialog;
	return KeyDialog;
}

int
GetString(char *string, int len, int width)
{
	int finish = 0;
	int pos;
	uns8 begX = cursorX;
	uns8 begY = cursorY;

	PrintLetter('_');
	for (pos = width; pos > 1; pos--)
	{
		PrintLetter(' ');
	}
	pos = 0;
	SetCursor(begX, begY);
	ScreenRefresh();
	for (;;)
	{
		switch (GetKey())
		{
			case XK_Return:
				finish = 1;
				goto quit;
			case ESCAPE:
			case XK_Tab:
				finish = 2;
				goto quit;
			case XK_BackSpace:
				if (pos > 0)
				{
					string[--pos] = '\0';
					if (pos < width - 1)
					{
						PrintLetter(' ');
						cursorX -= 2;
					}
					else
					{
						SetCursor(begX, begY);
						PrintString(string + (pos - width) + 1);
					}
				}
				break;
			default:
				if (KeyDialog >= 0x20 && KeyDialog <= 0x7e && pos + 1 < len)
				{
					string[pos++] = (char)KeyDialog;
					string[pos] = '\0';
					if (pos < width)
					{
						PrintLetter(KeyDialog);
					}
					else
					{
						SetCursor(begX, begY);
						PrintString(string + (pos - width) + 1);
					}
				}
				break;
		}
		PrintLetter('_');
		cursorX--;
		ScreenRefresh();
	}
  quit:

	PrintLetter(' ');
	string[pos] = '\0';
	return (finish != 2);
}

int
ConfirmBox(char *string)
{
	int oldInDialog;
	KeySym key;

	oldInDialog = EnterOSD();
	ClearScreen();
	SetCursor((COLS - strlen(string)) / 2, ROWS / 2 - 1);
	PrintString(string);
	ScreenRefresh();
	key = GetKey();

	LeaveOSD(oldInDialog);
	return (key == XK_y || key == XK_Y);
}

int
EnterOSD(void)
{
	int oldInDialog = InDialog;

	if (!oldInDialog)
	{
		XAutoRepeatOn(Dpy);

		(void)memset(KeyPorts, 0xff, sizeof(KeyPorts));
		saveScreenSelect = ScreenSelect;
		ScreenSelect = RAM5;
		SaveScreen();
		SetBorderColor((MENU_TEXT_ATTR & PAPER) >> 3);
#ifdef LINE_VIDEO
		saveVline = Vline;
		Vline = 0;
#endif
#if defined(PCSPKR_AUDIO) || defined(SUN_AUDIO)
		if ((OldSp = SpOn))
		{
			SpeakerOnOff();
		}
#endif
#ifdef AYCHIP_AUDIO
		if ((OldAy = AyOn))
		{
			AyOnOff();
		}
#endif
	}
	InDialog = 1;
	return oldInDialog;
}

void
LeaveOSD(int oldInDialog)
{

	if (!oldInDialog)
	{
		XAutoRepeatOff(Dpy);
		RestoreScreen();
		SetBorderColor(LastBorderColor);
		ScreenSelect = saveScreenSelect;
		ForceScreenRefresh();
#ifdef LINE_VIDEO
		Vline = saveVline;
#endif
#if defined(PCSPKR_AUDIO) || defined(SUN_AUDIO)
		if (OldSp)
		{
			SpeakerOnOff();
		}
#endif
#ifdef AYCHIP_AUDIO
		if (OldAy)
		{
			AyOnOff();
		}
#endif
	}
	InDialog = oldInDialog;
}

void
SaveScreen(void)
{
	(void)memcpy(screenBackup, RealMemory[ScreenSelect],
				 PIXEL_LENGTH + ATTR_LENGTH);
}

void
RestoreScreen(void)
{
	(void)memcpy(RealMemory[ScreenSelect], screenBackup,
				 PIXEL_LENGTH + ATTR_LENGTH);
}

static int
direntsCompare(void *d1, void *d2)
{
	return (strcmp(((mydirent *)d1)->name, ((mydirent *)d2)->name));
}

static void
direntDestroy(void *d)
{
	assert(((mydirent *)d)->name != NULL);
	free(((mydirent *)d)->name);
}

static int
readDirectory(char *fext1, char *fext2, char * fext3)
{
	DIR *dir;
	struct dirent *ent;
	struct stat buf;
	char *currentPath, *p;
	mydirent *myDirent;
	if (!(currentPath = getcwd(NULL, MAXPATHLEN))
		|| !(dir = opendir(currentPath)))
	{
		Msg(M_PERR, "couldn't open directory <%s>", currentPath);
		free(currentPath);
		return -1;
	}
	while ((ent = readdir(dir)))
	{

		if (ent->d_name[0] == '.' && ent->d_name[1] != '.')
		{
			continue;
		}
		(void)stat(ent->d_name, &buf);
		if (S_ISDIR(buf.st_mode))
		{

			myDirent = Malloc(sizeof(mydirent), "readDirectory");
			myDirent->type = DIR_TYPE;
			myDirent->name = Strdup(ent->d_name, "readDirectory");
			dirDirectoriesList = SortedInsertList((void *)myDirent,
												  direntsCompare,
												  dirDirectoriesList);
			numDirectories++;
		}
		else if (S_ISREG(buf.st_mode))
		{

			if (!fext1
				|| ((p = strrchr(ent->d_name, '.')) != NULL
					&& (!strcasecmp(++p, fext1)
						|| (fext2 && !strcasecmp(p, fext2))
						|| (fext3 && !strcasecmp(p, fext3)))))
			{

				myDirent = Malloc(sizeof(mydirent), "readDirectory");
				myDirent->type = FILE_TYPE;
				myDirent->name = Strdup(ent->d_name, "readDirectory");
				dirFilesList = SortedInsertList((void *)myDirent,
												direntsCompare,
												dirFilesList);
				numFiles++;
			}
		}
	}
	(void)closedir(dir);
	free(currentPath);
	return 0;
}

static mydirent *
getFileOrDir(int which)
{
	assert(which >= 0);
	if (which < numDirectories)
	{
		return ((mydirent*) RetrieveElemList(dirDirectoriesList, which));
	}
	which -= numDirectories;
	if (which < numFiles)
	{
		return ((mydirent*) RetrieveElemList(dirFilesList, which));
	}
	return NULL;
}

static void
displayFiles(int page)
{
	int cnt, len;
	char *name;
	mydirent *disp;
	uns8 attr= GetAttr();
	SetAttr(MENU_SELECT_ATTR);
	SetCursor(MENU_CURSOR_X + 1, MENU_CURSOR_Y);
	DrawBox(COLS - 2 * (MENU_CURSOR_X + 1), PAGE_SIZE);

	for (cnt = page * PAGE_SIZE; cnt < (page + 1) * PAGE_SIZE; cnt++)
	{
		if (!(disp = getFileOrDir(cnt)))
		{
			break;
		}
		name  = disp->name;
		for (len = COLS - 2 * (MENU_CURSOR_X + 1); len > 0 && *name; len--)
		{
			PrintLetter(*name++);
		}
		if (disp->type == DIR_TYPE && len)
		{
			PrintLetter('/');
		}
		SetCursor(MENU_CURSOR_X + 1, cursorY + 1);
	}

	SetAttr(attr);
}

static char *
addExtension(char *string, char *ext)
{
	char *p;

	if (!(p = strrchr(string, '.')))
	{
		p = (char*)Malloc(strlen(string) + strlen(ext) + 2, "addExtension");
		(void)strcpy(p, string);
		(void)strcat(p, ".");
		(void)strcat(p, ext);
	}
	else
	{
		p = Strdup(string, "addExtension");
	}
	return p;
}

char *
FileSelector(char *header, int valid, char *fext1, char *fext2, char *fext3)
{
	static int item = 0;
	int oldInDialog;
	mydirent *file;
	struct stat buf;
	char name[MAXFILELEN];
	char *returnName = NULL;
	char *returnTotal = NULL;

	oldInDialog = EnterOSD();

	(void)chdir(saveFSelDir);

	{
		char *selectorText[PAGE_SIZE + 2];
		int i;
		selectorText[0] = header;
		for (i = 1; i <= NOPTS(selectorText); i++)
		{
			selectorText[i] = " ";
		}
		selectorText[i] = NULL;
		DisplayMenu(selectorText);
	}
	SetCursor(0, ROWS - NOPTS(menuBottom) - 2);
	PrintString("   TAB     switch fields        ");
	ScreenRefresh();

	if (readDirectory(fext1, fext2, fext3) == -1)
	{
		goto quit;
	}

	if (item >= numFiles + numDirectories)
	{
		item = numFiles + numDirectories - 1;
	}
	displayFiles(item / PAGE_SIZE);
	SetCursor(0, PAGE_SIZE + 3);
	PrintString(" File [                       ] ");
	for (;;)
	{
		switch (BrowseList(&item, numFiles + numDirectories, PAGE_SIZE,
						   displayFiles))
		{

			case XK_Tab:

				{
					int ret;
					SetAttr(MENU_SELECT_ATTR);
					SetCursor(7, PAGE_SIZE + 3);
					ret = GetString(name, MAXFILELEN, 23);
					SetAttr(MENU_TEXT_ATTR);
					SetCursor(0, PAGE_SIZE + 3);
					PrintString(" File [                       ] ");
					if (ret)
					{
						if (stat(name, &buf) != -1)
						{
							if (S_ISDIR(buf.st_mode))
							{
								(void)chdir(name);
								goto chdir;
							}
							else if (S_ISREG(buf.st_mode))
							{
								if (valid
									|| ConfirmBox("Overwrite file ? [y/n]"))
								{
									returnName = addExtension(name, fext1);
								}
								goto quit;
							}
						}
						else if (errno == ENOENT && !valid)
						{
							returnName = addExtension(name, fext1);
							goto quit;
						}
					}
				}
				break;
			case XK_Return:

				file = getFileOrDir(item);
				assert(file != NULL);
				if (file->type == DIR_TYPE)
				{
					(void)chdir(file->name);
				  chdir:
					DestroyList(dirFilesList, direntDestroy);
					numFiles = 0;
					dirFilesList = NULL;
					DestroyList(dirDirectoriesList, direntDestroy);
					numDirectories = 0;
					dirDirectoriesList = NULL;
					if (readDirectory(fext1, fext2, fext3) == -1)
					{
						goto quit;
					}
					displayFiles(0);
					item = 0;
				}
				else
				{
					assert(file->type == FILE_TYPE);
					if (valid || ConfirmBox("Overwrite file ? [y/n]"))
					{
						returnName = Strdup(file->name, "FileSelector");
					}
					goto quit;
				}
				break;
			case ESCAPE:
				goto quit;
		}
	}

  quit:
	LeaveOSD(oldInDialog);
	DestroyList(dirFilesList, direntDestroy);
	numFiles = 0;
	dirFilesList = NULL;
	DestroyList(dirDirectoriesList, direntDestroy);
	numDirectories = 0;
	dirDirectoriesList = NULL;

	if (!getcwd(saveFSelDir, MAXPATHLEN))
	{
		Msg(M_FATAL, "getcwd failed");
	}

	if (returnName)
	{
		if (*returnName == '/'
			|| !strncmp(returnName, "./", 2)
			|| !strncmp(returnName, "../", 3))
		{

			returnTotal = Strdup(returnName, "FileSelector");
		}
		else
		{

			returnTotal = (char*)Malloc(strlen(saveFSelDir) +
										strlen(returnName) + 2, "FileSelector");
			(void)strcpy(returnTotal, saveFSelDir);
			(void)strcat(returnTotal, "/");
			(void)strcat(returnTotal, returnName);
		}
		free(returnName);
	}

	(void)chdir(saveDir);
	return returnTotal;
}

static void
displayMenuTitle(char *title)
{
	uns8 attr= GetAttr();

	SetAttr(BRIGHT | (BLACK << 3) | WHITE);
	PrintLetter(' ');
	PrintString(title);

	while (cursorX < COLS)
	{
		PrintLetter(' ');
	}

	if (!GETCFG(mono))
	{
		SetCursor(COLS - 6, cursorY);
		SetAttr(BRIGHT | (BLACK << 3) | RED);
		PrintLetter('\200');
		SetAttr(BRIGHT | (RED << 3) | YELLOW);
		PrintLetter('\200');
		SetAttr(BRIGHT | (YELLOW << 3) | GREEN);
		PrintLetter('\200');
		SetAttr(BRIGHT | (GREEN << 3) | CYAN);
		PrintLetter('\200');
		SetAttr(BRIGHT | (CYAN << 3) | BLACK);
		PrintLetter('\200');
	}

	SetAttr(attr);
}

void
DisplayMenu(char **menu)
{
	int i;
	char **s;
	SetAttr(MENU_TEXT_ATTR);
	ClearScreen();

	displayMenuTitle(*menu++);

	SetAttr(MENU_SELECT_ATTR);
	while (*menu)
	{

		SetCursor(0, cursorY + 1);
		PrintLetter('\201');

		if (*menu[0] == '`')
		{
			SetAttr(MENU_DISABLED_ATTR);
			PrintString(&menu[0][1]);
			SetAttr(MENU_SELECT_ATTR);
		}
		else
		{
			PrintString(menu[0]);
		}
		while (cursorX < COLS - 1)
		{
			PrintLetter(' ');
		}

		PrintLetter('\205');
		menu++;
	}

	SetCursor(0, cursorY + 1);
	PrintLetter('\202');
	for (i = COLS - 2; i > 0; i--)
	{
		PrintLetter('\203');
	}
	PrintLetter('\204');

	SetAttr(MENU_TEXT_ATTR);
	SetCursor(0, ROWS - NOPTS(menuBottom) - 1);
	for (s = menuBottom; *s; s++)
	{
		PrintString(*s);
		SetCursor(0, cursorY + 1);
	}
	ScreenRefresh();
}

KeySym
BrowseList(int *curItem, int maxItems, int itemsPerPage, void (*printPage)(int))
{
	int i;
	uns16 mem;
	int item, page;
	KeySym key;
	assert(*curItem >= 0 && *curItem < maxItems);
	item = *curItem % itemsPerPage;
	page = *curItem / itemsPerPage;
	for (;;)
	{

		SetCursor(MENU_CURSOR_X, MENU_CURSOR_Y + item);
		mem = xyToAttr(cursorX, cursorY);
		if (GETCFG(mono))
		{
			for (i = COLS; i > 0; i--, mem++)
			{
				WR_BYTE(mem, Reverse[RD_BYTE(mem) & ~FLASH]);
			}
		}
		else
		{
			for (i = COLS; i > 0; i--, mem++)
			{
				WR_BYTE(mem,(RD_BYTE(mem) & ~PAPER) | (CYAN << 3));
			}
		}
		ScreenRefresh();
		key = GetKey();

		mem = xyToAttr(MENU_CURSOR_X, MENU_CURSOR_Y + item);
		if (GETCFG(mono))
		{
			for (i = COLS; i > 0; i--, mem++)
			{
				WR_BYTE(mem, Reverse[RD_BYTE(mem) & ~FLASH]);
			}
		}
		else
		{
			for (i = COLS; i > 0; i--, mem++)
			{
				WR_BYTE(mem,(RD_BYTE(mem) & ~PAPER) | (WHITE << 3) | BRIGHT);
			}
		}
		ScreenRefresh();
		switch (key)
		{
			case XK_Prior:
				if (--page < 0)
				{
					goto home;
				}
				if (printPage)
				{
					printPage(page);
				}
				break;
			case XK_Next:

				if (item + ++page * itemsPerPage >= maxItems)
				{
					goto end;
				}
				if (printPage)
				{
					printPage(page);
				}
				break;
			case XK_Up:
				if (--item < 0)
				{
					if (page > 0)
					{

						item = itemsPerPage - 1;
						page--;
						if (printPage)
						{
							printPage(page);
						}
					}
					else
					{
						goto end;
					}
				}
				break;
			case XK_Home:
		  home:
				item = 0;
				page = 0;
				if (printPage)
				{
					printPage(page);
				}
				break;
			case XK_Down:

				if (++item + page * itemsPerPage >= maxItems)
				{
					goto home;
				}
				if (item > itemsPerPage - 1)
				{
					if (page < maxItems / PAGE_SIZE)
					{

						item = 0;
						page++;
						if (printPage)
						{
							printPage(page);
						}
					}
					else
					{
						goto home;
					}
				}
				break;
			case XK_End:
			  end:
				item = (maxItems - 1) % itemsPerPage;
				page = (maxItems - 1) / itemsPerPage;
				if (printPage)
				{
					printPage(page);
				}
				break;
			default:
				goto quit;
		}
	}
  quit:
	*curItem = item + page * itemsPerPage;
	return KeyDialog;
}

