/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Authors: Thomas Ahn Kjaer & Des Herriott
 *          Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#include "config.h"
#ifdef XZX_PLUS3
#ifndef lint
static char rcsid[] = "$Id: fdc.c,v 4.2 1998/01/06 21:21:30 erik Rel $";
#endif

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include "z80.h"
#include "debug.h"
#include "resource.h"
#include "mem.h"
#include "io.h"
#include "util.h"
#include "dialog.h"
#include "fdc.h"

#define MAXSEC			18
#define IMG_DISK_INFO	0x100
#define IMG_TRACK_INFO	0x100
#define DISKREAD		255
#define DISKWRITE		254
#define E_NOTREADY		0
#define E_WRPROT		1
#define E_SEEK			2
#define E_CRC			3
#define E_READ			4
#define E_FORMAT		6
#define E_UNKNOWN		7
#define E_NOBOOT		35
#define E_OK			255
typedef struct {
	uns8 track;
	uns8 head;
	uns8 sector;
	uns8 bps;
	uns8 state1;
	uns8 state2;
	uns8 unused[2];
} sectorInfo;
typedef struct
{
	uns8 track;
	uns8 head;

	uns8 bps;
	uns8 spt;
	uns8 gap3;
	uns8 fill;
	sectorInfo si[MAXSEC];

	uns8 snum;
	uns8 fsec;
} trackInfo;
typedef struct
{
	uns8 tracks;
	uns8 heads;
	uns16 tsize;
	char *fn;
	FILE *fp;
} diskDef;
#ifdef DEBUG
#define DEB(x)			{ if (GETCFG(debug) & D_FDC) { x } }
#else
#define DEB(x)
#endif

static uns8 extractDiskInfo(uns8 *, diskDef *);
static uns8 extractTrackInfo(uns8 *, trackInfo *);
static uns8 discRW(uns8, uns8, uns8, diskDef *, uns8 *);
static void discError(uns8);
static void dosBoot(void);
static void ddInterface(void);
static void ddInit(void);
static void ddReadSector(void);
static void ddWriteSector(void);
static void ddCheckSector(void);
static void ddTestUnsuit(void);
static void ddLogin(void);
static void ddAsk1(void);
static void ddDriveStatus(void);
static void ddLseek(void);


static diskDef drives[2] = {
	{ 0, 0, 0, NULL, NULL },
	{ 0, 0, 0, NULL, NULL }
};
static uns8 sbuff[0x1000];
static uns8 wbuff[0x400];

void
FdcInit(void)
{
	struct stat buf;
	int i;
	for (i = 0; i < 2; i++)
	{
		if (!stat(GETCFG(disks[i]), &buf) && S_ISREG(buf.st_mode))
		{

			drives[i].fn = GETCFG(disks[i]);
		}
		else
		{
			Msg(M_WARN, "invalid disk image <%s>", GETCFG(disks[i]));

			free(GETCFG(disks[i]));
			SETCFG(disks[i], NULL);
		}
	}

	if (drives[0].fn
		&& drives[1].fn
		&& !strcmp(GETCFG(disks[0]), GETCFG(disks[1])))
	{
		Msg(M_WARN, "disk image <%s> allready assigned to drive %c",
			GETCFG(disks[1]), 'A');

		drives[1].fn = NULL;

		free(GETCFG(disks[1]));
		SETCFG(disks[1], NULL);
	}
}

static uns8
extractDiskInfo(uns8 *s, diskDef *d)
{
	static char ident[] = "MV - CPCEMU Disk-File\r\nDisk-Info\r";
	if (memcmp(s, ident, 8))
	{
		return 1;
	}
	s += 0x30;
	d->tracks = *(s++);
	d->heads = *(s++);
	d->tsize = *(s++);
	d->tsize |= (*(s++) << 8);
	return 0;
}

static uns8
extractTrackInfo(uns8 *s, trackInfo *t)
{
	static char ident[] = "Track-Info\r";
	if (memcmp(s, ident, 5))
	{
		return 1;
	}
	s += 0x10;
	t->track = *(s++);
	t->head = *(s++);
	s+=2;

	t->bps = *(s++);
	t->spt = *(s++);
	t->gap3 = *(s++);
	t->fill = *(s++);

	(void)memcpy(t->si, s, t->spt * sizeof(sectorInfo));
	t->fsec = t->si[0].sector;
	return 0;
}

static uns8
discRW(uns8 command, uns8 track, uns8 sector, diskDef *d, uns8 *secnum)
{
	long offset;
	trackInfo *curTrack;
	uns8 retval, i;
#ifdef NO_FPOS_T
	static long position;
#else
	static fpos_t position;
#endif

	if (!d->fn && FdcInsertDisk(d == &drives[1]) == -1)
	{
		return E_NOTREADY;
	}

	if (!(d->fp = Fopen(d->fn, "rb")))
	{
		return E_NOTREADY;
	}
	else
	{
		if (fread(sbuff, sizeof(uns8), IMG_DISK_INFO, d->fp) != IMG_DISK_INFO)
		{
			(void)fclose(d->fp);
			return E_NOTREADY;
		}
		else
			if (extractDiskInfo(sbuff, d))
			{
				(void)fclose(d->fp);
				return E_FORMAT;
			}
	}

	curTrack = (trackInfo *)Malloc(sizeof(trackInfo), "diskRW");

	offset = IMG_DISK_INFO + track * d->tsize;
	if (fseek(d->fp, offset, SEEK_SET))
	{
		retval = E_SEEK;
		goto ende;
	}

	if (fread(sbuff, sizeof(uns8), IMG_TRACK_INFO, d->fp) != IMG_TRACK_INFO)
	{
		retval = E_READ;
		goto ende;
	}

	if (extractTrackInfo(sbuff, curTrack))
	{
		retval = E_FORMAT;
		goto ende;
	}
	sector += curTrack->fsec;
	DEB(Msg(M_DEBUG, "reading track/sector = %d/%d", track, sector););
	retval = E_SEEK;
	for (i = 0; i < curTrack->spt; i++)
	{
		if (curTrack->si[i].sector == sector)
		{
			uns16 sector_size = 0x080 << curTrack->bps;
			(*secnum) = curTrack->si[i].sector;

			offset = i * sector_size;
			if (!fseek(d->fp, offset, SEEK_CUR))
			{
				retval = E_OK;
				if (command == DISKREAD)
				{
					if (fread(sbuff, sizeof(uns8), sector_size, d->fp)
						!= sector_size)
					{
						retval = E_CRC;
					}
				}
				else
				{
#ifdef NO_FPOS_T
					position = ftell(d->fp);
#else
					(void)fgetpos(d->fp, &position);
#endif
					if (!(d->fp = freopen(d->fn, "r+b", d->fp)))
					{
						free(curTrack);
						return E_WRPROT;
					}
#ifdef NO_FPOS_T
					(void)fseek(d->fp, position, SEEK_SET);
#else
					(void)fsetpos(d->fp, &position);
#endif
					if (fwrite(wbuff, sizeof(uns8), sector_size, d->fp)
						!= sector_size)
					{
						retval = E_WRPROT;
					}
				}
			}
		}
	}
  ende:
	free(curTrack);
	(void)fclose(d->fp);
	return retval;
}

static void
discError(uns8 errorcode)
{
	SCF();
	CCF();
	A = errorcode;
}

int
FdcInsertDisk(int drive)
{
	char *filename;
	if ((filename = FileSelector("Insert Disk", 1, "dsk", NULL, NULL)))
	{

		FdcEjectDisk(drive);

		if (!drives[1 - drive].fn || strcmp(filename, drives[1 - drive].fn))
		{
			drives[drive].fn = filename;
			return 0;
		}
		Msg(M_WARN, "disk image <%s> allready assigned to drive %c",
			filename, (char)('B' - drive));
		free(filename);
	}
	return -1;
}

void
FdcEjectDisk(int drive)
{
	if (drives[drive].fn)
	{
		free(drives[drive].fn);
		drives[drive].fn = NULL;
	}
}

char *
FdcDiskName(int drive)
{
	return (drives[drive].fn);
}

static void
dosBoot(void)
{
	uns8 secnum, errorcode;
	uns16 chksum, i;

	errorcode = discRW(DISKREAD, 0, 0, (diskDef *)drives, (uns8 *)&secnum);
	if (errorcode != E_OK)
	{
		discError(errorcode);
		RET();
		return;
	}

	chksum = 0;
	for (i = 0; i < 512; i++)
	{
		chksum += sbuff[i];
	}
	if ((chksum % 256) != 3)
	{
		Msg(M_WARN, "checksum fail <%d>", chksum);
		discError(E_NOBOOT);
		RET();
		return;
	}

	OutBank678(A);

	for (i = 0; i < 512; i++)
	{
		WR_BYTE(0xfe00 + i, sbuff[i]);
	}

	WR_BYTE(23388, Last0x7FFD);
	WR_BYTE(23399, Last0x1FFD);
	PC = 0xfe10;
}

static void
ddInterface(void)
{
	DEB(Msg(M_DEBUG, "DD INTERFACE"););
	SCF();
	RET();
}

static void
ddInit(void)
{
	RET();
}

static void
ddReadSector(void)
{
	uns8 errorcode, secnum, unit;
	int old_0xc000, i;
	unit = C & 0x01;
	DEB(Msg(M_DEBUG, "DD READ SECTOR");
		Msg(M_DEBUG, " page at 0xc000 %d", B);
		Msg(M_DEBUG, " unit %d", C);
		Msg(M_DEBUG, " logical track %d", D);
		Msg(M_DEBUG, " logical sector %d", E);
		Msg(M_DEBUG, " load at address 0x%04x", HL););
	old_0xc000 = RPAGE(0xc000);
	PageIn(3, RAM0 + B);

	errorcode = discRW(DISKREAD, D, E, (diskDef *)&drives[unit], &secnum);
	if (errorcode != E_OK)
	{
		discError(errorcode);
		RET();
		return;
	}

	for (i = 0; i < 512; i++)
	{
		WR_BYTE(HL + i, sbuff[i]);
	}
	PageIn(3, old_0xc000);
	SCF();
	RET();
}

static void
ddWriteSector(void)
{
	uns8 errorcode, secnum, unit;
	int old_0xc000, i;
	unit = C & 0x01;
	DEB(Msg(M_DEBUG, "DD WRITE SECTOR");
		Msg(M_DEBUG, " page at 0xc000 %d", B);
		Msg(M_DEBUG, " unit %d", C);
		Msg(M_DEBUG, " logical track %d", D);
		Msg(M_DEBUG, " logical sector %d", E);
		Msg(M_DEBUG, " load at address 0x%04x", HL););
	old_0xc000 = RPAGE(0xc000);
	PageIn(3, RAM0 + B);

	for (i = 0; i < 512; i++)
	{
		wbuff[i] = RD_BYTE(HL + i);
	}
	PageIn(3, old_0xc000);

	errorcode = discRW(DISKWRITE, D, E, (diskDef *)&drives[unit], &secnum);
	if (errorcode != E_OK)
	{
		discError(errorcode);
		RET();
		return;
	}
	SCF();
	RET();
}

static void
ddCheckSector(void)
{
	uns8 errorcode, secnum, unit;
	int old_0xc000, i;
	unit = C & 0x01;
	DEB(Msg(M_DEBUG, "DD CHECK SECTOR");
		Msg(M_DEBUG, " page at 0xc000 %d", B);
		Msg(M_DEBUG, " unit %d", C);
		Msg(M_DEBUG, " logical track %d", D);
		Msg(M_DEBUG, " logical sector %d", E);
		Msg(M_DEBUG, " load at address %d", HL););
	old_0xc000 = RPAGE(0xc000);
	PageIn(3, RAM0 + B);

	errorcode = discRW(DISKREAD, D, E, (diskDef *)&drives[unit], &secnum);
	if (errorcode != E_OK)
	{
		discError(errorcode);
		RET();
		return;
	}

	for (i = 0; i < 512; i++)
	{
		if (sbuff[i] != RD_BYTE(HL + i))
		{
			break;
		}
	}
	errorcode = (i == 512);
	PageIn(3, old_0xc000);
	if (!errorcode)
	{
		SET(Z_FLAG);
	}
	else
	{
		CLR(Z_FLAG);
	}
	SCF();
	RET();
}

static void
ddTestUnsuit(void)
{

	SCF();
	RET();
}

static void
ddLogin(void)
{
	uns8 unit, errorcode, secnum, disctype = 0;
	uns16 all = 0, hash = 0;
	unit = C & 0x1;
	DEB(Msg(M_DEBUG, "DD LOGIN unit %d", unit););

	errorcode = discRW(DISKREAD, 0,0, (diskDef *)&drives[unit], &secnum);
	if (errorcode != E_OK)
	{
		discError(errorcode);
		RET();
		return;
	}

	DEB(Msg(M_DEBUG, " found sector %d", secnum);
		Msg(M_DEBUG, " tracks=%d, sbuff[0]=%d", drives[unit].tracks, sbuff[0]););
	switch ((secnum & 0xc0) >> 6)
	{
		case 0:
			if (drives[unit].tracks == 80 && sbuff[0] == 3)
			{
				DEB(Msg(M_DEBUG, " PCW format"););
				disctype = 3;
				all = 0x00f0;
				hash = 0x0040;
			}
			if (drives[unit].tracks == 40)
			{
				DEB(Msg(M_DEBUG, " +3DOS format"););
				disctype = 0;
				all = 0x00c0;
				hash = 0x0010;
			}
			break;
		case 3:
			if (drives[unit].tracks == 40)
			{
				DEB(Msg(M_DEBUG, " CPC data format"););
				disctype = 2;
				all = 0x00c0;
				hash = 0x0010;
			}
			break;
		default:
			discError(E_FORMAT);
			RET();
			return;
	}
	A = disctype;
	DE = all;
	HL = hash;
	SCF();
	PC = 0x1cdb;
}

static void
ddAsk1(void)
{
	DEB(Msg(M_DEBUG, "DD ASK 1"););
	SCF();
	RET();
}

static void
ddDriveStatus(void)
{
	A = 0x20 | C;
	RET();
}

static void
ddLseek(void)
{
	DEB(Msg(M_DEBUG, "DD L SEEK, unit=%d, track=%d", C, D););
	SCF();
	RET();
}

void
FdcDdCheck(void)
{
	switch(PC)
	{

		case 0x1b1f:
			dosBoot();
			break;
		case 0x1f27:
			ddInterface();
			break;
		case 0x1f32:
			ddInit();
			break;
		case 0x1f47:
			DEB(Msg(M_DEBUG, "DD SETUP"););
			RET();
			break;
		case 0x1e7c:
			DEB(Msg(M_DEBUG, "DD SET RETRY"););
			RET();
			break;
		case 0x1bff:
			ddReadSector();
			break;
		case 0x1c0d:
			ddWriteSector();
			break;
		case 0x1c16:
			ddCheckSector();
			break;
		case 0x1c24:
			DEB(Msg(M_DEBUG, "DD FORMAT"););
			RET();
			break;
		case 0x1c36:
			DEB(Msg(M_DEBUG, "DD READ ID"););
			RET();
			break;
		case 0x1e65:
			ddTestUnsuit();
			break;
		case 0x1c80:
			ddLogin();
			break;
		case 0x1edd:
			ddAsk1();
			break;
		case 0x1ee9:
			ddDriveStatus();
			break;
		case 0x1e75:
			DEB(Msg(M_DEBUG, "DD EQUIPMENT"););
			RET();
			break;
		case 0x1bda:
			DEB(Msg(M_DEBUG, "DD ENCODE"););
			RET();
			break;
		case 0x1f76:
			ddLseek();
			break;
		case 0x20c3:
			DEB(Msg(M_DEBUG, "DD L READ"););
			RET();
			break;
		case 0x20cc:
			DEB(Msg(M_DEBUG, "DD L WRITE"););
			RET();
			break;
	}
}
#endif

