/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*               EST_Utterance access functions (from Lisp)              */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include "festival.h"
#include "festivalP.h"

static LISP item_features(LISP sitem);
static LISP item_features(EST_Item *s);
static LISP stream_tree_to_lisp(EST_Item *s);

LISP utt_iform(EST_Utterance &utt)
{
    return read_from_lstring(strintern(utt_iform_string(utt)));
}

const EST_String utt_iform_string(EST_Utterance &utt)
{
    return utt.f("iform").string();
}

const EST_String utt_type(EST_Utterance &utt)
{
    return utt.f("type").string();
}

static LISP utt_feat(LISP utt, LISP feat)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String f = get_c_string(feat);
    return lisp_val(u->f(f));
}

static LISP item_utt(LISP item)
{
    EST_Item *i = get_c_item(item);

    return siod_make_utt(get_utt(i));
}

static LISP item_sub_utt(LISP item)
{
    EST_Item *i = get_c_item(item);
    EST_Utterance *u = new EST_Utterance;

    sub_utterance(*u,i);
    
    return siod_make_utt(u);
}

static LISP utt_set_feat(LISP utt, LISP name, LISP value)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String n = get_c_string(name);
    
    if (TYPEP(value,tc_flonum))
	u->f.set(n,get_c_float(value));
    else
	u->f.set(n,get_c_string(value));

    return value;
}

static LISP utt_save(LISP utt, LISP fname, LISP ltype)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);
    if (fname == NIL)
	filename = "save.utt";
    EST_String type = get_c_string(ltype);
    if (ltype == NIL) type = "est_ascii";

    if (type == "est_ascii")
    {
	if (u->save(filename,type) != write_ok)
	{
	    cerr << "utt.save: saving to \"" << filename << "\" failed" <<
		endl;
	    festival_error();
	}
    }
    else
    {
	cerr << "utt.save: unknown save format" << endl;
	festival_error();
    }

    return utt;
}

static LISP utt_save_relation(LISP utt, LISP rname, LISP fname)
{
    // Save relation in named file
    EST_Utterance *u = get_c_utt(utt);
    EST_String relname = get_c_string(rname);
    EST_String filename = get_c_string(fname);
    if (fname == NIL)
	filename = "save.utt";
    EST_Relation *r = u->relation(relname);

    if (r->save(filename) != write_ok)
    {
	cerr << "utt.save.relation: saving to \"" << filename << "\" failed" <<
	    endl;
	festival_error();
    }
    return utt;
}

static LISP utt_load(LISP utt, LISP fname)
{
    EST_Utterance *u;
    if (utt == NIL)
	u = new EST_Utterance;
    else
	u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);

    if (u->load(filename) != 0)
    {
	cerr << "utt.load: loading from \"" << filename << "\" failed" <<
	    endl;
	festival_error();
    }

    if (utt == NIL)
	return siod_make_utt(u);
    else
	return utt;
}

static LISP utt_relation_load(LISP utt, LISP lrelname, LISP lfilename)
{
    EST_Utterance *u;
    if (utt == NIL)
	u = new EST_Utterance;
    else
	u = get_c_utt(utt);
    EST_String filename = get_c_string(lfilename);
    EST_String relname = get_c_string(lrelname);
    EST_Relation *rel = u->create_relation(relname);
    
    if (rel->load(filename,"esps") != 0)
    {
	cerr << "utt.load.relation: loading from \"" << filename << 
	    "\" failed" << endl;
	festival_error();
    }

    if (utt == NIL)
	return siod_make_utt(u);
    else
	return utt;
}

static LISP find_items(EST_Item *node)
{
    if (node == 0)
	return NIL;
    else
	return l_append(cons(siod_make_item(node),
			     find_items(down(node))),
			find_items(next(node)));
}

static LISP utt_copy_relation(LISP utt, LISP l_old_name, LISP l_new_name)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String old_name = get_c_string(l_old_name);
    EST_String new_name = get_c_string(l_new_name);

    u->create_relation(new_name);

    u->relation(new_name)->f = u->relation(old_name)->f;

    copy_relation(*u->relation(old_name), *u->relation(new_name));
	
    return utt;
}

static LISP utt_relation_print(LISP utt, LISP l_name)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String name = get_c_string(l_name);

    cout << *u->relation(name);
    return NIL;
}

static LISP utt_relation_items(LISP utt, LISP rname)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String relationname = get_c_string(rname);
	
    return find_items(u->relation(relationname)->head());
}

// could be merged with above

LISP utt_relation_tree(LISP utt, LISP sname)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String relname = get_c_string(sname);

    return stream_tree_to_lisp(u->relation(relname)->head());
}

static LISP stream_tree_to_lisp(EST_Item *s)
{
    if (s == 0)
	return NIL;
    else
    {
	LISP desc = cons(strintern(s->name()),
			 cons(item_features(s),NIL));
	return cons(cons(desc,stream_tree_to_lisp(s->down())),
		    stream_tree_to_lisp(s->next()));
    }
}

static LISP set_item_name(LISP litem, LISP newname)
{
    // Set a stream's name to newname
    EST_Item *s = get_c_item(litem);
    
    if (s != 0)
	s->set_name(get_c_string(newname));
    return litem;
}

static LISP utt_relation(LISP utt, LISP relname)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String rn = get_c_string(relname);
    EST_Item *r;

    r = u->relation(rn)->head();
    
    return siod_make_item(r);
}

static LISP utt_relation_create(LISP utt, LISP relname)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String rn = get_c_string(relname);

    u->create_relation(rn);
    
    return utt;
}

static LISP utt_relation_delete(LISP utt, LISP relname)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String rn = get_c_string(relname);

    u->delete_relation(rn);
    
    return utt;
}

static LISP utt_relationnames(LISP utt)
{
    // Return list of relation names
    EST_Utterance *u = get_c_utt(utt);
    LISP relnames = NIL;
    EST_Litem *p;

    for (p = u->relations.list.head(); p; p = next(p))
	relnames = cons(rintern(u->relations.list(p).k),relnames);

    return reverse(relnames);
}

static LISP item_relations(LISP si)
{
    // Return list of relation names
    EST_Item *s = get_c_item(si);
    LISP relnames = NIL;
    EST_Litem *p;

    for (p = s->relations().list.head(); p; p = next(p))
	relnames = cons(rintern(s->relations().list(p).k),relnames);

    return reverse(relnames);
}

static LISP item_relation_name(LISP si)
{
    // Return list of relation names
    EST_Item *s = get_c_item(si);

    return rintern(s->relation_name());
}

static LISP item_relation_remove(LISP item, LISP relname)
{
    EST_String rn = get_c_string(relname);
    EST_Item *si = get_c_item(item);
    remove_item(si,rn);
    // Just in case someone tries to access this again 
    // we set its contents to be 0 which will picked up by get_c_item
    USERVAL(item) = 0;
    return NIL;
}

static LISP utt_relation_append(LISP utt, LISP relname, LISP li)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String rn = get_c_string(relname);
    EST_Relation *r = u->relation(rn);
    EST_Item *s=0;

    if (!r)
	return NIL;
    if (siod_item_p(li))
	s = get_c_item(li);

    s = r->append(s);

    if (consp(li))
    {
	s->set_name(get_c_string(car(li)));
	add_item_features(s,car(cdr(li)));
    }
    
    return siod_make_item(s);
}

static LISP item_next(LISP li)
{
    return (li == NIL) ? NIL : siod_make_item(next(get_c_item(li)));
}

static LISP item_prev(LISP li)
{
    return (li == NIL) ? NIL : siod_make_item(prev(get_c_item(li)));
}

static LISP item_up(LISP li)
{
    return (li == NIL) ? NIL : siod_make_item(up(get_c_item(li)));
}

static LISP item_down(LISP li)
{
    return (li == NIL) ? NIL : siod_make_item(down(get_c_item(li)));
}

static LISP item_parent(LISP li)
{
    if (li == NIL) 
	return NIL;
    else 
	return siod_make_item(parent(get_c_item(li)));
}

static LISP item_daughter1(LISP li)
{
    if (li == NIL) 
	return NIL;
    else 
	return siod_make_item(daughter1(get_c_item(li)));
}

static LISP item_daughter2(LISP li)
{
    if (li == NIL) 
	return NIL;
    else 
	return siod_make_item(daughter2(get_c_item(li)));
}

static LISP item_daughtern(LISP li)
{
    if (li == NIL) 
	return NIL;
    else 
	return siod_make_item(daughtern(get_c_item(li)));
}

static LISP item_next_item(LISP li)
{
    return (li == NIL) ? NIL : siod_make_item(next_item(get_c_item(li)));
}

static LISP item_append_daughter(LISP li,LISP nli)
{
    EST_Item *l = get_c_item(li);
    EST_Item *s = 0;

    if (siod_item_p(li))
	s = get_c_item(nli);

    s = l->append_daughter(s);

    if (consp(nli))
    {
	s->set_name(get_c_string(car(nli)));
	add_item_features(s,car(cdr(nli)));
    }
    
    return siod_make_item(s);
}

static LISP item_insert_parent(LISP li,LISP nparent)
{
    EST_Item *l = get_c_item(li);
    EST_Item *s = 0;

    if (siod_item_p(nparent))
	s = get_c_item(nparent);

    s = l->insert_parent(s);

    if (consp(nparent))
    {
	s->set_name(get_c_string(car(nparent)));
	add_item_features(s,car(cdr(nparent)));
    }
    
    return siod_make_item(s);
}

static LISP item_insert(LISP li,LISP nli,LISP direction)
{
    EST_Item *n = get_c_item(li);
    EST_String dir;
    EST_Item *s;

    if (siod_item_p(nli))
	s = get_c_item(nli);
    else
	s = 0;

    if (direction)
	dir = get_c_string(direction);
    else
	dir = "after";

    if (dir == "after")
	s = n->insert_after(s);
    else if (dir == "before")
	s = n->insert_before(s);
    else if (dir == "above")
	s = n->insert_above(s);
    else if (dir == "below")
	s = n->insert_below(s);
    else
    {
	cerr << "item.insert: unknown direction \"" << dir << "\"" << endl;
	festival_error();
    }

    if (consp(nli))  // specified information
    {
	s->set_name(get_c_string(car(nli)));
	add_item_features(s,car(cdr(nli)));
    }

    return siod_make_item(s);
}

static LISP item_move_tree(LISP from,LISP to)
{
    EST_Item *f = get_c_item(from);
    EST_Item *t = get_c_item(to);

    if (move_sub_tree(f,t) == TRUE)
	return truth;
    else
	return NIL;  
}

static LISP item_merge_item(LISP from,LISP to)
{
    EST_Item *f = get_c_item(from);
    EST_Item *t = get_c_item(to);

    merge_item(f,t);
    return truth;
}

static LISP item_exchange_tree(LISP from,LISP to)
{
    EST_Item *f = get_c_item(from);
    EST_Item *t = get_c_item(to);

    if (exchange_sub_trees(f,t) == TRUE)
	return truth;
    else
	return NIL;  
}

static LISP item_relation(LISP lingitem,LISP relname)
{
    EST_Item *li = get_c_item(lingitem);
    EST_String rn = get_c_string(relname);
    return siod_make_item(li->as_relation(rn));
}

void utt_cleanup(EST_Utterance &u)
{
    // Remove all relations
    // This is called in the Initialization to ensure we can 
    // continue with a nice clean utterance

    u.relations.clear();
}

LISP make_utterance(LISP args,LISP env)
{
    /* Make an utterance structure from given input */
    (void)env;
    EST_Utterance *u = new EST_Utterance;
    EST_String t;
    LISP lform;

    u->f.set("type",get_c_string(car(args)));
    lform = car(cdr(args));
    u->f.set("iform",siod_sprint(lform));

    return siod_make_utt(u);
}

static LISP item_name(LISP litem)
{
    EST_Item *s = get_c_item(litem);

    return strintern(s->name());
}

static LISP item_delete(LISP litem)
{
    EST_Item *s = get_c_item(litem);

    s->unref_all();

    return NIL;
}

static LISP item_remove_feature(LISP litem,LISP fname)
{
    EST_Item *s = get_c_item(litem);
    EST_String f = get_c_string(fname);

    s->f_remove(f);

    return rintern("t");
}

static LISP item_features(LISP litem)
{
    // Return assoc list of features on this stream
    return item_features(get_c_item(litem));
}

static LISP item_features(EST_Item *s)
{
    LISP features = NIL;
    EST_TBI *p;

    for (p=s->features().head(); p != 0; p=next(p))
    {
	const EST_Val &v = s->features().fval(p);
	LISP fpair;
	
	if (v.type() == val_int)
	    fpair = make_param_int(s->features().fname(p), v.Int());
	else if (v.type() == val_float)
	    fpair = make_param_float(s->features().fname(p), v.Float());
	else
	    fpair = make_param_lisp(s->features().fname(p),strintern(v.string()));
	features = cons(fpair,features);
    }

    return reverse(features);
}

void add_item_features(EST_Item *s,LISP features)
{
    // Add LISP specified features to s;
    LISP f;

    for (f=features; f != NIL; f=cdr(f))
    {
	EST_String n = get_c_string(car(car(f)));
	EST_String v;
	
	if (consp(car(cdr(car(f)))))
	    v = siod_sprint(car(cdr(car(f))));
	else
	    v = get_c_string(car(cdr(car(f))));
	if (v.matches(RXint))
	    s->fset(n,atoi(v));
	else if (v.matches(RXdouble))
	    s->fset(n,atof(v));
	else
	    s->fset(n,v);
    }
}

void festival_utterance_init(void)
{
    // declare utterance specific Lisp functions 

    // Standard functions
    init_fsubr("Utterance",make_utterance,
 "(Utterance TYPE DATA)\n\
  Build an utterance of type TYPE from DATA.  Different TYPEs require\n\
  different types of data.  New types may be defined by defUttType.\n\
  [see Utterance types]");
    init_subr_2("utt.load",utt_load,
 "(utt.load UTT FILENAME)\n\
  Loads UTT with the streams and stream items described in FILENAME.\n\
  The format is Xlabel-like as saved by utt.save.  If UTT is nil a new\n\
  utterance is created, loaded and returned.  If FILENAME is \"-\"\n\
  the data is read from stdin.");
    init_subr_2("utt.feat",utt_feat,
 "(utt.feat UTT FEATNAME)\n\
  Return value of feature name in UTT.");
    init_subr_3("utt.set_feat",utt_set_feat,
 "(utt.set_feat UTT FEATNAME VALUE)\n\
  Set feature FEATNAME with VALUE in UTT.");
    init_subr_3("utt.relation.load",utt_relation_load,
 "(utt.relation.load UTT RELATIONNAME FILENAME)\n\
  Loads (and creates) RELATIONNAME from FILENAME into UTT.  FILENAME\n\
  should contain simple Xlabel format information.  The label part\n\
  may contain the label proper followed by semi-colon separated\n\
  pairs of feature and value.");
    init_subr_3("utt.save",utt_save,
 "(utt.save UTT FILENAME TYPE)\n\
  Save UTT in FILENAME in an Xlabel-like format.  If FILENAME is \"-\"\n\
  then print output to stdout.  TYPE may be nil or est_ascii");
    init_subr_3("utt.save.relation",utt_save_relation,
 "(utt.save UTT RELATIONNAME FILENAME)\n\
  Save relation RELATIONNAME in FILENAME in an Xlabel-like format. \n\
  If FILENAME is \"-\" then print output to stdout.");

    init_subr_3("utt.copy_relation", utt_copy_relation,
  "(utt.copy_relation UTT FROM TO)\n\
   copy relation \"from\" to a new relation \"to\"");
    init_subr_2("utt.relation.print", utt_relation_print,
  "(utt.relation.print UTT NAME)\n\
   print contents of relation NAME");

    init_subr_2("utt.relation.items",utt_relation_items,
 "(utt.relation.items UTT RELATIONNAME)\n\
  Return a list of stream items in RELATIONNAME in UTT. \n\
  If this relation is a tree, the parent streamitem is listed before its \b\
  daughters.");
    init_subr_2("utt.relation_tree",utt_relation_tree,
 "(utt.relation_tree UTT RELATIONNAME)\n\
  Return a tree of stream items in RELATIONNAME in UTT.  This will give a\n\
  simple list if the relation has no ups and downs. \n\
  [see Accessing an utterance]");
    init_subr_1("item.name",item_name,
 "(item.name ITEM)\n\
  Returns the name of ITEM. [see Accessing an utterance]");
    init_subr_1("item.delete",item_delete,
 "(item.delete ITEM)\n\
  Remove this item from all relations it is in and delete it.");
    init_subr_2("item.set_name",set_item_name,
 "(item.set_name ITEM NAME)\n\
  Sets ITEM's name to NAME. [see Accessing an utterance]");
    init_subr_1("item.features",item_features,
 "(item.features ITEM)\n\
  Returns all features in ITEM as an assoc list.");
    init_subr_2("item.remove_feature",item_remove_feature,
 "(item.remove_feature ITEM FNAME)\n\
  Remove feature named FNAME from ITEM.  Returns t is successfully\n\
  remove, nil if not found.");

    // New Utterance architecture (Relations and items)
    init_subr_2("utt.relation",utt_relation,
 "(utt.relation UTT RELATIONNAME)\n\
  Return root item of relation RELATIONNAME in UTT.");
    init_subr_2("utt.relation.create",utt_relation_create,
 "(utt.relation.create UTT RELATIONNAME)\n\
  Create new relation called RELATIONNAME in UTT.");
    init_subr_2("utt.relation.delete",utt_relation_delete,
 "(utt.relation.delete UTT RELATIONNAME)\n\
  Delete relation from utt, it the stream items are not linked elsewhere\n\
  in the utterance they will be deleted too.");
    init_subr_2("item.relation.remove",item_relation_remove,
 "(item.relation.remove ITEM RELATIONNAME)\n\
  Remove this item from Relation, if it apears in no other relation it\n\
  will be deleted too, in contrast item.delete will remove an item\n\
  from all other relations, while this just removes it from this relation.\n\
  Note this will also remove all daughters of this item in this \n\
  relation from this relation.");
    init_subr_1("utt.relationnames",utt_relationnames,
 "(utt.relationnames UTT)\n\
  List of all relations in this utterance.");
    init_subr_3("utt.relation.append",utt_relation_append,
 "(utt.relation.append UTT RELATIONNAME ITEM)\n\
  Append ITEM to top of RELATIONNAM in UTT.  ITEM may be\n\
  a LISP description of an item or an item itself.");
    init_subr_1("item.next",item_next,
 "(item.next ITEM)\n\
  Return the next ITEM in the current relation, or nil if there is\n\
  no next.");
    init_subr_1("item.prev",item_prev,
 "(item.prev ITEM)\n\
  Return the previous ITEM in the current relation, or nil if there\n\
  is no previous.");
    init_subr_1("item.up",item_up,
 "(item.up ITEM)\n\
  Return the item above ITEM, or nil if there is none.");
    init_subr_1("item.down",item_down,
 "(item.down ITEM)\n\
  Return the item below ITEM, or nil if there is none.");
    init_subr_1("item.parent",item_parent,
 "(item.parent ITEM)\n\
  Return the item of ITEM, or nil if there is none.");
    init_subr_1("item.daughter1",item_daughter1,
 "(item.daughter1 ITEM)\n\
  Return the first daughter of ITEM, or nil if there is none.");
    init_subr_1("item.daughter2",item_daughter2,
 "(item.daughter2 ITEM)\n\
  Return the second daughter of ITEM, or nil if there is none.");
    init_subr_1("item.daughtern",item_daughtern,
 "(item.daughtern ITEM)\n\
  Return the last daughter of ITEM, or nil if there is none.");
    init_subr_1("item.next_item",item_next_item,
 "(item.next_item ITEM)\n\
  Will give next item in this relation visiting every item in the \n\
  relation until the end.  Traverses in pre-order, root followed by \n\
  daughters (then siblings).");
    init_subr_2("item.append_daughter",item_append_daughter,
 "(item.append_daughter ITEM1 ITEM2)\n\
  Add a ITEM2 a new daughter to ITEM1 in the relation of ITEM1.\n\
  If ITEM2 is of type item then it is added directly otherwise\n\
  ITEM2 is treated as a description of an item and a one is created\n\
  with that description (name features).");
    init_subr_2("item.insert_parent",item_insert_parent,
 "(item.insert_parent ITEM1 ITEM2)\n\
  Insert a new parent between this ITEM1 and its parentm in ITEM1's \n\
  relation.  If ITEM2 is of type item then it is added directly, \n\
  otherwise it is treated as a description of an item and  one is created\n\
  with that description (name features).");
    init_subr_3("item.insert",item_insert,
 "(item.insert ITEM1 ITEM2 DIRECTION)\n\
  Insert ITEM2 in ITEM1's relation with repsect to DIRECTION.  If DIRECTION\n\
  is unspecified, after, is assumed.  Valid DIRECTIONS as before, after,\n\
  above and below.  Use the functions item.insert_parent and\n\
  item.append_daughter for specific tree adjoining.  If ITEM2 is of\n\
  type item then it is added directly, otherwise it is treated as a\n\
  description of an item and new one is created.");
    init_subr_2("item.relation",item_relation,
 "(item.relation ITEM RELATIONNAME)\n\
  Return the item such whose relation is RELATIONNAME.  If ITEM\n\
  is not in RELATIONNAME then nil is return.");
    init_subr_1("item.relations",item_relations,
 "(item.relationnames ITEM)\n\
  Return a list of names of the relations this item is in.");
    init_subr_1("item.relation.name",item_relation_name,
 "(item.relationname ITEM)\n\
  Return the name of the relation this ITEM is currently being viewed\n\
  through.");
    init_subr_2("item.move_tree",item_move_tree,
 "(item.move_tree FROM TO)\n\
  Move contents, and descendents of FROM to TO. Old daughters of TO are\n\
  deleted.  FROM will be deleted too if it is being viewed as the same\n\
  same relation as TO.  FROM will be deleted from its current place in\n\
  TO's relation. Returns t if successful, returns nil if TO is within FROM.");
    init_subr_2("item.exchange_trees",item_exchange_tree,
 "(item.exchange_tree FROM TO)\n\
  Exchanged contents of FROM and TO, and descendents of FROM and TO.\n\
  Returns t if successful, or nil if FROM or TO contain each other.");
    init_subr_2("item.merge",item_merge_item,
 "(item.merge FROM TO)\n\
  Merge FROM into TO making them the same items.  All features in FROM\n\
  are merged into TO and all references to FROM are made to point to TO.");
    init_subr_1("item.get_utt",item_utt,
  "(item.get_utt ITEM)\n\
  Get utterance from given ITEM (if possible).");
    init_subr_1("sub_utt",item_sub_utt,
  "(sub_utt ITEM)\n\
  Return a new utterance that contains a copy of this item and all its\n\
  descendents and related descendents.");
    
    init_subr_1("audio_mode",l_audio_mode,
 "(audio_mode MODE)\n\
 Control audio specific modes.  Five subcommands are supported. If\n\
 MODE is async, start the audio spooler so that Festival need not wait\n\
 for a waveform to complete playing before continuing.  If MODE is\n\
 sync wait for the audio spooler to empty, if running, and they cause\n\
 future plays to wait for the playing to complete before continuing.\n\
 Other MODEs are, close which waits for the audio spooler to finish\n\
 any waveforms in the queue and then closes the spooler (it will restart\n\
 on the next play), shutup, stops the current waveform playing and empties\n\
 the queue, and query which lists the files in the queue.  The queue may\n\
 be up to five waveforms long. [see Audio output]");

}
