/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                                                                       */
/*                 Author: Paul Taylor                                   */
/*                   Date: 6 Jan 1998                                    */
/* --------------------------------------------------------------------- */
/*            LPC residual synthesis alternative version                 */
/*                                                                       */
/*************************************************************************/

#include "UniSyn.h"
#include "module_support.h"
#include "EST_inline_utils.h"
#include "siod.h"
#include "EST_sigpr.h"
#include "us_diphone.h"
#include "EST_error.h"


void debug_options(EST_Relation &source_lab, EST_Relation &unit,
		   EST_Track &source_coef,
		   EST_Track &target_coef, EST_TVector <EST_Wave> &frames);

void add_wave_to_utterance(EST_Utterance &u, EST_Wave &sig, 
			const EST_String &name);

void target_times(EST_Track &source_pm, EST_Track &target_pm, 
		  EST_Relation &unit, EST_IVector &map);


float get_p_float(EST_String name, float def)
{
    if (siod_get_lval(name,  NULL) == NIL)
	return def;

    return get_c_float(siod_get_lval(name, "Error"));
}

void us_lpc_synthesis(EST_Utterance &utt)
{
    EST_Relation *source_lab;
    EST_IVector map;
    EST_FrameVector *frames;
    EST_Track *source_coef, *target_coef;
    EST_Wave *sig;
    EST_FVector gain;

    frames = (EST_FrameVector *) 
	utt.relation("Frames", 1)->head()->fP("frame", 1);
    source_coef = (EST_Track*)
	utt.relation("SourceCoef", 1)->head()->fP("coefs",1);
    target_coef = (EST_Track*)
	utt.relation("TargetCoef", 1)->head()->fP("coefs", 1);

    source_lab = utt.relation("SourceSegments");

    make_mapping(*source_lab, *source_coef, *(utt.relation("Segment")), 
		 *target_coef, map);

    sig = new EST_Wave;

    td_synthesis(*source_coef, *frames, *target_coef, *sig, map);
    lpc_synthesis(*source_coef, *target_coef, *frames, *sig, map);

//    sig->rescale(get_c_float(siod_get_lval("us_gain", "No gain defined")));
    add_wave_to_utterance(utt, *sig, "Wave");

    debug_options(*source_lab, *(utt.relation("Segment")),
		  *source_coef, *target_coef, *frames);    

}

void lpc_synthesis(EST_Track &source_coef, EST_Track &target_coef, 
		   EST_TVector <EST_Wave> &frames, EST_Wave &sig, 
		   EST_IVector &map)
{
    (void) frames;

    extract_frames(source_coef, target_coef, map);
#if 1
    // fast version but not very neat 
    lpc_filter_fast(target_coef, sig, sig);
#else
    // neat version but about 3 times slower
    EST_Wave res;
    res = sig;
    res.rescale(0.5);
    lpc_filter_1(target_coef, res, sig);
#endif
}


void debug_options(EST_Relation &source_lab, EST_Relation &unit,
		   EST_Track &source_coef,
		   EST_Track &target_coef, EST_TVector <EST_Wave> &frames)
{
    EST_IVector map;
    EST_Wave sig;
    EST_Relation pm_lab;

    if (siod_get_lval("us_debug_save_source", NULL) != NIL)
    {
	make_mapping(source_lab, source_coef, source_lab, source_coef, map);
	td_synthesis(source_coef, frames, source_coef, sig, map);
	sig.save("source_sig.wav", "nist");
    }

    if (siod_get_lval("us_debug_save_source_coefs", NULL) != NIL)
	source_coef.save("source_coefs.pm", "est");

    if (siod_get_lval("us_debug_save_target_pm", NULL) != NIL)
	target_coef.save("target_coefs.pm");

    if (siod_get_lval("us_debug_save_source_lab", NULL) != NIL)
	source_lab.save("source.lab");

    if (siod_get_lval("us_debug_save_target_unit_lab", NULL) != NIL)
	unit.save("target_unit.lab");

}

void debug_units(EST_Relation &unit_stream)
{
    if (siod_get_lval("us_debug_save_unit_lab", NULL) != NIL)
	unit_stream.save("source.lab");
}

void concatenate_coefs(EST_Relation &unit_stream, EST_Track &source_lpc)
{
    EST_Item *u;
    int num_source_frames, num_source_channels;
    float prev_time, abs_offset, rel_offset, period, offset;
    int i, j, k, l;
    EST_Track *coefs;

    num_source_frames = num_source_channels = 0;

//    cout << "coef no ola\n";

    for (u = unit_stream.head(); u; u = next(u))
    {
	EST_Track *t = (EST_Track *)u->fP("coefs");
	num_source_frames += t->num_frames();
	num_source_channels = t->num_channels();
    }

    source_lpc.resize(num_source_frames, num_source_channels);
    prev_time = 0.0;

    // copy basic information
    for (i = 0, l = 0, u = unit_stream.head(); u; u = next(u))
    {
	coefs = (EST_Track *)u->f("coefs").ptr();
	source_lpc.copy_setup(*coefs);

	for (j = 0; j < coefs->num_frames(); ++j, ++i)
	{
	    for (k = 0; k < coefs->num_channels(); ++k)
		source_lpc(i, k) = coefs->a_no_check(j, k);
	    source_lpc.t(i) = coefs->t(j) + prev_time;
	}

	prev_time = source_lpc.t(i - 1);
    }

    // adjust pitchmarks
    abs_offset = 0.0;
    rel_offset = 0.0;
    // absolute offset in seconds
    abs_offset = get_c_float(siod_get_lval("us_abs_offset", "zz"));
    // relative offset as a function of local pitch period
    rel_offset = get_c_float(siod_get_lval("us_rel_offset", "zz"));
//    cout << "abs_offset = " << abs_offset 
//	<< " rel_offset" << rel_offset << endl;

    for (i = 0; i < source_lpc.num_frames(); ++i)
    {
	period = get_time_frame_size(source_lpc, (i));
	offset = abs_offset + (rel_offset * period);
//	cout << "rel_offset = " << rel_offset 
//	    << " abs_offset = " << abs_offset << endl;
//	cout << "offset = " << offset << endl;
	source_lpc.t(i) = source_lpc.t(i) + offset;
    }
//    cout << "source lpc size:" << source_lpc.num_frames() << endl;
}

void window_signal(EST_Wave &sig, EST_Track &pm, 
		   EST_TVector<EST_Wave> &frames, int &i, float window_factor)
{
    int j;    
    int first_pm, last_pm;
    float first_pos, last_pos, period2, period;

    for (j = 0; j < pm.num_frames(); ++j, ++i)
    {
	// i.e. average of local pitch period
	period = get_time_frame_size(pm, j);
	last_pos = pm.t(j) + period;
	first_pos = pm.t(j) - period;
	
	period2 = period * 2.0;
	first_pos += period2 - (period2 * window_factor);
	last_pos += (period2 * window_factor) - period2;
	
	first_pm = (int)(first_pos * (float)sig.sample_rate());
	last_pm = (int)(last_pos * (float)sig.sample_rate());
	
	if (i >= frames.length()) // exceptionally may be required
	    frames.resize((int)(frames.length()*1.2));
	window_frame(frames[i], sig, first_pm, last_pm);
    }
}

void window_signal(EST_Relation &unit_stream, EST_TVector<EST_Wave> &frames, 
		   float window_factor)
{
    int i;
    EST_Wave *sig;
    EST_Item *u;
    EST_Track *coefs;
    int num = 0;
    
    for (u = unit_stream.head(); u; u = next(u))
	num += ((EST_Track *)u->f("coefs").ptr())->num_frames();
    frames.resize(num);
    
    for (i = 0, u = unit_stream.head(); u; u = next(u))
    {
	sig = (EST_Wave *)u->f("sig").ptr();
	coefs = (EST_Track *)u->f("coefs").ptr();

	window_signal(*sig, *coefs, frames, i, window_factor);
    }
    frames.resize(i);  // i should mostly = num, but just in case
}

void extract_frames(EST_Track &source_lpc, EST_Track &target_coef, 
		    EST_IVector &map)
{
    int i, j;
    int m;

    if (map.n() > target_coef.num_frames())
	m = target_coef.num_frames();
    else
	m = map.n();

    for (i = 0; i < m; ++i)
	for (j = 0; j < target_coef.num_channels(); ++j)
	    target_coef.a_no_check(i, j) = 
		source_lpc.a_no_check(map.a_no_check(i), j);
    // There can be one or two frames at the end of target_coed without
    // a map.  Here we zero them, blindly assuming they are in silence
    for ( ; i < target_coef.num_frames(); i++)
	for (j = 0; j < target_coef.num_channels(); ++j)
	    target_coef.a_no_check(i, j) = 0;
}

void td_synthesis(EST_Track &source_pm, EST_TVector<EST_Wave> &frames,
		  EST_Track &target_pm, EST_Wave &target_sig,
		  EST_IVector &map)
{
    int t_start;
    int i, j;
    float sr;
    (void) source_pm;
    int last_sample=0;
    
    if (frames.length()> 0)
	sr = (float)frames(0).sample_rate();
    else
	sr = 16000; // sort of, who cares, its going to be a 0 sample waveform

    if (map.n() > 0)
	last_sample = (int)(target_pm.end() * sr) + 
	    (frames(map(map.n()-1)).num_samples() / 2);

    target_sig.resize(last_sample);
    target_sig.fill(0);
    target_sig.set_sample_rate((int)sr);
    
    for (i = 0; i < map.n(); ++i)
    {
	const EST_Wave &frame = frames(map(i));
	t_start = ((int)(target_pm.t(i) * sr)
		   - (frame.num_samples() / 2));
	
	for (j = 0; j < frame.num_samples(); ++j)
	    if (j+t_start>=0)
		target_sig.a_no_check(j + t_start) += frame.a_no_check(j);
//		target_sig.a(j + t_start) += frame.a(j);
    }    
}

void target_times(EST_Track &source_pm, EST_Track &target_pm, 
		  EST_Relation &unit, EST_IVector &map)
{
    EST_Item *p;
    int i;
    float old_end;
    int old_index_end, new_index_end;
    
    for (p = unit.head(); p; p = next(p))
    {
	old_end = p->fF("end");
	old_index_end = source_pm.index(old_end);
	for (i = 0; i < map.n(); ++i)
	    if (map(i) > old_index_end)
		break;
	new_index_end = i - 1;
	*cdebug << "old end: " << old_end << endl;
	*cdebug << "old_index_end: " << old_index_end << endl;
	*cdebug << "new_index_end: " << new_index_end << endl;
	p->fset("end", target_pm.t(new_index_end));
    }
}

// might have hanning windowing in future etc
// USE EST_WINDOW functions!!
void window_frame(EST_Wave &frame, EST_Wave &whole, 
		  int start, int end)
{
    int i, j, send;
    EST_TBuffer<float> window;
    
    if (frame.num_samples() != (end-start))
	frame.resize(end-start);
    frame.set_sample_rate(whole.sample_rate());
    // Ensure we have a safe end
    if (end <= whole.num_samples())
	send = end;
    else
	send = whole.num_samples();
    
    EST_Window::make_window(window, end - start, "hanning");

    // To allow a_no_check access we do this is three stages
    for (i = 0, j = start; j < 0; ++i, ++j)
	frame.a_no_check(i) = 0;
    for ( ; j < send; ++i, ++j)
	frame.a_no_check(i) = (int)((float)whole.a_no_check(j) * window(i));
    for ( ; j < end; ++j,++i)
	frame.a_no_check(i) = 0;
}

void adjust_energy(EST_TVector<EST_Wave> &frames,  EST_FVector &gain)
{
    for (int i = 0; i < frames.n(); ++i)
    {
	*cdebug << "gain " << gain(i) << endl;
	frames[i].rescale(gain(i));
    }
}

int operator != (EST_Wave a, EST_Wave b)
{ (void)a; (void)b; return 1; }
int operator == (EST_Wave a, EST_Wave b)
{  (void)a; (void)b; return 0; }


#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TVector.cc"

template class EST_TVector<EST_Wave>;

#endif
static const EST_Wave def_val_s;
static EST_Wave error_return_s;
const EST_Wave *EST_TVector<EST_Wave>::def_val=&def_val_s;
EST_Wave *EST_TVector<EST_Wave>::error_return=&error_return_s;
