/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1996,1997                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify and distribute this software and its */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  November 1997                           */
/*-----------------------------------------------------------------------*/
/*  Build a WFST for a set of context dependent rewrite rules using the  */
/*  the algorithms from "An Efficient Compiler for Weighted Rewrite      */
/*  Rules", by Mehryar Mohri and Richard Sproat ACL 1996                 */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include "EST.h"
#include "EST_WFST.h"

static int wfst_build_main(int argc, char **argv);

int main(int argc, char **argv)
{

    wfst_build_main(argc,argv);

    exit(0);
    return 0;
}

static int wfst_build_main(int argc, char **argv)
{
    // Top level function generates a WFST from rules
    EST_Option al;
    EST_StrList files;
    EST_String outfile;

    parse_command_line2(argc, argv,
       EST_String("Usage:\n")+
       "wfst_build <options> <rulefile0> <rulefile1> ...\n"+
       "Build a weighted finite state transducer (WFST) from rules/wfsts\n"+
       "-type <string> {kk} Input rule type: kk, lts, rg, compose or regex\n"+
       "-determinize        Determinize WFST before saving it\n"+
       "-detmin             Determinize and minimize WFST before saving it\n"+
       "-o <ofile>          Output file for saved WFST (default stdout)\n"+
       "-heap <int> {210000}\n"+
       "                    Set size of Lisp heap, needed for large rulesets\n"+
       "-q                  Quiet mode, no summary generated\n",
		       files, al);
    
    if (al.present("-o"))
	outfile = al.val("-o");
    else
	outfile = "-";

    siod_init(al.ival("-heap"));

    LISP ruleset;
    LISP inalpha, outalpha;
    EST_WFST *wfst = new EST_WFST;

    if (al.val("-type") == "kk")
    {
	ruleset = car(vload(files(files.head()),1));
	kkcompile(ruleset,*wfst);
    }
    else if (al.val("-type") == "lts")
    {
	ruleset = car(vload(files(files.head()),1));
	ltscompile(ruleset,*wfst);
    }
    else if (al.val("-type") == "rg")
    {
	ruleset = car(vload(files(files.head()),1));
	rgcompile(ruleset,*wfst);
    }
    else if (al.val("-type") == "compose")
    {
	EST_WFST a,b;
	
	if (a.load(files.nth(0)) != format_ok) exit(-1);
	if (b.load(files.nth(1)) != format_ok) exit(-1);
	
	wfst->compose(a,b);
    }
//     else if (al.val("-type") == "concatenate")
//     {
// 	EST_WFST a,b;
	
// 	if (a.load(files.nth(0)) != format_ok) exit(-1);
// 	if (b.load(files.nth(1)) != format_ok) exit(-1);
	
// 	wfst->concatenate(a,b);
//     }
    else if (al.val("-type") == "difference")
    {
	EST_WFST a,b;
	
	if (a.load(files.nth(0)) != format_ok) exit(-1);
	if (b.load(files.nth(1)) != format_ok) exit(-1);
	
	wfst->difference(a,b);
    }
    else if (al.val("-type") == "regex")
    {
	ruleset = car(vload(files(files.head()),1));
	inalpha = siod_nth(0,ruleset);
	outalpha = siod_nth(1,ruleset);
	wfst->build_from_regex(inalpha,outalpha,car(cdr(cdr(ruleset))));
    }
    else
    {
	cerr << "wfst_build: unknown rule type \"" << al.val("-type") 
	    << "\"" << endl;
	exit(-1);
    }	

    if (al.present("-determinize"))
    {
	EST_WFST *dwfst = new EST_WFST;
	dwfst->determinize(*wfst);
	if (!al.present("-q"))
	{
	    cout << "wfst_build summary: " << endl;
	    cout << "   non-deterministic wfst: " << 
		wfst->summary() << endl;
	    cout << "       deterministic wfst: " << 
		dwfst->summary() << endl;
	}
	delete wfst;
	wfst = dwfst;
    }
    else if (al.present("-detmin"))
    {
	if (!al.present("-q"))
	{
	    cout << "wfst_build summary: " << endl;
	    cout << "   non-deterministic wfst: " << 
		wfst->summary() << endl;
	}
	EST_WFST *dwfst = new EST_WFST;
	dwfst->determinize(*wfst);
	delete wfst;
	if (!al.present("-q"))
	    cout << "       deterministic wfst: " << 
		dwfst->summary() << endl;
	EST_WFST *mwfst = new EST_WFST;
	mwfst->minimize(*dwfst);
	if (!al.present("-q"))
	    cout << "           minimized wfst: " << 
		mwfst->summary() << endl;
	delete dwfst;
	wfst = mwfst;
    }
    else
    {
	if (!al.present("-q"))
	    cout << "wfst_build: " << wfst->summary() << endl;
    }

    wfst->save(outfile);
    delete wfst;

    return 0;
}

