/*
	==========================================================================
	Module: AForm.js
	==========================================================================
	Pre-canned functions to map the user interface into JavaScripts.
	==========================================================================
	The Software, including this file, is subject ot the End User License
	Agreement.
	Copyright (c) 1997, Adobe Systems Incorporated, All Rights Reserved.
	==========================================================================
*/

console.println("Acrobat Forms Built-in Functions Version 3.5");

RE_NUMBER_ENTRY_DOT_SEP = new Array(
	"[+-]?\\d*\\.?\\d*"
);
RE_NUMBER_COMMIT_DOT_SEP = new Array(
	"[+-]?\\d+(\\.\\d+)?",		/* -1.0 or -1 */
	"[+-]?\\.\\d+",				/* -.1 */
	"[+-]?\\d+\\."				/* -1. */
);
RE_NUMBER_ENTRY_COMMA_SEP = new Array(
	"[+-]?\\d*,?\\d*"
);
RE_NUMBER_COMMIT_COMMA_SEP = new Array(
	"[+-]?\\d+([.,]\\d+)?",		/* -1,0 or -1 */
	"[+-]?[.,]\\d+",				/* -,1 */
	"[+-]?\\d+[.,]"				/* -1, */
);
RE_ZIP_ENTRY = new Array(
	"\\d{0,5}"
);
RE_ZIP_COMMIT = new Array(
	"\\d{5}"
);
RE_ZIP4_ENTRY = new Array(
	"\\d{0,5}(\\.|[- ])?\\d{0,4}"
);
RE_ZIP4_COMMIT = new Array(
	"\\d{5}(\\.|[- ])?\\d{4}"
);
RE_PHONE_ENTRY = new Array(
	"\\d{0,3}(\\.|[- ])?\\d{0,3}(\\.|[- ])?\\d{0,4}",		/* 555-1234 or 408 555-1234 */
	"\\(\\d{0,3}",											/* (408 */
	"\\(\\d{0,3}\\)(\\.|[- ])?\\d{0,3}(\\.|[- ])?\\d{0,4}",	/* (408) 555-1234 */
		/* (allow the addition of parens as an afterthought) */
	"\\(\\d{0,3}(\\.|[- ])?\\d{0,3}(\\.|[- ])?\\d{0,4}",	/* (408 555-1234 */
	"\\d{0,3}\\)(\\.|[- ])?\\d{0,3}(\\.|[- ])?\\d{0,4}",	/* 408) 555-1234 */
	"011(\\.|[- \\d])*"										/* international */
);
RE_PHONE_COMMIT = new Array(
	"\\d{3}(\\.|[- ])?\\d{4}",							/* 555-1234 */
	"\\d{3}(\\.|[- ])?\\d{3}(\\.|[- ])?\\d{4}",			/* 408 555-1234 */
	"\\(\\d{3}\\)(\\.|[- ])?\\d{3}(\\.|[- ])?\\d{4}",	/* (408) 555-1234 */
	"011(\\.|[- \\d])*"									/* international */
);
RE_SSN_ENTRY = new Array(
	"\\d{0,3}(\\.|[- ])?\\d{0,2}(\\.|[- ])?\\d{0,4}"
);
RE_SSN_COMMIT = new Array(
	"\\d{3}(\\.|[- ])?\\d{2}(\\.|[- ])?\\d{4}"
);

/* ==== Convenience Objects ==== */

/* Stock color definitions for ease of use. */
color = new Object();
color.transparent = new Array("T");
color.black = new Array("G", 0);
color.white = new Array("G", 1);
color.red = new Array("RGB", 1, 0, 0);
color.green = new Array("RGB", 0, 1, 0);
color.blue = new Array("RGB", 0, 0, 1);
color.cyan = new Array("CMYK", 1, 0, 0, 0);
color.magenta = new Array("CMYK", 0, 1, 0, 0);
color.yellow = new Array("CMYK", 0, 0, 1, 0);

/* Font definitions for ease of use */
font = new Object();
font.Times = "Times-Roman";
font.TimesB = "Times-Bold";
font.TimesI = "Times-Italic";
font.TimesBI = "Times-BoldItalic";
font.Helv = "Helvetica";
font.HelvB = "Helvetica-Bold";
font.HelvI = "Helvetica-Oblique";
font.HelvBI = "Helvetica-BoldOblique";
font.Cour = "Courier";
font.CourB = "Courier-Bold";
font.CourI = "Courier-Oblique";
font.CourBI = "Courier-BoldOblique";
font.Symbol = "Symbol";
font.ZapfD = "ZapfDingbats";

/* Border style definitions for ease of use */
border = new Object();
border.s = "solid";
border.d = "dashed";
border.b = "beveled";
border.i = "inset";
border.u = "underline";

/* Radio/Check button style definitions for ease of use */
style = new Object();
style.ch = "check";
style.cr = "cross";
style.di = "diamond";
style.ci = "circle";
style.st = "star";
style.sq = "square"; 

/* highlight modes of on a push button */
highlight = new Object();
highlight.n = "none";
highlight.i = "invert";
highlight.p = "push";
highlight.o = "outline";

/* zoom types for a document */
zoomtype = new Object();
zoomtype.none = "NoVary";
zoomtype.fitW = "FitWidth";
zoomtype.fitH = "FitHeight";
zoomtype.fitP = "FitPage";
zoomtype.fitV = "FitVisibleWidth"

/* ==== Functions ==== */

/* these may be used a lot -- they are language independent */

AFDigitsRegExp = new RegExp();
AFDigitsRegExp.compile("\\d+");
AFPMRegExp = new RegExp();
AFPMRegExp.compile(IDS_PM, "i");
AFAMRegExp = new RegExp();
AFAMRegExp.compile(IDS_AM, "i");
AFTimeLongRegExp = new RegExp();
AFTimeLongRegExp.compile("\\d{1,2}:\\d{1,2}:\\d{1,2}");
AFTimeShortRegExp = new RegExp();
AFTimeShortRegExp.compile("\\d{1,2}:\\d{1,2}");

function AFBuildRegExps(array)
/* Takes an array of strings and turns it into an array of compiled regular
 * expressions -- is used for the definitions that follow */
{
	var retVal = new Array();

	retVal.length = array.length;
	for(var it = 0; it < array.length; it++)
	{
		retVal[it] = new RegExp();
		retVal[it].compile(array[it], "i");
	}
	return retVal;
}

/* these may be used a lot -- they are NOT language independent and are 
 * derived from the localizable (RE_xxx) stuff above */

AFNumberDotSepEntryRegExp = AFBuildRegExps(RE_NUMBER_ENTRY_DOT_SEP);
AFNumberDotSepCommitRegExp = AFBuildRegExps(RE_NUMBER_COMMIT_DOT_SEP);
AFNumberCommaSepEntryRegExp = AFBuildRegExps(RE_NUMBER_ENTRY_COMMA_SEP);
AFNumberCommaSepCommitRegExp = AFBuildRegExps(RE_NUMBER_COMMIT_COMMA_SEP);
AFZipEntryRegExp = AFBuildRegExps(RE_ZIP_ENTRY);
AFZipCommitRegExp = AFBuildRegExps(RE_ZIP_COMMIT);
AFZip4EntryRegExp = AFBuildRegExps(RE_ZIP4_ENTRY);
AFZip4CommitRegExp = AFBuildRegExps(RE_ZIP4_COMMIT);
AFPhoneEntryRegExp = AFBuildRegExps(RE_PHONE_ENTRY);
AFPhoneCommitRegExp = AFBuildRegExps(RE_PHONE_COMMIT);
AFSSNEntryRegExp = AFBuildRegExps(RE_SSN_ENTRY);
AFSSNCommitRegExp = AFBuildRegExps(RE_SSN_COMMIT);
AFMonthsRegExp = AFBuildRegExps(IDS_MONTH_INFO.split(/\[\d+\]/));

function AFExactMatch(rePatterns, sString)
{	/* match a string against an array of RegExps */
	var it;

	if(!rePatterns.length && rePatterns.test(sString) && RegExp.lastMatch == sString)
		return true;
	for(it = 0; it < rePatterns.length; it++)
		if(rePatterns[it].test(sString) && RegExp.lastMatch == sString)
			return it + 1;
	return 0;
}

function AFExtractNums(string)
{	/* returns an array of numbers that it managed to extract from the given 
	 * string or null on failure */
	var nums = new Array();

	if (string.charAt(0) == '.' || string.charAt(0) == ',')
		string = "0" + string;
		 
	while(AFDigitsRegExp.test(string)) {
		nums.length++;
		nums[nums.length - 1] = RegExp.lastMatch;
		string = RegExp.rightContext;
	}
	if(nums.length >= 1) return nums;
	return null;
}

function AFMakeNumber(string)
{	/* attempts to make a number out of a string that may not use '.' as the
	 * seperator; it expects that the number is fairly well-behaved other than
	 * possibly having a non-JavaScript friendly separator */
	var type = typeof string;

	if (type == "number")
		return string;
	if (type != "string")
		return null;

	var array = AFExtractNums(string);

	if(array)
		return array.join(".") * (string.indexOf("-") >= 0 ? -1.0 : 1.0);
	else
		return null;
}

function AFExtractRegExp(rePattern, string)
{	/* attempts to match the pattern given against the string given; on 
	 * success, returns an array containing (at index 0) the initial
	 * string with the matched text removed and (at index 1) the matched
	 * text; on failure, returns null */
	var retVal = new Array();

	if(rePattern.test(string))
	{
		retVal.length = 2;
		retVal[0] = RegExp.leftContext + RegExp.rightContext;
		retVal[1] = RegExp.lastMatch;
		return retVal;
	}
	return null;
}

function AFExtractTime(string)
{	/* attempts to extract a WELL FORMED time from a string; returned 
	 * is an array in the same vein as AFExtractRegExp or null on
	 * failure. a WELL FORMED time looks like 12:23:56pm */
	
	var pm = "";
	var info;

	info = AFExtractRegExp(AFPMRegExp, string);
	if(info)
	{
		pm = info[1];
		string = info[0];
	}
	info = AFExtractRegExp(AFAMRegExp, string);
	if(info)
		string = info[0];
	info = AFExtractRegExp(AFTimeLongRegExp, string);
	if(info)
	{
		info[1] += pm;
		return info;
	}
	info = AFExtractRegExp(AFTimeShortRegExp, string);
	if(info)
	{
		info[1] += pm;
		return info;
	}
	return null;
}

function AFDateGetUTC(date)
{	/* returns the utc for a date object */
	return Date.UTC(date.getYear(), date.getMonth(), date.getDate(),
		date.getHours(), date.getMinutes(), date.getSeconds());
}

function AFGetMonthIndex(string)
{	/* attempts to identify the given string as a month or a valid abbreviation,
	 * it expects the given string to be the valid month from the matced RegExp.
	 * returns the month index (January = 1) or zero on failure */
	var monthre = new RegExp(string + "\\[(\\d+)\\]", "i");
	var result = monthre.exec(IDS_MONTH_INFO);
	
	if(string && result) return 1.0 * result[1];
	return 0;
}

function AFMatchMonth(string)
{	/* attempts to find a valid month embedded in a string; returns the month
	 * index (January = 1) or zero on failure */

	for(var it = 0; it < AFMonthsRegExp.length; it++)
		if(AFMonthsRegExp[it].test(string))
			return AFGetMonthIndex(RegExp.lastMatch);
	return 0;
}

function AFGetMonthString(index)
{	/* returns the string corresponding to the given month or a string that
	 * is indicative of the fact that the index was invalid */
	var monthre = new RegExp("(\\w+)\\[" + index + "\\]");
	var result = monthre.exec(IDS_MONTH_INFO);

	if(result) return result[1];
	return IDS_INVALID_MONTH;
}

function AFParseTime(string, date)
{	/* attempts to parse a string containing a time; returns null on failure
	 * or a Date object on success. Time can be in ugly format. */
	var pm;
	var nums = AFExtractNums(string);
	if (!date)
		date = new Date();
	var hour;

	if(!string) return date;
	if(!nums) return null;
	if(nums.length < 2 || nums.length > 3) return null;
	if(AFPMRegExp.test(string)) pm = true;
	else pm = false;
	hour = new Number(nums[0]); /* force it to number */
	if(pm)
	{
		if(hour != 12) hour += 12;
	}
	else
	{
		if(hour == 12) hour = 0;
	}
	date.setHours(hour);
	date.setMinutes(nums[1]);
	if(nums.length == 3) date.setSeconds(nums[2]);
	else date.setSeconds(0);
	return date;
}

function AFParseDate(string, longEntry, shortEntry, wordMonthEntry, monthYearEntry)
{	/* attempts to parse a string containing some form of date; returns null
	 * on failure or a Date object on success */
	var nums;
	var year, month;
	var date;
	var info = AFExtractTime(string);

	if(!string) return new Date();

	if(info)
		string = info[0];

	date = new Date();
	nums = AFExtractNums(string);
	if(!nums) return null;
	if(nums.length == 3)
	{
		year = nums[eval(longEntry.charAt(0))];
		date.setYear(year);
		date.setMonth(nums[eval(longEntry.charAt(1))] - 1);
		date.setDate(nums[eval(longEntry.charAt(2))]);
		if (info)
			AFParseTime(info[1], date);
		return date;
	}
	month = AFMatchMonth(string);
	if(nums.length == 2)
	{
		if(month)
		{
			year = nums[eval(wordMonthEntry.charAt(0))];
			date.setYear(year);
			date.setMonth(month - 1);
			date.setDate(nums[eval(wordMonthEntry.charAt(1))]);
			if (info)
				AFParseTime(info[1], date);
			return date;
		}
		if(monthYearEntry)
		{
			date.setDate(1);
			year = nums[eval(monthYearEntry.charAt(0))];
			date.setYear(year);
			date.setMonth(nums[eval(monthYearEntry.charAt(1))] - 1);
		}
		else
		{
			date.setMonth(nums[eval(shortEntry.charAt(0))] - 1);
			date.setDate(nums[eval(shortEntry.charAt(1))]);
		}
		if (info)
			AFParseTime(info[1], date);
		return date;
	}
	if(month && nums.length == 1)
	{
		if(monthYearEntry)
		{
			year = nums[0];
			date.setDate(1);
			date.setMonth(month - 1);
			date.setYear(year);
		}
		else
		{
			date.setMonth(month - 1);
			date.setDate(nums[0]);
		}
		if (info)
			AFParseTime(info[1], date);
		return date;
	}

	return null;
}

function AFParseDateWithPDF(value, pdf)
/* Intelligently parses a string for a date specification using the format for
 * hints */
{
	switch (pdf) {
	
		case 0:
		case 1:
		case 2:
		case 4:
		case 5:
		case 6:
		case 10:
		case 11:
		case 12:
		case 13:
			return AFParseDate(value, "201", "01", "10", "");	/* standard: ymd, md, yd */
			break;
		case 3:
		case 8:
		case 9:
			return AFParseDate(value, "201", "", "10", "10");	/* month / year only */
			break;
		case 7:
			return AFParseDate(value, "012", "01", "10", "");	/* backwards: y-m-d */
			break;
	}
	return null;
}

function AFMergeChange(event)
{	/* merges the last change with the uncomitted change */
	var prefix, postfix;
	var value = event.value;

	if(event.willCommit) return event.value;
	if(event.selStart >= 0)
		prefix = value.substring(0, event.selStart);
	else prefix = "";
	if(event.selEnd >= 0 && event.selEnd <= value.length)
		postfix = value.substring(event.selEnd, value.length);
	else postfix = "";
	return prefix + event.change + postfix;
}

function AFRange_Validate(bGreaterThan, nGreaterThan, bLessThan, nLessThan)
{       /* This function validates the current event to ensure that its value is 
	** within the specified range. */
	var cError = "";

	if (bGreaterThan && bLessThan) {
		if (event.value < nGreaterThan || event.value > nLessThan)
			cError = util.printf(IDS_GT_AND_LT, nGreaterThan, nLessThan);
	} else if (bGreaterThan) {
		if (event.value < nGreaterThan)
			cError = util.printf(IDS_GREATER_THAN, nGreaterThan);
	} else if (bLessThan) {
		if (event.value > nLessThan)
			cError = util.printf(IDS_LESS_THAN, nLessThan);
	}
	
	if (cError != "") {
		app.alert(cError, 0);
		event.rc = false;
	}
}

function AFSimpleInit(cFunction)
{	/* Convenience function used by AFSimple_Calculate. */
	switch (cFunction)
	{
		case "PRD":
			return 1.0;
			break;
		case "MIN":
			return Number.MAX_VALUE;
			break;
		case "MAX":
			return Number.MIN_VALUE;
			break;
	}

	return 0.0;
}

function AFSimple(cFunction, nValue1, nValue2)
{	/* Convenience function used by AFSimple_Calculate. */
	var nValue = 1.0 * nValue1;

	/* Have to do this otherwise JavaScript thinks it's dealing with strings. */
	nValue1 = 1.0 * nValue1;
	nValue2 = 1.0 * nValue2;

	switch (cFunction)
	{
		case "AVG":
		case "SUM":
			nValue = nValue1 + nValue2;
			break;
		case "PRD":
			nValue = nValue1 * nValue2;
			break;
		case "MIN":
			if (nValue2 < nValue1)
				nValue = nValue2;
			break;
		case "MAX":
			if (nValue2 > nValue1)
				nValue = nValue2;
			break;
	}

	return nValue;
}

function AFSimple_Calculate(cFunction, cFields)
{   /* Calculates the sum, average, product, etc. of the listed field values. */
	var cField = "", nFields = 0, i, j, f, a;
	var nValue = AFSimpleInit(cFunction);
	var nTemp;

	/* Loop over the fields string extracting the user specified fields. */
	for (i = 0; i < cFields.length; i++) {
		var c = cFields.charAt(i);

		if (c == ",") {
			/* Found a field, process it's value. */
			f = this.getField(cField);

			a = f.getArray();
			for (var j = 0; j < a.length; j++) {
				nTemp = AFMakeNumber(a[j].value);
				nValue = AFSimple(cFunction, nValue, nTemp);
				nFields++;
			}
			cField = "";

			/* Skip any white space after the comma. */
			while (i + 1 < cFields.length && cFields.charAt(i + 1) == " ") {
				i++;
			}
		} else {        
			/* Build up the field name. */
			cField += c;
		}
	}
	
	/* Don't forget to process the last field, if any. */
	if (cField != "") {
		f = this.getField(cField);

		var a = f.getArray();

		for (var j = 0; j < a.length; j++) {
			nTemp = AFMakeNumber(a[j].value);
			nValue = AFSimple(cFunction, nValue, nTemp);
			nFields++;
		}
	}

	if (cFunction == "AVG" && nFields > 0)
		nValue /= nFields;

	event.value = nValue;
}

function AFNumber_Keystroke(nDec, sepStyle, negStyle, currStyle, strCurrency, bCurrencyPrepend)
{       /* This function validates the current keystroke event to make sure the
		key pressed is reasonable for a numeric field. */

	var value = AFMergeChange(event);
	var commit, noCommit;

	if(!value) return;
	if(sepStyle > 1)
	{
		commit = AFNumberCommaSepCommitRegExp;
		noCommit = AFNumberCommaSepEntryRegExp;
	}
	else
	{
		commit = AFNumberDotSepCommitRegExp;
		noCommit = AFNumberDotSepEntryRegExp;
	}
	if(!AFExactMatch(event.willCommit ? commit : noCommit, value))
	{
		app.beep(0);
		event.rc = false;
	}
}

function AFPercent_Keystroke(nDec, sepStyle)
{
		AFNumber_Keystroke(nDec, sepStyle, 0, 0, "", true);
}

function AFSpecial_Keystroke(psf)
{       /* This function validates the current keystroke event to make sure the
		key pressed is reasonable for a "special" field. */
		
	/* The special formats, indicated by psf, are:
	
	psf             format
	---             ------
	0               zip code
	1               zip + 4
	2               phone
	3				SSN
	
	*/

	var value = AFMergeChange(event);
	var commit, noCommit;

	if(!value) return;
	switch (psf)
	{
		case 0:
			commit = AFZipCommitRegExp;
			noCommit = AFZipEntryRegExp;
			break;
		case 1:
			commit = AFZip4CommitRegExp;
			noCommit = AFZip4EntryRegExp;
			break;
		case 2:
			commit = AFPhoneCommitRegExp;
			noCommit = AFPhoneEntryRegExp;
			break;
		case 3:
			commit = AFSSNCommitRegExp;
			noCommit = AFSSNEntryRegExp;
			break;
	}		
	if(!AFExactMatch(event.willCommit ? commit : noCommit, value))
	{
		app.beep(0);
		event.rc = false;
	}
}

function AFDate_Keystroke(pdf)
{       /* This function validates the current keystroke event to make sure the
		key pressed is reasonable for a date field. */

		if(event.willCommit && !AFParseDateWithPDF(AFMergeChange(event), pdf))
						/* dates are only validated on commit */
		{
			app.beep(0);
			event.rc = false;
		}
}

function AFTime_Keystroke(ptf)
{       /* This function validates the current keystroke event to make sure the
		key pressed is reasonable for a time field. */

		if(event.willCommit && !AFParseTime(event.value, null))
						/* times are only validated on commit */
		{
			app.beep(0);
			event.rc = false;
		}
}

function AFNumber_Format(nDec, sepStyle, negStyle, currStyle, strCurrency, bCurrencyPrepend)
{       /* This function formats a numeric value according to the parameters. */

	var value = AFMakeNumber(event.value);
	var sign = (value < 0 ? -1 : 1);
	var f = event.target;

	if(value == null)
	{
		event.value = "";
		return;
	}	
	if ((negStyle == 2 /* ParensBlack */ || negStyle == 3 /* ParensRed */) && value < 0)
		var formatStr = "(";
	else 
		var formatStr = "";
	
	if (bCurrencyPrepend)
		formatStr = formatStr + strCurrency;
		
	formatStr = formatStr + "%," + sepStyle + "." + nDec + "f";
	if (! bCurrencyPrepend)
		formatStr = formatStr + strCurrency;
		
	if ((negStyle == 2 /* ParensBlack */ || negStyle == 3 /* ParensRed */) && value < 0)
		formatStr = formatStr + ")";

	if (negStyle != 0 /* MinusBlack */ || bCurrencyPrepend)
		value = Math.abs(value);
		
	if (sign > 0 )
		f.fgColor = color.black;
	else if (negStyle == 1 /* Red */ || negStyle == 3 /* ParensRed */)
		f.fgColor = color.red;
	else
		f.fgColor = color.black;
	
	var tmp = util.printf(formatStr, value);
	if (sign < 0 && bCurrencyPrepend && negStyle == 0)
		tmp = '-' + tmp; /* prepend the -ve sign */
	event.value = tmp;
}

function AFPercent_Format(nDec, sepStyle)
{       /* This function formats a percentage value according to the parameters. */

	var value = AFMakeNumber(event.value) * 100;
	
	var formatStr = "%," + sepStyle + "." + nDec + "f";
		
	if(value == null)
	{
		event.value = "";
		return;
	}	

	value = util.printf(formatStr, value);
	
	event.value = value + "%";
}

function AFSpecial_Format(psf)
{   /* This function formats a "special" value according to the "PropsSpecialFormat" parameter psf. */
	/* The special formats, indicated by psf, are: 0 = zip code, 1 = zip + 4, 2 = phone, 3 = SSN. */
	var value = event.value;

	if(!value) return;	
	switch (psf) {
	
		case 0:                         
			var formatStr = "99999";
			break;
		case 1:                         
			var formatStr = "99999-9999";
			break;
		case 2:                         /* must distinguish between 2 styles: with and without area code */
			var NumbersStr = util.printx("9999999999", value);      /* try to suck out 10 numeric chars */
			if (NumbersStr.length >= 10 )
				var formatStr = "(999) 999-9999";
			else 
				var formatStr = "999-9999";
			break;
		case 3:
			var formatStr = "999-99-9999";
			break;
	}
		
	event.value = util.printx(formatStr, value);
}

function AFDate_Format(pdf)
{       /* This function formats a date value according to the "PropsDateFormat" parameter pdf. */
	/* The date formats, indicated by pdf, are:
	
	pdf             format                                                          
	---             ------                                                          
	0               PDF_M_D         [ 1/3                           ]       
	1               PDF_M_D_YY      [ 1/3/81                        ]       
	2               PDF_MM_DD_YY    [ 01/03/81                      ]       
	3               PDF_MM_YY       [ 03/81                         ]
	4               PDF_D_MMM,      [ 3-Jan                         ]       
	5               PDF_D_MMM_YY    [ 3-Jan-81                      ]       
	6               PDF_DD_MMM_YY   [ 03-Jan-81                     ]       
	7               PDF_YY_MM_DD    [ 81-01-03                      ]
	8               PDF_MMM_YY      [ Jan-81                        ]       
	9               PDF_MMMM_YY     [ January-81            ]       
	10              PDF_MMM_D_YYYY  [ Jan 3, 1981           ]
	11              PDF_MMMM_D_YYYY [ January 3, 1981       ]       
	12              PDF_M_D_YY_12HR [ 1/3/81 2:30 PM        ]       
	13              PDF_M_D_YY_24HR [ 1/3/81 14:30          ]       

	*/

	var date = AFParseDateWithPDF(event.value, pdf);
	var formatStr;
	
	if(!event.value) return;	/* Blank fields remain blank */
	if(!date)
	{
		event.value = "";
		return;
	}
	
	switch (pdf) {
	
		case 0:
			formatStr = "m/d";
			break;
		case 1:
			formatStr = "m/d/yy";
			break;
		case 2:
			formatStr = "mm/dd/yy";
			break;
		case 3:
			formatStr = "mm/yy";
			break;
		case 4:
			formatStr = "d-mmm";
			break;
		case 5:
			formatStr = "d-mmm-yy";
			break;
		case 6:
			formatStr = "dd-mmm-yy";
			break;
		case 7:
			formatStr = "yy-mm-dd";
			break;
		case 8:
			formatStr = "mmm-yy";
			break;
		case 9:
			formatStr = "mmmm-yy";
			break;
		case 10:
			formatStr = "mmm d, yyyy";
			break;
		case 11:
			formatStr = "mmmm d, yyyy";
			break;
		case 12:
			formatStr = "m/d/yy h:MM tt";
			break;
		case 13:
			formatStr = "m/d/yy HH:MM";
			break;
	}
	
	event.value = util.printd(formatStr, date);
}

function AFTime_Format(ptf)
{       /* This function formats a time value according to the "PropsTimeFormat" parameter ptf. */
	/* The time formats, indicated by ptf, are:
	
	ptf             format                                                          
	---             ------                                                          
	0               PTF_24HR_MM     [ 14:30                         ]
	1               PTF_12HR_MM     [ 2:30 PM                       ]
	2               PTF_24HR_MM_SS  [ 14:30:15                      ]
	3               PTF_12HR_MM_SS  [ 2:30:15 PM            ]

	*/

	var date = new AFParseTime(event.value, null);
	var formatStr;
	
	if(!event.value) return;	/* Blank fields remain blank */
	if(!date) {
		event.value = "";
		return;
	}

	switch (ptf) {
	
		case 0:
			formatStr = "HH:MM";
			break;
		case 1:
			formatStr = "h:MM tt";
			break;
		case 2:
			formatStr = "HH:MM:ss";
			break;
		case 3:
			formatStr = "h:MM:ss tt";
			break;
	}
	
	event.value = util.printd(formatStr, date);
}
