/*
 * Copyright 1999-2005 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */
 
package org.apache.fop.layoutmgr;

import org.apache.fop.area.RegionReference;
import org.apache.fop.area.Area;
import org.apache.fop.area.Block;
import org.apache.fop.fo.pagination.StaticContent;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * LayoutManager for an fo:flow object.
 * Its parent LM is the PageSequenceLayoutManager.
 * This LM is responsible for getting columns of the appropriate size
 * and filling them with block-level areas generated by its children.
 */
public class StaticContentLayoutManager extends BlockStackingLayoutManager {

    private RegionReference region;
    private List blockBreaks = new ArrayList();

    public StaticContentLayoutManager(StaticContent node) {
        super(node);
    }

    /**
     * Sets the region reference
     * @param region region reference
     */
    public void setRegionReference(RegionReference region) {
        this.region = region;
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#getNextBreakPoss(LayoutContext)
     */
    public BreakPoss getNextBreakPoss(LayoutContext context) {

        // currently active LM
        LayoutManager curLM;

        while ((curLM = getChildLM()) != null) {
            // Make break positions and return page break
            // Set up a LayoutContext
            BreakPoss bp;
            LayoutContext childLC = context;
            if (!curLM.isFinished()) {
                if ((bp = curLM.getNextBreakPoss(childLC)) != null) {
                    blockBreaks.add(bp);
                    if (bp.isForcedBreak()) {
                        log.error("Forced breaks are not allowed in "
                                + "static content - ignoring");
                        return null;
                    }
                }
            }
        }
        setFinished(true);
        if (blockBreaks.size() > 0) {
            return new BreakPoss(
              new LeafPosition(this, blockBreaks.size() - 1));
        }
        return null;
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#addAreas(PositionIterator, LayoutContext)
     */
    public void addAreas(PositionIterator parentIter, LayoutContext layoutContext) {

        LayoutManager childLM;
        int iStartPos = 0;
        LayoutContext lc = new LayoutContext(0);
        while (parentIter.hasNext()) {
            LeafPosition lfp = (LeafPosition) parentIter.next();
            // Add the block areas to Area
            PositionIterator breakPosIter =
              new BreakPossPosIter(blockBreaks, iStartPos,
                                   lfp.getLeafPos() + 1);
            iStartPos = lfp.getLeafPos() + 1;
            while ((childLM = breakPosIter.getNextChildLM()) != null) {
                childLM.addAreas(breakPosIter, lc);
            }
        }

        blockBreaks.clear();
        flush();
        region = null;
    }


    /**
     * Add child area to a the correct container, depending on its
     * area class. A Flow can fill at most one area container of any class
     * at any one time. The actual work is done by BlockStackingLM.
     * @see org.apache.fop.layoutmgr.LayoutManager#addChildArea(Area)
     */
    public void addChildArea(Area childArea) {
        region.addBlock((Block)childArea);
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#getParentArea(Area)
     */
    public Area getParentArea(Area childArea) {
        return region;
    }

    /**
     * Markers are not allowed in static areas so this reports an
     * error and does nothing.
     *
     * @see org.apache.fop.layoutmgr.LayoutManager
     */
    public void addMarker(Map marks, boolean start, boolean isfirst) {
        // error markers not allowed in static
        log.error("Cannot add marker to static areas");
    }
}

