/*
 * BackupRestore.java
 *
 * Created on December 13, 2000, 6:08 AM
 */

package com.sun.enterprise.config.backup;

import com.sun.enterprise.config.backup.status.BackupStatus;
import com.sun.enterprise.config.backup.status.SnapShotStatus;
import com.sun.enterprise.config.backup.status.Status;
import com.sun.enterprise.config.backup.status.RemoveStatus;
import com.sun.enterprise.config.backup.status.RestoreStatus;
import com.sun.enterprise.config.backup.status.UndoRestoreStatus;
import com.sun.enterprise.config.backup.status.StatusConstants;
import com.sun.enterprise.config.backup.pluggable.EnvironmentFactory;
import com.sun.enterprise.config.backup.pluggable.BackupEnvironment;
import com.sun.enterprise.config.backup.pluggable.BackupSynchronizer;
import com.sun.enterprise.config.backup.phase.BackupPhases;
import com.sun.enterprise.config.backup.phase.RestorePhases;
import com.sun.enterprise.config.backup.phase.UndoRestorePhases;
import com.sun.enterprise.config.backup.phase.RemovePhases;

import com.sun.enterprise.config.backup.pluggable.BackupStorage;
import com.sun.enterprise.config.backup.utils.LoggerHelper;
import com.sun.enterprise.config.backup.utils.BackupHelper;
import com.sun.enterprise.config.backup.utils.LocalStringsHelper;
import com.sun.enterprise.config.backup.utils.FactoryHelper;

import java.io.File;

/**
 * This Interface is exposed by the mbean and also
 * by the backupRestore utility.
 * This class has all methods required to implement
 * backup-restore functionality
 *
 * @author  sridatta
 */
public class BackupRestoreImpl implements BackupRestore, StatusConstants {
                    
    public BackupRestoreImpl(String type) throws BackupException {
        BackupHelper.setExecutionTypeInEnvironment(type);
    }
    
    /**
     * Backsup the configuration into a default location
     * DefaultLocation is specified by $DomainRoot/backup
     * Use the other method backup(String targetFileName) if
     * you need to specify the source file
     *
     * @return BackupRestoreStatus Detailed backup information
     */
    public BackupStatus backup(String userInfo) {
        BackupStatus bs = BackupHelper.createBackupStatus(userInfo, true);
        String targetFile = BackupHelper.getAbsoluteBackupFileName(bs);
        return backup(targetFile, bs);
    }
    
    /**
     * backs up the configuration to the target filename
     * @param targetFile absolute path including the filename
     * @return BackupRestoreStatus Detailed backup information
     */
      public BackupStatus backup(String absoluteTargetFile, String userInfo) {
          BackupStatus bs = BackupHelper.createBackupStatus(userInfo, true);
          return backup(absoluteTargetFile, bs);
      }
      
      /**
       * backupStatus cannot be null 
       * It SHOULD be initialized
       */
      private synchronized BackupStatus backup(String absoluteTargetFile, 
                                        BackupStatus backupStatus) {
         BackupPhases bp = new BackupPhases();
         try {
              bp.assertStatusNotNull(backupStatus);
              bp.setStatusInRegistry(backupStatus);
              bp.assertStatusNotFailed();
              bp.lock();
              bp.setAbsoluteBackupFileName(absoluteTargetFile);
              long size = bp.backup();
              bp.setBackupFileSize(size);
              bp.setSuccess();
              bp.addStatusFile();
          } catch(BackupAbortException bae) {
              LoggerHelper.error("error_during_backup", bae);
              bp.setFailure(bae);
          }catch (Exception e) {
              LoggerHelper.error("error_during_backup", e);
              bp.setFailure(e);
          } finally {
              bp.monitorBackupDir();
              bp.addHistory();
              bp.unlock();
              bp.addAssistance();
          }
          return bp.getStatus();
      }
    
    /**
     * Restores the configuration from the last backup file
     *
     * @return RestoreStatus Detailed Status Information
     */
      public synchronized RestoreStatus restore() {
              File f = BackupHelper.getLastBackupFile();
              RestoreStatus rs = BackupHelper.createRestoreStatus(f);
        return restore(f, rs);
    }
    
    /**
      * restores the configuration from the specified file
     * Note that this can be a backup from this domain OR
     * from other domain!! Hence it can be used to replicate
     * servers.
     *
     * @param can be absolute or relative path.
     * @return RestoreStatus detailed status information
     */
      public synchronized RestoreStatus restore(String backupFileName) {
    
          
          String absName = null;
          RestoreStatus rs = null;
          //FIXME: NEEDS TO BE CLEANED UP
          try {
            absName = 
               BackupHelper.getAbsoluteBackupFileName(backupFileName);
          } catch(Exception e) {
                  return RestoreStatus.
                    createFailedRestoreStatus(
                        "wrong_backup_file", backupFileName);
          }
              
          File f = new File(absName);
          rs = BackupHelper.createRestoreStatus(f);
          return restore(f, rs);
    }
      
    private RestoreStatus restore(File f, RestoreStatus rs) {
        //lock
        //delete previous snapshot (yes, it is lost for ever)
        //perform a best-effort snapshot
        //restore files
        //enter history
        //return restorestatus
        RestorePhases rp = new RestorePhases();

        try {
              rp.assertStatusNotNull(rs);
              rp.setStatusInRegistry(rs);
              rp.assertStatusNotFailed();
              rp.lock();
              rp.takeSnapShot();
              rp.deleteTarget();
              rp.performRestore(f);
              rp.setSuccess();
          } catch (BackupAbortException bae) {
              LoggerHelper.error("error_during_restore: " + bae.getCode(), bae);
              rp.setFailure(bae);
          } catch (Exception e) {
              LoggerHelper.error("error_during_restore", e);
              rp.setFailure(e);
          } finally {
              rp.addHistory();
              rp.unlock();
              rp.addAssistance();
          }
          return rp.getStatus();          
    }
    
    /** 
     * This method can be used to undo a just performed restore.
     * Every restore keeps the information before restoring in
     * a file. So that restore can be undone.
     *
     * @UndoRestoreStatus status of the undo restore operation
     */
      public synchronized UndoRestoreStatus undoRestore() {
          LoggerHelper.info("risky_operation");
          UndoRestorePhases urp = new UndoRestorePhases();
          File f = BackupHelper.getLastSnapShotFile();
          UndoRestoreStatus urs = BackupHelper.createUndoRestoreStatus(f);        
        
        try {
              urp.assertStatusNotNull(urs);
              urp.setStatusInRegistry(urs);
              urp.assertStatusNotFailed();
              urp.lock();
              urp.deleteTarget();
              urp.performUndoRestore(f, urs);
          } catch (BackupAbortException bae) {
              LoggerHelper.error("error_during_undo_restore", bae);
              urp.setFailure(bae);
          } catch (Exception e) {
              LoggerHelper.error("error_during_undo_restore", e);
              urp.setFailure(e);
          } finally {
              urp.addHistory();
              urp.unlock();
              urp.addAssistance();
          }
          return urp.getStatus();          
    }
    
    /**
     * Get a report of all available backups
     */
    public synchronized BackupStatus[] listBackups() {
        BackupStatus[] bss = null;
        try {
            bss = FactoryHelper.getBackupStorageMonitor().listBackupStatus();
        } catch(Exception e) {
            LoggerHelper.error("error_listing_backups", e);
        } 
        return bss;
    }

    /**
     * remove by specifying a file name
     * relative path. (absolute path can be deleted by hand)
     */
    public synchronized RemoveStatus removeBackup(String backupFileName) {
        RemovePhases rp = new RemovePhases();
        return rp.deleteBackup(backupFileName, false);        
    }
    
    /**
     * Get a report of previous history
     * -1 for 
     */
    public synchronized BackupHistory getBackupHistory(int num) 
                                    throws BackupException {
        try {                                        
            return FactoryHelper.getHistoryManager().getBackupHistory(num);
        } catch(Exception e) {
            LoggerHelper.warning("error_retrieving_backup_history");
            LoggerHelper.fine("", e);
            throw new BackupException(
                "error_retrieving_backup_history", 
                LocalStringsHelper.getString("error_retrieving_backup_history"),e);
        }
    }
    
    /**
     * cleanup the history
     */
    public synchronized BackupHistory clearBackupHistory() {
        BackupHistory bh = null;
        try {
            bh = getBackupHistory(-1);    
        } catch(Exception e) {
            //ignore
        }
        try {
            FactoryHelper.getHistoryManager().clearHistory();
        } catch(Exception e) {
            LoggerHelper.warning("error_clearing_backup_history");
            LoggerHelper.fine("", e);
        }
        return bh;
    }
    
    private BackupEnvironment getEnv() throws BackupException {
        return FactoryHelper.getEnv();
    }
   
}
