package com.ibm.bsf.util;

import java.util.*;
import java.io.*;
import java.beans.*;
import com.ibm.bsf.*;

/**
 * This is a base implementation of the BSFEngine interface which
 * engine implementations may choose to extend to get the basic 
 * methods of the interface implemented. 
 * <p>
 *
 * @author   Sanjiva Weerawarana
 */
public abstract class BSFEngineImpl implements BSFEngine {
  protected BSFManager mgr; // my manager
  protected String lang; // my language string
  protected Vector declaredBeans; // BSFDeclaredBeans

  // properties mirrored from my manager. debug property may be set
  // directly to override what the manager gives (but if it changes 
  // in the manager it'll change here too)
  protected boolean debug;
  protected PrintStream debugStream;
  protected String classPath;
  protected String tempDir;
  protected ClassLoader classLoader;


  //////////////////////////////////////////////////////////////////////////

  /**
   * initialize the engine; called right after construction by 
   * the manager. Declared beans are simply kept in a vector and
   * that's it. Subclasses must do whatever they want with it.
   */
  public void initialize (BSFManager mgr, String lang,
                          Vector declaredBeans) throws BSFException {
    this.mgr = mgr;
    this.lang = lang;
    this.declaredBeans = declaredBeans;

    // initialize my properties from those of the manager. It'll send
    // propchange events to me in the future of they change.
    this.debug = mgr.getDebug ();
    this.debugStream = mgr.getDebugStream ();
    this.classPath = mgr.getClassPath ();
    this.tempDir = mgr.getTempDir ();
    this.classLoader = mgr.getClassLoader ();
  }

  //////////////////////////////////////////////////////////////////////////

  /** 
   * Turn on/off debugging output to System.err. 
   */
  public void setDebug (boolean debug) {
    this.debug = debug;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Receive property change events from the manager and update my fields
   * as needed.
   *
   * @param e PropertyChange event with the change data
   */
  public void propertyChange (PropertyChangeEvent e) {
    String name  = e.getPropertyName ();
    Object value = e.getNewValue ();

    if (name.equals ("debug")) {
      debug = ((Boolean) value).booleanValue ();
    } else if (name.equals ("debugStream")) {
      debugStream = (PrintStream) value;
    } else if (name.equals ("classPath")) {
      classPath = (String) value;
    } else if (name.equals ("tempDir")) {
      tempDir = (String) value;
    } else if (name.equals ("classLoader")) {
      classLoader = (ClassLoader) value;
    }
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Default impl of apply - calls eval ignorning parameters and returns
   * the result.
   */
  public Object apply (String source, int lineNo, int columnNo,
		       Object funcBody, Vector paramNames, Vector arguments)
       throws BSFException {
    return eval (source, lineNo, columnNo, funcBody);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Default impl of execute - calls eval and ignores the result.
   */
  public void exec (String source, int lineNo, int columnNo,
		    Object script) throws BSFException {
    eval (source, lineNo, columnNo, script);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Default impl of compileExpr - generates code that'll create a new
   * manager, evaluate the expression, and return the value.
   */
  public void compileExpr (String source, int lineNo, int columnNo,
			   Object expr, CodeBuffer cb) throws BSFException {
    ObjInfo bsfInfo = cb.getSymbol("bsf");

    if (bsfInfo == null) {
      bsfInfo = new ObjInfo (BSFManager.class, "bsf");
      cb.addFieldDeclaration ("com.ibm.bsf.BSFManager bsf = " +
                              "new com.ibm.bsf.BSFManager();");
      cb.putSymbol ("bsf", bsfInfo);
    }

    String evalString = (bsfInfo.objName +
                         ".eval(\"" +
                         lang + "\", \"" +
                         StringUtils.cleanString(source) +
                         "\", " + lineNo + ", " +
                         columnNo + "," +
                         StringUtils.lineSeparator +
                         StringUtils.getSafeString (expr.toString ()) +
                         ")");

    ObjInfo oldRet = cb.getFinalServiceMethodStatement ();

    if (oldRet != null && oldRet.isExecutable ()) {
      cb.addServiceMethodStatement (oldRet.objName + ";");
    }

    cb.setFinalServiceMethodStatement (new ObjInfo (Object.class, evalString));

    cb.addServiceMethodException ("com.ibm.bsf.BSFException");
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Default impl of compileApply - calls compileExpr ignorning parameters.
   */
  public void compileApply (String source, int lineNo, int columnNo,
			    Object funcBody, Vector paramNames, 
			    Vector arguments, CodeBuffer cb)
       throws BSFException {
    compileExpr (source, lineNo, columnNo, funcBody, cb);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Default impl of compileScript - generates code that'll create a new
   * manager, and execute the script.
   */
  public void compileScript (String source, int lineNo, int columnNo,
           Object script, CodeBuffer cb) throws BSFException {
    ObjInfo bsfInfo = cb.getSymbol("bsf");

    if (bsfInfo == null) {
      bsfInfo = new ObjInfo (BSFManager.class, "bsf");
      cb.addFieldDeclaration ("com.ibm.bsf.BSFManager bsf = " +
                              "new com.ibm.bsf.BSFManager();");
      cb.putSymbol ("bsf", bsfInfo);
    }

    String execString = (bsfInfo.objName +
                         ".exec(\"" +
                         lang + "\", \"" +
                         StringUtils.cleanString(source) +
                         "\", " + lineNo + ", " +
                         columnNo + "," +
                         StringUtils.lineSeparator +
                         StringUtils.getSafeString (script.toString ()) +
                         ")");

    ObjInfo oldRet = cb.getFinalServiceMethodStatement ();

    if (oldRet != null && oldRet.isExecutable ()) {
      cb.addServiceMethodStatement (oldRet.objName + ";");
    }

    cb.setFinalServiceMethodStatement (new ObjInfo (void.class, execString));

    cb.addServiceMethodException ("com.ibm.bsf.BSFException");
  }

  //////////////////////////////////////////////////////////////////////////

  public void declareBean (BSFDeclaredBean bean) throws BSFException {
    throw new BSFException (BSFException.REASON_UNSUPPORTED_FEATURE,
                            "language " + lang +
                            " does not support declareBean(...).");
  }

  //////////////////////////////////////////////////////////////////////////

  public void undeclareBean (BSFDeclaredBean bean) throws BSFException {
    throw new BSFException (BSFException.REASON_UNSUPPORTED_FEATURE,
                            "language " + lang +
                            " does not support undeclareBean(...).");
  }

  public void terminate () {
    mgr = null;
    declaredBeans = null;
    debugStream = null;
    classLoader = null;
  }
}
