/* BipartiteMatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.LinkedList;
import java.util.Queue;

/**
 * Invariant computer which returns the matching number(maximum size of
 * matching) for bipartite graphs.
 */
public class BipartiteMatchingNumber extends GeneralMatchingNumber {
    
    /** Array which represents the alternating tree. */
    private int[] alternatingTree;
    /** Array indicating vertices deleted from graph. */
    private boolean[] deleted;
    
    /**
     * Creates a new instance of BipartiteMatchingNumber
     */
    public BipartiteMatchingNumber() {
        super();
        alternatingTree = null;
        deleted = null;
    }
    
    /**
     * Computes the the matchingnumber of a bipartite graph in
     * O(m*n) time.
     */
    public int compute(int[][] adjlist) {
        super.adjlist = adjlist;
        initializeMatching(SIMPLE_GREEDY_HEUR);
        // Initialisation of an aiding list for alternating tree
        Queue<Integer> q = new LinkedList<Integer>();
        // Alternating tree
        alternatingTree = new int[adjlist.length];
        // Array indicating vertices in alternating tree
        boolean[] t = new boolean[adjlist.length];
        // Initialize array deleted
        deleted = new boolean[adjlist.length];
        for(int i = 0; i < deleted.length; i++) {
            deleted[i] = false;
        }
        // Keep track of nr of unvisited unsaturated vertices
        int unmatched = adjlist.length - (2*matchingSize);
        for(int v = 0; v < adjlist.length && unmatched >= 2; v++) {
            if(matching[v] == UNSATURATED) // if v is unsaturated
            {
                q.clear(); // Remove all current elements from list
                q.offer(new Integer(v));
                // Initialize array t
                for(int i = 0; i < t.length; i++)
                    t[i] = false;
                t[v] = true;
                // Initialize alternating tree
                for(int i = 0; i < alternatingTree.length; i++)
                    alternatingTree[i] = -1;
                // lastNode represents the last node of an augmenting path
                int lastNode = findAugmentingPath(q, t);
                if(lastNode != -1) { // found augmenting path!
                    updateMatching(lastNode);
                    unmatched = unmatched - 2;
                }
                //else...   cleanUpHeuristic(t): not used!
            }
        }
        return matchingSize;
    }
    
    /**
     * Updates the current matching via an augmenting path.
     * @param lastNode The last node of the augmenting path.
     */
    private void updateMatching(int lastNode) {
        while(alternatingTree[lastNode] != -1) // till root
        {
            if(matching[alternatingTree[lastNode]] != lastNode) {
                matching[lastNode] = alternatingTree[lastNode];
                matching[alternatingTree[lastNode]] = lastNode;
            }
            lastNode = alternatingTree[lastNode];
        }
        matchingSize++; // Current matching contains one edge more then the former.
    }
    
    /**
     * This method searches an augmenting path. If such an
     * augmenting path exists then it returns the last node of
     * the corresponding augmenting path otherwise it returns -1.
     * @param q List of vertices to be explored.
     * @param t Array booleans indicating vertices in alternating tree.
     * @return The last node of augmenting path or -1
     */
    private int findAugmentingPath(Queue<Integer> q, boolean[] t) {
        while(!q.isEmpty()) {
            int x = q.poll().intValue();
            for(int i = 0; i < adjlist[x].length; i++) // For all neighbours
            {
                int neighbour = adjlist[x][i];
                if(!deleted[neighbour] && !t[neighbour]) // T(y) = false
                {
                    if(matching[neighbour] != UNSATURATED) {
                        int z = matching[neighbour];
                        t[neighbour] = true; // T[y] <- true
                        t[z] = true; // T[z] <- true
                        alternatingTree[neighbour]=x; // P(y)<-x
                        alternatingTree[z]=neighbour; // P(z)<-y
                        q.offer(new Integer(z)); // Q <- Q U {z}
                    } else {
                        alternatingTree[neighbour]=x; // P(y) <-x
                        return neighbour; // return last vertex on augmenting path
                    }
                }
            }
        }
        return -1; // No augmenting found!
    }
    
    /**
     * This method checks all neighbours of the specified vertex
     * and looks for unsaturated neighbours.
     * @param v The vertex to examine.
     * @param t The current tree.
     * @return True if the given vertex has an unsaturated neighbour
     */
    private boolean lookAheadHeuristic(int vertex, boolean[] t) {
        for(int i = 0; i < adjlist[vertex].length; i++) {
            int neighbour = adjlist[vertex][i];
            if(!t[neighbour] && !deleted[neighbour] &&
               matching[neighbour] == UNSATURATED) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method deletes all vertices visited during a search for an
     * augmenting path if the search failed.
     */
    private void cleanUpHeuristic(boolean[] t) {
        for(int i = 0; i < t.length; i++) {
            if(t[i]) {
                deleted[i] = true;
            }
        }
    }
}
