/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.rte.comm;

import java.net.*;
import java.io.*;
import com.sap.dbtech.util.*;
/**
 * Low-level socket communication.
 *
 * as specified by the SAP DB runtime environment.
 *
 */
public class SocketComm extends JdbcCommunication
{
    // instance variables
    private static int counter = 0;
    private Socket socket;
    private InputStream instream;
    private OutputStream outstream;
    private int senderRef;
    private int receiverRef;
    private int maxSendLen;
    private int remoteSwapping;
    private String host;
    private int    port;
    private String dbname;
    private PacketLayout packetLayout;
    private boolean dbSession;
    public final static JdbcCommFactory factory = new JdbcCommFactory () {
        public JdbcCommunication open(String host, String dbname) throws RTEException  {
            return SocketComm.connectDB (host, dbname);
        }
    };
    // misc constants
    public static final int controlPacketSize_C = 16384;   // from vsp009c::csp9_ctrl_packet_size

    /**
     * creates a new socket connection to <i>host</i>.
     *
     * @exception RTEException
     */
    private
    SocketComm (
            String hostPort)
        throws RTEException
    {
        // this.host = host;
        int colon_idx=hostPort.indexOf(":");
        if(colon_idx == -1) {
            this.host=hostPort;
            this.port=RteC.defaultPort_C;
        } else {
            this.host=hostPort.substring(0, colon_idx);
            try {
                this.port=Integer.parseInt(hostPort.substring(colon_idx + 1));
            } catch(NumberFormatException nformatex) {
                throw new RTEException(MessageTranslator.translate(MessageKey.ERROR_UNKNOWN_HOST,
                                                                   hostPort,
                                                                   nformatex.getMessage(),
                                                                   new Integer(RteC.CommunicationErrorCodeMap_C[RteC.SQLSERVER_OR_DB_UNKNOWN_C])));
            }
        }
        this.openSocket();
    }
    /**
     * fills the RTE-specific header with standard values
     * @param messClass int a value from {@link RteC RteC.RSQL_*}
     */
    private void
    buildRTEHeader (
            StructuredBytes packet,
            int messClass)
    {
        packet.putInt4 (0, RteC.Header_ActSendLen_O);
        packet.putInt1 (3, RteC.Header_ProtocolID_O);
        packet.putInt1 (messClass, RteC.Header_MessClass_O);
        packet.putInt1 (0, RteC.Header_RTEFlags_O);
        packet.putInt1 (0, RteC.Header_ResidualPackets_O);
        packet.putInt4 (senderRef, RteC.Header_SenderRef_O);
        packet.putInt4 (receiverRef, RteC.Header_ReceiverRef_O);
        packet.putInt2 (0, RteC.Header_RTEReturnCode_O);
        packet.putInt2 (0, RteC.Header_Filler_O);
        packet.putInt4 (maxSendLen, RteC.Header_MaxSendLen_O);
        return;
    }
    /**
     * sends a cancel request for the current request (not implemented).
     */
    public void
    cancel ()
        throws java.sql.SQLException
    {
        DbPacketLayout  layout;
        StructuredBytes rawRequest;
        ConnectPacket   connectPacket;
        SocketComm cancelConnection;

        try {
            layout = new DbPacketLayout ();
            // create connect
            // set cancel request
            cancelConnection = new SocketComm (this.host+":"+this.port);
            rawRequest = createRawPacket (RteC.Connect_END_O_C);
            this.buildRTEHeader (rawRequest, RteC.RSQL_USER_CANCEL_REQUEST_C);
            rawRequest.putInt4 (this.senderRef, RteC.Header_ReceiverRef_O);
            connectPacket = new ConnectPacket (rawRequest.bytes (),
                dbname, layout);
            // send connect
            connectPacket.close ();
            cancelConnection.sendData (rawRequest, connectPacket.length ());
            // close, no reply expected
            cancelConnection.closeSocket();
        }
        catch (RTEException exc) {
            throw new java.sql.SQLException (exc.getMessage ());
        }
    }
    /**
     * closes the socket connection.
     */
    private void
    closeSocket ()
    {
        try {
            this.socket.close ();
        }
        catch (java.io.IOException ioExc) {
            // ignore
        }
        finally {
          this.socket = null;
        }
        return;
    }
    /**
     * opens a connection to a dbm- or repm-server
     * @return com.sap.sapdb.SocketComm
     * @param hostname java.lang.String the server name
     * @param dbname java.lang.String the database name
     * @param dbroot java.lang.String (optional) the location of database executables
     * @param dbname java.lang.String the name of the server program
     * @exception com.sap.sapdb.RTEException when no connection can be established
     */
    public static SocketComm
    connectAdmin (
            String hostname,
            String dbname,
            String dbroot,
            String program)
        throws RTEException
    {
        SocketComm result;
        PacketLayout layout = new CserverPacketLayout ();

        result = new SocketComm (hostname);
        result.sendAdminConnect (dbname, dbroot, program, layout);
        result.receiveConnect ();
        result.packetLayout = layout;
        return result;
    }
    /**
     * opens a connection to a SAP DB kernel
     * @return com.sap.sapdb.SocketComm
     * @param hostname java.lang.String the server name
     * @param dbname java.lang.String the database name
     * @exception com.sap.sapdb.RTEException.
     */
    public static SocketComm
    connectDB (
            String hostname,
            String dbname)
        throws RTEException
    {
        SocketComm result;
        PacketLayout layout;

        /*
         * info request
         */
        layout = SocketComm.doInfoRequest (hostname, dbname);
        //layout.dumpOn (System.out, "after Info request");
        /*
         * do actual connect
         */
        result = new SocketComm (hostname);
        result.dbSession = true;
        result.dbname = dbname;
        result.packetLayout = result.dbConnectExchange (dbname, layout);
        //result.packetLayout.dumpOn (System.out, "after connect exchange");
        return result;
    }
    /**
     * allocates a new communication packet
     * @return com.sap.dbtech.util.StructuredBytes
     * @param dataSize int the usable size
     */
    private static StructuredBytes
    createRawPacket (
            int dataSize)
    {
        return new StructuredBytes (dataSize + RteC.Header_END_O_C);
    }
    /**
     * sends and receives the connect packet.
     * @param dbname java.lang.String the database name
     * @param layout com.sap.dbtech.rte.comm.PacketLayout
     * @exception com.sap.dbtech.rte.comm.RTEException.
     */
    private PacketLayout
    dbConnectExchange (
            String dbname,
            PacketLayout layout)
        throws RTEException
    {
        StructuredBytes connectReply;

        this.sendDbConnect (dbname, layout);
        connectReply = this.receiveConnect ();
        layout = getPacketLayoutFromReply (connectReply, true);
        return layout;
    }
    /**
     * queries a database for the packet layout
     * @return com.sap.dbtech.rte.comm.DbPacketLayout a description of the packet layout
     * @param host java.lang.String the server name
     * @param dbname java.lang.String the database name
     * @exception com.sap.dbtech.rte.comm.RTEException when no connection
     *      could be established
     */
    private static DbPacketLayout doInfoRequest (
            String host,
            String dbname)
        throws RTEException
    {
        DbPacketLayout  layout;
        DbPacketLayout  result;
        SocketComm      connection;
        ConnectPacket   connectPacket;
        StructuredBytes rawRequest;
        StructuredBytes reply;
        int             rc;

        /*
         * send info request
         */
        layout = new DbPacketLayout ();
        connection = new SocketComm (host);
        rawRequest = createRawPacket (RteC.Connect_END_O_C);
        connection.buildRTEHeader (rawRequest, RteC.RSQL_INFO_REQUEST_C);
        connectPacket = new ConnectPacket (rawRequest.bytes (),
                dbname, layout);
        connectPacket.close ();
        connection.sendData (rawRequest, connectPacket.length ());
        /*
         * read info reply
         */
        reply = connection.receiveConnect ();
        rc = reply.getInt2 (RteC.Header_RTEReturnCode_O);
        if (rc != 0) {
            connection.release ();
            throw new CommunicationException (rc);
        }
        connection.release ();
        result = getPacketLayoutFromReply (reply, false);
        return result;
    }
    /**
     *
     */
    static private DbPacketLayout
    getPacketLayoutFromReply (
        StructuredBytes reply,
        boolean trustReply)
    {
        DbPacketLayout  result;
        int             packetSize;
        int             maxDataLen;
        int             minReplySize;
        int             maxSegmentSize;

        maxDataLen = reply.getInt4 (RteC.Header_END_O_C + RteC.Connect_MaxDataLen_O);
        minReplySize = reply.getInt4 (RteC.Header_END_O_C + RteC.Connect_MinReplySize_O);
        packetSize = reply.getInt4 (RteC.Header_END_O_C + RteC.Connect_PacketSize_O);
        if (trustReply) {
            maxSegmentSize = reply.getInt4 (RteC.Header_END_O_C + RteC.Connect_MaxSegmentSize_O);
        }
        else {
            maxSegmentSize = packetSize;
        }
        result = new DbPacketLayout (maxDataLen, maxSegmentSize,
                minReplySize, packetSize);
        return result;
    }
    /**
     * traces the content of the RTE header on stream for debugging purposes.
     * @param header com.sap.dbtech.util.StructuredMem
     * @param stream java.io.PrintStream
     */
    private static void dumpRteHeader(
            StructuredMem header,
            PrintStream stream)
    {
        stream.println("RTE Header");

        stream.print("ActSendLen: ");
        stream.println(header.getInt4(RteC.Header_ActSendLen_O));
        stream.print("ProtocolID: ");
        stream.println(header.getInt1(RteC.Header_ProtocolID_O));
        stream.print("MessClass: ");
        stream.println(header.getInt1(RteC.Header_MessClass_O));
        stream.print("RTEFlags: ");
        stream.println(header.getInt1(RteC.Header_RTEFlags_O));
        stream.print("ResidualPackets: ");
        stream.println(header.getInt1(RteC.Header_ResidualPackets_O));
        stream.print("SenderRef: ");
        stream.println(header.getInt4(RteC.Header_SenderRef_O));
        stream.print("ReceiverRef: ");
        stream.println(header.getInt4(RteC.Header_ReceiverRef_O));
        stream.print("RTEReturnCode: ");
        stream.println(header.getInt2(RteC.Header_RTEReturnCode_O));
        stream.print("Filler: ");
        stream.println(header.getInt2(RteC.Header_Filler_O));
        stream.print("MaxSendLen: ");
        stream.println(header.getInt4(RteC.Header_MaxSendLen_O));
        //Tracer.traceObject (null, header);
    }
    /**
     * Code to perform when this object is garbage collected.
     *
     * make sure that the socket is closed to prevent
     * accumulation of handles.
     */
    protected void
    finalize()
        throws Throwable
    {
        if (this.socket != null) {
            this.release ();
        }
        super.finalize();
    }
    /**
     * creates a new request packet.
     *
     * Clients are expected to reuse these packets and to request
     * additional packets only when packets are used in parallel
     * @return StructuredMem
     */
    public StructuredMem
    getRequestPacket ()
    {
        StructuredBytes result;

        result = new RteSocketPacket (this.packetLayout.maxCmdDataLength());
        return result;
    }
    /**
     * tests whether the SocketCommunication is still valid.
     * @return boolean
     */
    public boolean
    isConnected () {
        return this.socket != null;
    }
    /**
     * returns the port to connect to the vserver.
     *
     * There seems to be no method getprotbyname in java.net.
     * @return int
     */
    private int
    lookupPort ()
    {
        return this.port;
    }
    /**
     * opens a socket connection.
     *
     * This converts any java.net specific exceptions to RTEException.
     * @param host java.lang.String
     * @exception RTEException
     */
    private void openSocket ()
        throws RTEException
    {
        try {
            this.socket = new Socket (this.host, this.lookupPort ());
            this.instream = this.socket.getInputStream();
            this.outstream = this.socket.getOutputStream();
        }
        catch (UnknownHostException uhexc){
            throw new RTEException
                (MessageTranslator.translate(MessageKey.ERROR_UNKNOWN_HOST,
                                             this.host,
                                             uhexc.getMessage(),
                                             new Integer(RteC.CommunicationErrorCodeMap_C[RteC.SQLSERVER_OR_DB_UNKNOWN_C])),
                 RteC.CommunicationErrorCodeMap_C[RteC.SQLSERVER_OR_DB_UNKNOWN_C]);
        }    catch (IOException ioexc){
            throw new RTEException
                (MessageTranslator.translate(MessageKey.ERROR_HOST_CONNECT,
                                             this.host,
                                             ioexc.getMessage(),
                                             new Integer(RteC.CommunicationErrorCodeMap_C[RteC.SQLSTART_REQUIRED_C])),
                 RteC.CommunicationErrorCodeMap_C[RteC.SQLSTART_REQUIRED_C]);

        }
        try {
            this.socket.setSoLinger (true, 15);
        }
        catch (SocketException soExc) {
            // ignore
        }

    }
    /**
     * receives a reply to a pending request.
     * @return StructuredMem
     * @exception com.sap.sapdb.RTEException.
     */
    public StructuredMem
    receive ()
        throws RTEException
    {
        return this.receiveData ();
    }
    /**
     * creceive the connect reply
     */
    private StructuredBytes
    receiveConnect()
        throws RTEException
    {
        int expectedReplyLen = RteC.Header_END_O_C + RteC.Connect_END_O_C;
        byte [] connectReply = new byte [expectedReplyLen];
        int replyLen;
        InputStream instream;
        StructuredBytes reply;
        int transmittedReplyLen;

        try {
            /*
             * read header
             */
            replyLen = this.instream.read (connectReply, 0, expectedReplyLen);
            if (replyLen < RteC.Header_END_O_C) {
                throw new RTEException
                    (MessageTranslator.translate(MessageKey.ERROR_RECV_CONNECT),
                     RteC.CommunicationErrorCodeMap_C[RteC.SQLRECEIVE_LINE_DOWN_C]);
            }
            /*
             * get swapping
             */
            this.remoteSwapping = connectReply [RteC.Header_END_O_C + 1];
            reply = this.replyMem (connectReply);
            transmittedReplyLen = reply.getInt4 (RteC.Header_ActSendLen_O);
            if (transmittedReplyLen != replyLen) {
                throw new RTEException (MessageTranslator.translate(MessageKey.ERROR_REPLY_GARBLED),
                                        RteC.CommunicationErrorCodeMap_C[RteC.SQLRECEIVE_LINE_DOWN_C]);
            }
        }
        catch (java.io.IOException ioexc) {
            throw new RTEException (MessageTranslator.translate(MessageKey.ERROR_CONNECT_RECVFAILED,
                                                                ioexc.getMessage ()),
                                    RteC.CommunicationErrorCodeMap_C[RteC.SQLNOTOK_C]);
        }
        /*
         * use connect information
         */
        this.senderRef=reply.getInt4 (RteC.Header_SenderRef_O);
        int replyRC = reply.getInt2 (RteC.Header_RTEReturnCode_O);
        if (replyRC != 0) {
            throw new RTEException (RteC.getCommunicationError (replyRC),RteC.getCommunicationErrorMapCode (replyRC));
        }
        return reply;
    }
    /**
     * blocks to wait for a reply
     * @return StructuredMem
     * @exception com.sap.dbtech.rte.comm.RTEException
     */
    private StructuredMem
    receiveData()
        throws RTEException
    {
        boolean firstPacket = true;
        java.io.InputStream instream;
        byte[] header = new byte[RteC.Header_END_O_C];
        int headerLength;
        int dataLength = 0;
        int bytesRead = 0;
        int actSendLength;
        int chunkRead;
        byte[] readBuf = null;
        StructuredBytes replyHeader = null;
        StructuredBytes result = null;
        int rteRC;

        //Tracer.println ("receiveData: " + this); //#print
        //Tracer.whereAmI ();
        try {
            /*
             * get input stream
             */
            instream = this.instream;
            while (firstPacket || (bytesRead < dataLength)) {
                //Tracer.println ("firstPacket: " + firstPacket); //#print
                //Tracer.println ("dataLength: " + dataLength); //#print
                //Tracer.println ("bytesRead: " + bytesRead); //#print
                /*
                 * read header
                 */
                //System.out.print ("reading: "); //#print
                headerLength = instream.read(header, 0, RteC.Header_END_O_C);
                if (headerLength != RteC.Header_END_O_C) {
                    RTEException rteExc = new RTEException
                        (MessageTranslator.translate(MessageKey.ERROR_DATA_RECVFAILED),
                         RteC.CommunicationErrorCodeMap_C[RteC.SQLRECEIVE_LINE_DOWN_C]);
                    throw rteExc;
                }
                if (firstPacket) {
                    replyHeader = this.replyMem(header);
                    firstPacket = false;
                    rteRC = replyHeader.getInt2 (RteC.Header_RTEReturnCode_O);
                    if (rteRC != 0) {
                        throw new CommunicationException (rteRC);
                    }
                    dataLength = replyHeader.getInt4(RteC.Header_MaxSendLen_O) - headerLength;
                    //Tracer.print ("packetLen: " + dataLength); //#print
                    //Tracer.println ("dataLength: " + dataLength); //#print
                    /*
                     * alloc result
                     */
                    readBuf = new byte[dataLength];
                    result = new StructuredBytes(readBuf);
                }
                /*
                 * get length of residual packet
                 */
                actSendLength = replyHeader.getInt4(RteC.Header_ActSendLen_O) - headerLength;
                //System.out.print (" actSendLength: " + actSendLength + " | "); //#print
                if ((actSendLength < 0)
                        || (bytesRead + actSendLength > readBuf.length)) {
                    //Tracer.println ("actSendLength out of bounds: " + actSendLength);
                }
                /*
                 * read bytes
                 */
                while (actSendLength > 0) {
                    try {
                        chunkRead = instream.read(readBuf, bytesRead, actSendLength);
                        //System.out.print (chunkRead + ", ");//#print
                    }
                    catch (ArrayIndexOutOfBoundsException arrayExc) {
                        //Tracer.println ("replyHeader");
                        //replyHeader.traceOn (Tracer.getLog());
                        //Tracer.println ("received packet");
                        //result.traceOn (Tracer.getLog(), bytesRead - 1000, bytesRead + 1000);
                        throw new ArrayIndexOutOfBoundsException
                            (MessageTranslator.translate(MessageKey.ERROR_CHUNKOVERFLOW,
                                                         Integer.toString(actSendLength),
                                                         Integer.toString(bytesRead),
                                                         Integer.toString(readBuf.length)));
                    }
                    if (chunkRead < 0) {
                        throw new RTEException(MessageTranslator.translate(MessageKey.ERROR_DATA_RECVFAILED),
                                               RteC.CommunicationErrorCodeMap_C[RteC.SQLRECEIVE_LINE_DOWN_C]);
                    }
                    bytesRead += chunkRead;
                    actSendLength -= chunkRead;
                }
            }
            //System.out.println (" -- Done!");
        }
        catch (java.io.IOException ioexc) {
            throw new RTEException(MessageTranslator.translate(MessageKey.ERROR_DATA_RECVFAILED_REASON,
                                                               ioexc.getMessage()),
                                   RteC.CommunicationErrorCodeMap_C[RteC.SQLRECEIVE_LINE_DOWN_C]);
        }
        if (Tracer.traceAny_C && Tracer.isOn (9)) {
            Tracer.traceObject ("REPLY:", result);
        }
        return result;
    }
    /**
     * tries to reconnect after a timeout
     * @exception com.sap.dbtech.rte.comm.RTEException.
     */
    public void
    reconnect ()
        throws RTEException
    {
        try {
          this.socket.close ();
          this.socket = null;
        }
        catch (Exception ignore) {
          this.socket = null;
        }
        this.openSocket ();
        if (this.dbSession) {
            this.dbConnectExchange (this.dbname, this.packetLayout);
        }
        else {
            throw new RTEException
                ( MessageTranslator.translate(MessageKey.ERROR_ADMIN_RECONNECT),
                  RteC.CommunicationErrorCodeMap_C[RteC.SQLTIMEOUT_C]);
        }
    }
    /**
     * closes the connection
     */
    public void
    release ()
    {
        try {
            StructuredBytes rawPacket = createRawPacket(0);
            this.buildRTEHeader (rawPacket, RteC.RSQL_USER_RELEASE_REQUEST_C);
            this.sendData (rawPacket, 0);
            this.socket.close ();
        }
        catch (java.io.IOException ioexc) {
            // ignore
        }
        catch (RTEException rteexc) {
            // ignore
        }
        finally {
            this.socket = null;
        }
    }
    /**
     * converts the raw reply into a readable StructuredMem.
     *
     * This is done by wrapping the bytes into an
     * appropriate reader, not by converting the actual data.
     * @return com.sap.dbtech.util.StructuredMem
     * @param buf byte[]
     */
    private StructuredBytes
    replyMem (
            byte [] buf)
        throws RTEException
    {
        StructuredBytes result;
        switch (this.remoteSwapping) {
                case RteC.notSwapped_C:
                    result = new StructuredBytes (buf);
                    break;
                case RteC.fullSwapped_C:
                    result = new FullswapMem (buf);
                    break;
                default:
                    throw new RTEException (MessageTranslator.translate(MessageKey.ERROR_INVALID_SWAPPING),-708);
        }
        return result;
    }
    /**
     * sends client data to the server.
     * @param userPacket StructuredMem
     * @param len int
     * @exception com.sap.sapdb.RTEException
     */
    public void request(StructuredMem userPacket, int len) throws RTEException {
        if (Tracer.traceAny_C && Tracer.isOn(9)) {
            Tracer.traceObject("REQUEST:", userPacket, len + 24);
        }
        StructuredBytes rawPacket = ((RteSocketPacket) userPacket).rteHeader;
        this.buildRTEHeader(rawPacket, RteC.RSQL_USER_DATA_REQUEST_C);
        this.sendData(rawPacket, len);
    }
    /**
     * sends connect packet for dbm- or repm-Server.
     * @param db java.lang.String the database name
     * @param dbroot java.lang.String (optional) location of database executables
     * @param pgm java.lang.String name of server program
     * @param PacketLayout layout
     * @exception com.sap.sapdb.jdbc.RTEException.
     */
    private void
    sendAdminConnect (
            String dbname,
            String dbroot,
            String pgm,
            PacketLayout layout)
        throws RTEException
    {
        StructuredBytes rawRequest;
        ConnectPacket   connectPacket;

        rawRequest = createRawPacket(layout.maxDataLength());
        this.buildRTEHeader (rawRequest, RteC.RSQL_CTRL_CONN_REQUEST_C);
        connectPacket = new ConnectPacket (rawRequest.bytes (),
            dbname, layout);
        connectPacket.addConnectString (dbroot, RteC.ARGID_DBROOT_C);
        connectPacket.addConnectString (pgm, RteC.ARGID_SERVERPGM_C);
        connectPacket.close ();
        this.sendData (rawRequest, connectPacket.length ());
        return;
    }
    /**
     * Writes logically <i>len</i> bytes of <i>data</i>.
     *
     * The RTE-header is filled and the whiole packet is sent.
     * @param packet StructuredBytes
     * @param len int
     * @exception com.sap.sapdb.jdbc.RTEException.
     */
    private void
    sendData (
            StructuredBytes packet,
            int len)
        throws RTEException
    {
        OutputStream ostream;

        //Tracer.println ("send: (" + len + ") " + this); //#print
        //Tracer.whereAmI ();
        this.setActSendLen (packet, len);
        try {
            if(outstream==null)
                outstream=socket.getOutputStream ();
            ostream = outstream;
        }
        catch (IOException ioExc1) {
            throw new RTEException ("sendData: getOutputStream failed",
            RteC.CommunicationErrorCodeMap_C[RteC.SQLSEND_LINE_DOWN_C]);
        }
        try {
            ostream.write (packet.bytes (),
                0, len + RteC.Header_END_O_C);
        }
        catch (IOException ioExc2) {
            throw new RTEException (MessageTranslator.translate(MessageKey.ERROR_SEND_WRITE),
                                    RteC.CommunicationErrorCodeMap_C[RteC.SQLSEND_LINE_DOWN_C]);
        }
        return;
    }

    /**
     * sends connect packet for SAP DB kernel.
     * @param dbname java.lang.String the database name
     * @param layout PacketLayout a suggestion for the packet layout
     * @exception com.sap.sapdb.jdbc.RTEException.
     */
    private void
    sendDbConnect (
            String dbname,
            PacketLayout layout)
        throws RTEException
    {
        StructuredBytes rawRequest;
        ConnectPacket   connectPacket;

        rawRequest = createRawPacket (layout.maxDataLength());
        this.buildRTEHeader (rawRequest, RteC.RSQL_USER_CONN_REQUEST_C);
        connectPacket = new ConnectPacket (rawRequest.bytes (),
            dbname, layout);
        connectPacket.close ();
        this.sendData (rawRequest, connectPacket.length ());
    }
    /**
     * sets the len of client data in the RTE-header
     * @param sendLen int
     */
    private void
    setActSendLen (
            StructuredBytes rawPacket,
            int sendLen)
    {
        sendLen += RteC.Header_END_O_C;
        rawPacket.putInt4 (sendLen, RteC.Header_ActSendLen_O);
        rawPacket.putInt4 (sendLen, RteC.Header_MaxSendLen_O);
        return;
    }

    /**
     * sleep on this connection.
     *
     * used only for debugging.
     */
    public static void
    sleep ()
    {
        try {
            Thread.sleep (1000);
        }
        catch (InterruptedException exc) {
            // ignore
        }
    }
    /**
     * mimics the C-API sqlaconnect.
     * @param host java.lang.String
     * @param db java.lang.String
     * @exception com.sap.sapdb.RTEException
     */
    static public SocketComm
    sqlaconnect (
            String host,
            String db)
        throws RTEException
    {
        return null;
    }
    /**
     * mimics the C-API sqlareceive
     * @return StructuredMem
     * @exception com.sap.sapdb.RTEException.
     */
    public StructuredMem
    sqlareceive ()
        throws RTEException
    {
        return this.receive ();
    }
    /**
     * mimics the C-API sqlarelease
     * @exception com.sap.sapdb.RTEException
     */
    public void
    sqlarelease ()
        throws RTEException
    {
        this.release ();
        return;
    }
    /**
     * mimics the C-API sqlarequest
     * @param len int
     * @exception com.sap.sapdb.RTEException
     */
    public void
    sqlarequest (
            StructuredMem userPacket,
            int len)
        throws RTEException
    {
        this.request (userPacket, len);
    }
    /**
     * mimics the C-API sqlxconnect
     * @param host java.lang.String
     * @param db java.lang.String
     * @param dbroot java.lang.String
     * @param pgm java.lang.String
     * @exception com.sap.sapdb.RTEException.
     */
    static public SocketComm
    sqlxconnect (
            String host,
            String db,
            String dbroot,
            String pgm)
        throws RTEException
    {
        SocketComm result;

        result = connectAdmin (host, db, dbroot, pgm);
        return result;
    }


    /**
     *
     */
    public String
    toString ()
    {
        String result;

        result = super.toString () + " on " + this.socket;
        return result;
    }
}
