/**************************************************************
*   
*   Creation Date: <97/06/21 17:01:31 samuel>
*   Time-stamp: <2001/03/19 00:04:48 samuel>
*   
*	<memory.c>
*	
*	Memory functions
*   
*   Copyright (C) 1997, 1999, 2000, 2001 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
*   There is a bug in the kernel which prevent us from mapping
*   the last ROM page 0xfffff000 (probably due to an owerflow). 
**************************************************************/

#include "mol_config.h"
#include <sys/mman.h>

/* #define VERBOSE */

#include "debugger.h"
#include "memory.h"
#include "wrapper.h"
#include "promif.h"
#include "mmu_mappings.h"
#include "res_manager.h"
#include "verbose.h"
#include "booter.h"
#include "session.h"

SET_VERBOSE_NAME("memory");

//#define DEBUG_LOCK_MEM 
//#define DEBUG_NO_ZERO_PAGES

#define WANTED_RAM_BASE		((char*)0x40000000)
#define	DEFAULT_RAM_SIZE	64

struct mmu_mapping ram={0,0,0,0,0};
struct mmu_mapping rom={0,0,0,0,0};	/* Initialized by owbooter/oldworld.c */

static int	fmapped_ram = 0;	/* file-mapped RAM */

static int	save_ram( void  );
static void	load_ram( void  );
static void	map_ram( void );


/**************************************************************
*  mem_init
*
**************************************************************/

void 
mem_init( void ) 
{
	session_save_proc( save_ram, NULL, kStaticChunk );

	if( loading_session() )
		load_ram();
	else
		map_ram();
#if 1
	_track_dirty_RAM( ram.lvbase, ram.size );
#endif
	printm("%dMB RAM mapped at %p\n", ram.size/(1024*1024), ram.lvbase );

	_add_mmu_mapping( &ram );
	_set_mac_ram( (ulong)ram.lvbase, (ulong)ram.size, ram.mbase /* mac physical */ );

#ifdef DEBUG_NO_ZERO_PAGES
{ 	/* Used to track problems with the zero pages */ 
	int i = 0;
	for(i=0; i<ram.size; i++ )
		ram.lvbase[i] = 0;
}
#endif
}


/**************************************************************
*  mem_cleanup
*    
**************************************************************/

void 
mem_cleanup( void ) 
{
	if( ram.lvbase ) {
		_remove_mmu_mapping( &ram );
		munmap( ram.lvbase, ram.size );
		ram.lvbase = 0;
	}
}


/**************************************************************
*  map_ram
*    Allocate RAM
**************************************************************/

static int
save_ram( void )
{
	int 	size,i, j, fd, wsize;
	char 	*lvbase;
	off_t 	offs;
	char 	*p=NULL;

	if( fmapped_ram ){
		if( msync( ram.lvbase, ram.size, MS_SYNC ) < 0 ){
			LOG_ERR("msync");
			return 1;
		}
		return 0;
	}

	/* Save dirty page information... */
	if( !(size = _get_dirty_RAM( NULL )) )
		return 1;

	p = alloca(size);
	_get_dirty_RAM( p );

	if( write_session_data( "RAMp", 0, p, size ) )
		return 1;

	/* Prepare to save RAM */
	align_session_data( 0x1000 );

	if( write_session_data( "RAM", 0, NULL, ram.size ) )
		return 1;
	if( get_session_data_fd( "RAM", 0, &fd, &offs, NULL ) )
		return 1;

	lseek( fd, offs, SEEK_SET );
	ftruncate( fd, offs );

	/* ...And the actual data */
	lvbase = ram.lvbase;
	for(wsize=0, i=0; i<size; i++ ){
		for(j=0; j<8 && lvbase + wsize < ram.lvbase + ram.size; j++ ) {
			if( p[i] & (1<<j) ) {
				wsize += 0x1000;
			} else {
				if( wsize && write( fd, lvbase, wsize ) != wsize )
					return 1;
				lvbase += wsize + 0x1000;

				wsize = 0;
				lseek( fd, 0x1000, SEEK_CUR );
			}
		}
	}
	if( wsize && write( fd, lvbase, wsize ) != wsize )
		return 1;

	return 0;
}


static void
load_ram( void )
{
	int 	fd;
	ssize_t size;
	off_t 	offs;

	if( get_session_data_fd( "RAM", 0, &fd, &offs, &size ) ) {
		printm("Failed restoring RAM\n");
		exit(1);
	}
	ram.flags = MAPPING_RW;
	ram.size = size;
	ram.lvbase = mmap( WANTED_RAM_BASE, ram.size, PROT_EXEC | PROT_READ | PROT_WRITE, MAP_SHARED, fd, offs );
	if( (int)ram.lvbase == -1 ){
		LOG_ERR("Could not map RAM");
		exit(1);
	}
	fmapped_ram = 1;
}


static void 
map_ram( void ) 
{
	int rsize;

	if( (rsize = get_numeric_res("ram_size")) == -1 )
		rsize = DEFAULT_RAM_SIZE;
	if( rsize >= 0x1000 ){
		printm("The RAM size should be given in MB!\n");
		rsize = DEFAULT_RAM_SIZE;
	}

	ram.flags = MAPPING_RW;
	ram.size = rsize * 1024 * 1024;
	ram.mbase = 0;

	if( !(ram.lvbase = map_zero( WANTED_RAM_BASE, ram.size )) ) {
		printm("Failed to map RAM. Probably an of memory condition.\n");
		exit(1);
	}

#ifdef DEBUG_LOCK_MEM
	printm("DEBUG: Locking memory...\n");
	if( mlock( ram.lvbase, ram.size ) ) {
		perrorm("mlock");
	}
#endif
}


/**************************************************************
*  map_phys_mem
*    Map physical memory (through /dev/mem)
*    phys_ptr doesn't need to to be page-aligned
*
*    Cache-bit settings?
**************************************************************/

char *
map_phys_mem( char *wanted_ptr, ulong phys_ptr, size_t size, int prot  ) 
{
	ulong	page_offs = (phys_ptr & 0xfff);
	ulong	page_start = phys_ptr - page_offs;
	int	fd;
	char	*ret;

	fd = open( "/dev/mem", O_RDWR );
	if( fd == -1 ) {
		perrorm("/dev/mem, open:");
		return 0;
	}

	size = size + page_offs;
	if( size & 0xfff )
		size = (size & 0xfffff000) + 0x1000;

	ret = (char*) mmap( wanted_ptr, size, prot, MAP_SHARED,
			    fd, (off_t)page_start );

	if( (int)ret ==-1 ) {
		perrorm("map_phys_mem: mmap");
		close( fd );
		return NULL;
	}
	close( fd );

	ret += page_offs;
	return ret;
}

/* unmap_mem is a wrapper of munmap. It does the same page 
 * alignment as map_phys_mem above.
 */
int
unmap_mem( char *start, size_t length )
{
	ulong	page_offs;
	ulong	page_start;

	/* mmap wants page-aligned parameters */
	page_offs = (ulong)start & 0xfff;
	page_start = (ulong)start - page_offs;
	length += page_offs;

	if( length & 0xfff )
		length = (length & 0xfffff000UL) + 0x1000;
	return munmap( start, length );
}



/**************************************************************
*  map_zero
*	Allocate memory
**************************************************************/

char *
map_zero( char *wanted_addr, size_t size )
{
	int 	fd;
	char 	*ptr;

	fd = open( "/dev/zero", O_RDWR );
	if( fd == -1 ) {
		perrorm("/dev/zero, open");
		return 0;
	}

	ptr = mmap( wanted_addr, size, PROT_EXEC | PROT_READ | PROT_WRITE, 
		    MAP_PRIVATE, fd, 0 );

	if( (int)ptr==-1 ) {
		perror("map_zero: mmap");
		close( fd );
		return 0;
	}
	close( fd );
	return ptr;
}


/**************************************************************
*  mac_phys_to_lvptr / lvptr_to_mac_phys
*  verify_lvrange / verify_mrange
*    Translate mac physical ptr to linux virtual
*   	0:	RAM
*	1:	ROM
*	< 0	error, not in RAM/ROM
**************************************************************/

int 
mphys_to_lvptr( ulong mptr, char **ret )
{
	/* check that the address is valid (in RAM/ROM) */
	if( mptr >= ram.mbase && mptr <= ram.mbase + ram.size-1 ) {
		*ret = ram.lvbase + mptr - ram.mbase;
		return 0;
	}
	if( rom.size && mptr >= rom.mbase && mptr <= rom.mbase + rom.size-1 ) {
		*ret = rom.lvbase + mptr - rom.mbase;
		return 1;
	}
	return -1;
}

int
lvptr_to_mphys( char *lvptr, ulong *ret )
{
	/* check that the address is valid (in RAM/ROM) */
	if( lvptr >= ram.lvbase && lvptr <= ram.lvbase + ram.size-1 ) {
		*ret = (ulong)(ram.mbase + lvptr - ram.lvbase);
		return 0;
	}
	if( rom.size && lvptr >= rom.lvbase && lvptr <= rom.lvbase + rom.size-1 ) {
		*ret = (ulong)(rom.mbase + lvptr - rom.lvbase);
		return 1;
	}
	return -1;	
}

int 
verify_lvrange( char *base, size_t len )
{
	/* check that the address is valid (in RAM/ROM) */
	if( base >= ram.lvbase && base + len  <= ram.lvbase + ram.size-1 ) {
		return 0;
	}
	if( rom.size && base >= rom.lvbase && base + len <= rom.lvbase + rom.size-1 ) {
		return 1;
	}
	return -1;
}

int 
verify_mrange( ulong mbase, size_t len )
{
	/* check that the address is valid (in RAM/ROM) */
	if( mbase >= ram.mbase && mbase + len <= ram.mbase + ram.size-1 ) {
		return 0;
	}
	if( rom.size && mbase >= rom.mbase && mbase + len <= rom.mbase + rom.size-1 ) {
		return 1;
	}
	return -1;
}


/* --- MMU stuff below, for use of the DEBUGGER only! ---- */ 
/* --- DO NOT use these functions in the emulation --- */


/**************************************************************
*  ea_to_mphys
*    ret: 0 translation found, 1 no translation
**************************************************************/

int 
ea_to_mphys( ulong ea, int context, ulong *mphys, int data_access )
{
	mol_PTE_t pte;
	int ret;

	if( data_access )
		ret = _translate_dea( context, ea, &pte );
	else
		ret = _translate_iea( context, ea, &pte );
	if( !ret && mphys )
		*mphys = (pte.rpn << 12) | (ea & 0xfff);

	return ret;
}

int 
ea_to_lvptr( ulong ea, int context, char **lvptr, int data_access )
{
	ulong mphys=-1;
	if( ea_to_mphys( ea, context, &mphys, data_access ))
		return 1;

	if( mphys_to_lvptr( mphys, lvptr ) >= 0 )
		return 0;
	return 1;
}


/**************************************************************
*  lvptr_is_rom
*    TRUE if linux virtual pointer is in ROM
**************************************************************/

int 
mphys_is_rom( char *lvptr ) 
{
	ulong	v = (ulong)lvptr;
	
	/* must be overflow safe */
	if( rom.size && v >= rom.mbase && v <= rom.mbase + rom.size -1 )
		return 1;
	return 0;
}
