/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */ 

import java.net.*;
import java.io.*;
import java.util.*;

import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.methods.*;
import org.apache.commons.httpclient.methods.multipart.*;
import org.apache.commons.httpclient.params.*;

/**
 *	The class XHTMLForm represents a form that may be submitted to a web application
 *	via an http put operation.  A client may use this class to fetch a form from a web 
 *	server and then query to find out what the form fields are, or a form may
 *	be programatically created.
 *
 *	The mechanics of the put are handled by the HttpClient class from
 *	Apache Jakarta Commons.
 */
public class XHTMLForm
{
	String				formTitle = "";
	String				formMessage = "";
	String				formAction = "";
	boolean				multipartEncoded = false;
	Map /* name, IFormItem */	formItems;
	Map /* label, FormItemSubmit */	submitButtons;

	public static String MULTIPART = "multipart/form-data";
	public static String URLENCODED = "application/x-www-form-urlencoded";
	
	public interface IFormItem
	{
		//
		// ItemType is an immutable enumeration class representing
		// the different kinds of form items supported.
		//
		static class ItemType
		{
			private String itemType;
			
			public static final ItemType TEXT = new ItemType("text");
			public static final ItemType PASSWORD = new ItemType("password");
			public static final ItemType READONLY = new ItemType("readonly");
			public static final ItemType HIDDEN = new ItemType("hidden");
			public static final ItemType CHECKBOX = new ItemType("checkbox");
			public static final ItemType RADIO = new ItemType("radio");
			public static final ItemType DROPDOWN = new ItemType("select");
			public static final ItemType SUBMIT = new ItemType("submit");
			public static final ItemType FILE = new ItemType("file");
			
			private static final ItemType[] PRIVATE_VALUES = { TEXT, PASSWORD, READONLY, HIDDEN, CHECKBOX, RADIO, DROPDOWN, SUBMIT, FILE };
			public static final List VALUES = Collections.unmodifiableList(Arrays.asList(PRIVATE_VALUES));
			
			private ItemType(String theType)
			{
				itemType = theType;
			}
			
			public String toString()
			{
				return itemType;
			}
			
			public static ItemType valueOf(String typeString)
			{
				ItemType result = null;
				
				for(int i=0; (i<PRIVATE_VALUES.length) && (result == null); ++i)
				{
					if(PRIVATE_VALUES[i].toString().equalsIgnoreCase(typeString))
						result = PRIVATE_VALUES[i];
				}
				
				return result;
			}
		}
		
		/**
		 *	Return the type of this form item
		 */
		public ItemType getType();

		/**
		 *	Return the descriptive label of this form item
		 */
		public String getLabel();
		
		/**
		 *	Set the descriptive label of this form item.
		 */
		public void setLabel(String theLabel);
		
		/**
		 *	Return the current value of this form item
		 */
		public String getValue();
		
		/**
		 *	Set the current value of this form item if this
		 *	value came from a 'selected' item.  Note that for
		 *	most item types, the concept of 'selected' is
		 *	meaningless and therefore ignored.
		 */
		public void setValue(String theValue, boolean isSelected);
		
		/**
		 *	Given the label of a radio button or dropdown,
		 *	set the value of this control accordingly.  Calls
		 *	to this method are ignored for items that are not
		 *	multiple-choice items.
		 */
		public void selectValue(String itemLabel);
		
		/**
		 *	Add another possible value for this form item.
		 *	Unconstrained form items (e.g. type TEXT or type
		 *	HIDDEN) will ignore calls to this method.
		 */
		public void setPossibleValue(String theValue, String theDescription);
		
		/**
		 *	Returns an array containing all of the legal values of
		 *	a radio button or drop-down list.  If a zero-size array
		 *	is returned, that indicates that the value of this
		 *	item is unconstrained.
		 */
		public String[] getValueList();
		
		/**
		 *	Returns an array containing the descriptions, which is
		 *	to say the human-readable comments that are associated
		 *	with each value.
		 */
		public String[] getDescriptionList();
		
		/**
		 *	Return the multipart post data for this form item
		 */		
		public Part getMultipartPostData(String itemName) throws IOException;
	
	}
	
	//
	// This is a static inner class used to cache information about
	// a form.  Clients usually interact with form items via methods of XHTMLForm.
	//
	static class FormItem implements IFormItem
	{
		
		ItemType	formItemType;
		String		value;
		String		label;
		
		protected FormItem()
		{
			formItemType = ItemType.TEXT;
			value = "";
		}
		
		protected FormItem(String theValue)
		{
			formItemType = ItemType.TEXT;
			value = theValue;
		}
		
		protected FormItem(ItemType theType, String theValue)
		{
			formItemType = theType;
			value = theValue == null ? "" : theValue;
		}
		
		public static IFormItem create(String theValue)
		{
			return FormItem.create(ItemType.TEXT, theValue);
		}
		
		public static IFormItem create(ItemType theType, String theValue)
		{
			IFormItem result = null;
			
			if(theType == ItemType.FILE)
			{
				result = new FormFileUploadItem(); 
			}
			else if((theType == ItemType.DROPDOWN) || (theType == ItemType.RADIO))
			{
				result = new FormItemMultipleChoice(theType, theValue);
			}
			else if(theType == ItemType.CHECKBOX)
			{
				result = new FormItemCheckbox(theType, theValue);
			}
			else
			{
				result = new FormItem(theType, theValue);
			}
			
			return result;
		}
		
		/**
		 *	Return the type of this form item
		 */
		public ItemType getType()
		{
			return formItemType;
		}
		
		/**
		 *	Return the descriptive label of this form item
		 */
		public String getLabel()
		{
			return label;
		}
		
		/**
		 *	Set the descriptive label of this form item.
		 */
		public void setLabel(String theLabel)
		{
			label = theLabel;
		}
		
		/**
		 *	Return the current value of this form item
		 */
		public String getValue()
		{
			return value;
		}
		
		/**
		 *	Set the current value of this form item if this
		 *	value came from a 'selected' item.  Note that for
		 *	most item types, the concept of 'selected' is
		 *	meaningless and therefore ignored.
		 */
		public void setValue(String theValue, boolean isSelected)
		{
			// TO DO:  Should we THROW if this item type is HIDDEN / READONLY?
			
			value = theValue == null ? "" : theValue;
		}

		public void selectValue(String theDescription)
		{
		}
				
		/**
		 *	Add another possible value for this form item.
		 *	Unconstrained form items (e.g. type TEXT or type
		 *	HIDDEN) ignore this method.
		 */
		public void setPossibleValue(String theValue, String theDescription)
		{
		}
		
		/**
		 *	Returns an array containing all of the legal values of
		 *	a radio button or drop-down list.  If a zero-size array
		 *	is returned, that indicates that the value of this
		 *	item is unconstrained.
		 */
		public String[] getValueList()
		{
			return new String[0]; // (String[])valueList.toArray();
		}
		
		/**
		 *	Returns an array containing the descriptions, which is
		 *	to say the human-readable comments that are associated
		 *	with each value.
		 */
		public String[] getDescriptionList()
		{
			return new String[0];
		}

		public Part getMultipartPostData(String itemName) throws IOException
		{
			return new StringPart(itemName, this.getValue());
		}

	}
	
	static class FormItemSubmit extends FormItem
	{
		String submitParameterName;
		
		FormItemSubmit(String name, String value)
		{
			super(ItemType.SUBMIT, ((value == null) || (value.equals(""))) ? "Submit" : value);
			submitParameterName = name;
		}
		
		public String getSubmitParameterName()
		{
			return submitParameterName;
		}
	}

	static class FormItemCheckbox extends FormItem
	{
		protected FormItemCheckbox(ItemType theType, String theValue)
		{
			super(theType, theValue);
			
			possibleValue = new String[1];
			checkboxDescription = new String[1];
			
			possibleValue[0] = theValue;
			checkboxDescription[0] = "";
		}

		String[]		possibleValue = null;
		String[]		checkboxDescription = null;
		
		public void setPossibleValue(String theValue, String theDescription)
		{
			possibleValue[0] = theValue;
			checkboxDescription[0] = theDescription;
		}
		
		public void setValue(String theValue, boolean isSelected)
		{
			if(isSelected)
				super.setValue(theValue, true);
			else
				super.setValue("", true);
		}

		public String[] getValueList()
		{
			return possibleValue;
		}
		
		public String[] getDescriptionList()
		{
			return checkboxDescription;
		}		
	}
	
	static class FormItemMultipleChoice extends FormItem
	{
		protected FormItemMultipleChoice(ItemType theType, String theValue)
		{
			super(theType, theValue);
			possibleValues = new OrderedMap();
		}
		
		//
		// LIMITATION:
		//
		// We store the possible values and their descriptions
		// in a hash map.  The description is the map key and
		// the value is the map value; this means that the use of
		// this class *requires* that the items in the multiple-choice
		// selection all have unique descriptions.  This code
		// will fail, for example, if there are a set of radio
		// buttons with empty descriptions next to, say, images
		// on an HTML page.
		//
		// The fix would be to store the possible values in a list
		// of PAIR elements, but doing this would require the
		// we provide our own 'get', 'values' and 'keySet' methods.
		// Perhaps as a future enhancement...
		//
		Map		possibleValues = null;
		
		public void setPossibleValue(String theValue, String theDescription)
		{
			possibleValues.put(theDescription, theValue);
		}
		
		public void setValue(String theValue, boolean isSelected)
		{
			if(isSelected)
				super.setValue(theValue, true);
		}
		
		public void selectValue(String theDescription)
		{
			if(possibleValues.containsKey(theDescription))
			{
				this.setValue((String)possibleValues.get(theDescription), true);
			}
		}


		public String[] getValueList()
		{
			//
			// Remember, the choice description is the map key; the choice value is the map value.
			//
			return (String[]) possibleValues.values().toArray(new String[0]);
		}
		
		public String[] getDescriptionList()
		{
			//
			// Remember, the choice description is the map key; the choice value is the map value.
			//
			return (String[]) possibleValues.keySet().toArray(new String[0]);
		}

	}
	
	//
	// Form items that allow files to be uploaded are represented by this class.
	//
	static class FormFileUploadItem extends FormItem
	{
		File		theFile = null;
		String		contentType;
		
		protected FormFileUploadItem()
		{
			super(ItemType.FILE, "");
		}
		
		protected FormFileUploadItem(String fullPathToFile) throws FileNotFoundException
		{
			super(ItemType.FILE, "");
			
			this.setFile(fullPathToFile);
		}
		
		protected FormFileUploadItem(String fullPathToFile, String theContentType) throws FileNotFoundException
		{
			super(ItemType.FILE, "");
			
			this.setFile(fullPathToFile, theContentType);
		}
				
		public static FormFileUploadItem create(String fullPathToFile, String theContentType) throws FileNotFoundException
		{
			return new FormFileUploadItem(fullPathToFile, theContentType);
		}
		
		String getFilename()
		{
			return this.getValue();
		}
		
		void setFilename(String theFilename)
		{
			this.setValue(theFilename, true);
		}
		
		//
		// If the caller does not specify the content type directly
		// then we will try to guess what it should be by peeking at
		// the bytes at the beginning of the input stream.
		//
		String getContentType()
		{
			if((contentType == null) && (theFile != null))
			{
				try
				{
					InputStream inStream = new FileInputStream(theFile);
					contentType = URLConnection.guessContentTypeFromStream(inStream);
				}
				catch(Exception e)
				{
					// just fall back on default behavior if we can't guess the content type
				}
			}
			if(contentType == null)
			{
				contentType = "application/octet-stream";
			}
			
			return contentType;
		}
		
		void setContentType(String theContentType)
		{
			contentType = theContentType;
		}
		
		void setFile(String fullPathToFile) throws FileNotFoundException
		{
			System.err.println("About to trim filename for path " + fullPathToFile);
			String fileName = fullPathToFile.substring(fullPathToFile.lastIndexOf('/') + 1);
			fileName = fileName.substring(fileName.lastIndexOf('\\') + 1);
			System.err.println("Result of trimmed filename is " + fileName);
			
			formItemType = ItemType.FILE;
			this.setFilename(fileName);
			this.theFile = new File(fullPathToFile);
			
			System.err.println("set file");
		}

		void setFile(String fullPathToFile, String theContentType) throws FileNotFoundException
		{
			this.setFile(fullPathToFile);
			this.setContentType(theContentType);
		}
		
		File getFile()
		{
			return this.theFile;
		}
		
		
		public Part getMultipartPostData(String itemName) throws IOException
		{
			return new FilePart(itemName, this.getFilename(), this.getFile(), this.getContentType(), FilePart.DEFAULT_CHARSET);
		}

	}
	
	/**
	 *	Create a new empty XHTMLForm object.  This form may be used
	 *	by calling XHTMLForm.setItemValue and XHTMLForm.setAction.
	 *	Note that the form method is always POST; method GET is not
	 *	supported by this class.
	 */
	XHTMLForm()
	{
		System.err.println("Make maps");
		formItems = new OrderedMap();
		submitButtons = new OrderedMap();
		multipartEncoded = false; // If not multipart, then encoding type is URLENCODED;
	}
	
	/**
	 *	Call this with a 'true' parameter to use the MULTIPART content encoding,
	 *	or false to use URLENCODED content encoding.  This class does not support
	 *	any other encoding scheme.
	 *
	 *	This method will generate the multipart boundary the first time
	 *	it is called.	
	 */
	public void setMultipartEncoded(boolean useMultipartEncoding)
	{
		multipartEncoded = useMultipartEncoding;
	}
	
	/**
	 *	Returns 'true' if MULTIPART encoding is being used, or false
	 *	if URLENCODED content encoding is being used.
	 */
	public boolean isMultipartEncoded()
	{
		return multipartEncoded;
	}
	
	/**
	 *	Returns the form title.
	 */
	public String getTitle()
	{
		return formTitle;
	}
	
	/**
	 *	Specify the form title.
	 */
	public void setTitle(String theTitle)
	{
		formTitle = theTitle;
	}
	
	/**
	 *	Returns the form message.
	 */
	public String getMessage()
	{
		return formMessage;
	}
	
	/**
	 *	Specify the form Message.
	 */
	public void setMessage(String theMessage)
	{
		formMessage = theMessage;
	}
	
	
	/**
	 *	Returns the URL that this form should be submitted to.
	 */
	public String getAction()
	{
		return formAction;
	}
	
	/**
	 *	Specify the URL that this form should be submitted to.
	 */
	public void setAction(String theAction)
	{
		formAction = theAction;
	}
	
	/**
	 *	Return the named form item
	 */
	/* package-private */
	IFormItem getFormItem(String key)
	{
		return (IFormItem)formItems.get(key);		
	}
	
	/**
	 *	Return a set containing all of the form item names.
	 */
	public Set getFormNameSet()
	{
		return formItems.keySet();		
	}
	
	public boolean hasUploadFileItem()
	{
		boolean result = false;
		Iterator itemNameIter = this.getFormNameSet().iterator();
		while((result == false) && (itemNameIter.hasNext()))
		{
			XHTMLForm.FormItem currentItem = (XHTMLForm.FormItem) this.getFormItem((String)itemNameIter.next());
			if(currentItem instanceof XHTMLForm.FormFileUploadItem)
			{
				result = true;
			}
		}
		
		return result;
	}
	
	public String[] getUploadItemNames()
	{
		List result = new ArrayList();
		
		Iterator itemNameIter = this.getFormNameSet().iterator();
		while(itemNameIter.hasNext())
		{
			String currentItemName = (String)itemNameIter.next();
			XHTMLForm.FormItem currentItem = (XHTMLForm.FormItem) this.getFormItem(currentItemName);
			if(currentItem instanceof XHTMLForm.FormFileUploadItem)
			{
				result.add(currentItemName);
			}
		}
		
		return (String[])result.toArray(new String[0]);
	}
	
	/* package-private */
	FormItemSubmit getSubmitButton(String submitButtonLabel)
	{
		return (FormItemSubmit)submitButtons.get(submitButtonLabel);
	}
	
	public Set getSubmitButtonLabels()
	{
		return submitButtons.keySet();
	}
	
	/**
	 *	Set the value of a form item cached in this object.
	 */
	public void setItemValue(String key, String value)
	{
		this.setItemValue(key, IFormItem.ItemType.TEXT, value, null, true);
	}
	
	/**
	 *	Set the label for a form item
	 */
	public void setItemLabel(String key, String label)
	{
		if(label.length() > 0)
		{
			IFormItem item = this.getFormItem(key);
			if(item != null)
			{
				item.setLabel(label);
			}
		}
	}
	
	/**
	 *	Add a submit button to this form
	 */
	public void addSubmitButton(String name, String label)
	{
		this.setItemValue(name, IFormItem.ItemType.SUBMIT, label, null, true);
	}
	
	/**
	 *	Set the value of a form item cached in this object.
	 */
	public void selectItemValue(String key, String description)
	{
		IFormItem item = this.getFormItem(key);
		if(item != null)
		{
			item.selectValue(description);
		}
	}
	
	/* package-private */
	void setItemValue(String key, IFormItem.ItemType theType, String value, String description, boolean isSelected)
	{
		if(theType == IFormItem.ItemType.SUBMIT)
		{
			System.err.println("Make a new submit button whose label is " + value + " and whose name is " + key);
			FormItemSubmit selectItem = new FormItemSubmit(key, value);
			submitButtons.put(value, selectItem);
		}
		else
		{
			IFormItem item = this.getFormItem(key);
			if(item != null)
			{
				item.setValue(value, isSelected);
			}
			else
			{
				item = FormItem.create(theType, value);
				formItems.put(key, item);
			}

			if(description != null)
			{
				item.setPossibleValue(value, description);
				
				//
				// If the item does not have a label and if
				// 'setPossibleValue' did not use the provided
				// description, then we will use the description
				// as the item's label.
				//
				if((item.getLabel() == null) && (item.getDescriptionList().length == 0))
				{
					item.setLabel(description);
				}
			}
		}
	}
	
	/**
	 *	Specify an a file to upload when this form is submitted.
	 */
	public void setUploadFile(String key, String fullPathToFile, String theContentType) throws FileNotFoundException
	{
		System.err.println("make new form file upload item for " + key + ", file is " + fullPathToFile);
		IFormItem item = this.getFormItem(key);
		if((item != null) && (item instanceof FormFileUploadItem))
		{
			System.err.println("set file");
			((FormFileUploadItem)item).setFile(fullPathToFile, theContentType);
			System.err.println("done with set file");
		}
		else
		{
			System.err.println("make new file upload item");
			formItems.put(key, FormFileUploadItem.create(fullPathToFile, theContentType));
		}
		
		//
		// If we specify an upload file, then switch to multipart encoded
		//
		this.setMultipartEncoded(true);
		
		System.err.println("Done with setUploadFile");
		
	}

	/**
	 *	Specify an a file to upload when this form is submitted.  The
	 *	content type is inferred from the file.
	 */
	public void setUploadFile(String key, String fullPathToFile) throws FileNotFoundException
	{
		this.setUploadFile(key, fullPathToFile, null);
	}
	
	/**
	 *	Return the type of a form item cached in this object.
	 */
	public String getItemType(String key)
	{
		String result = null;
		
		IFormItem item = this.getFormItem(key);
		if(item != null)
		{
			result = item.getType().toString();
		}
		
		return result;
	}
	
	/**
	 *	Return the current value of one of the form items cached in this object.
	 */
	public String getItemValue(String key)
	{
		String result = null;
		
		IFormItem item = this.getFormItem(key);
		if(item != null)
		{
			result = item.getValue();
		}
		
		return result;
	}
	
	/**
	 *	Get an HttpMethod suitable for use with an HttpClient for posting the form
	 *	data associated with this object.
	 *
	 *	Caller should use method.getResponseBodyAsStream(), read the whole stream, close the stream
	 *	(or equivalent operations such as method.getResponseBody()) and then call method.releaseConnection().
	 */
	public HttpMethod getPostMethod(URL urlContext) throws IOException, MalformedURLException
	{
		return this.getPostMethod(urlContext, null);
	}
	
	public HttpMethod getPostMethod(URL urlContext, String submitButtonLabel) throws IOException, MalformedURLException
	{
		FormItemSubmit submitButton = this.getSubmitButton(submitButtonLabel);
		String submitButtonName = submitButton == null ? null : submitButton.getSubmitParameterName();

		return this.getPostMethod(urlContext, submitButtonName, submitButtonLabel);
	}
	
	public HttpMethod getPostMethod(URL urlContext, String submitButtonName, String submitButtonLabel) throws IOException, MalformedURLException
	{
		String postAction = new URL(urlContext, formAction).toString();
		System.err.println("Submit form to address: " + postAction);
		
		PostMethod postMethod = new PostMethod(postAction);
		
		if(this.isMultipartEncoded() == false)
		{
			for (Iterator i = formItems.entrySet().iterator(); i.hasNext();) 
			{
				Map.Entry entry = (Map.Entry)(i.next());

				String itemName = entry.getKey().toString();
				IFormItem item = (IFormItem)entry.getValue();
				
				postMethod.addParameter(itemName, item.getValue());
			}
			
			if((submitButtonName != null) && (submitButtonLabel != null))
			{
				postMethod.addParameter(submitButtonName, submitButtonLabel);
			}
		}
		else
		{
			List partList = new ArrayList();
			
			for (Iterator i = formItems.entrySet().iterator(); i.hasNext();) 
			{
				Map.Entry entry = (Map.Entry)(i.next());

				String itemName = entry.getKey().toString();
				IFormItem item = (IFormItem)entry.getValue();
				
				Part theItemPostData = item.getMultipartPostData(itemName);
				partList.add(theItemPostData);
			}

			if((submitButtonName != null) && (submitButtonLabel != null))
			{
				partList.add(new StringPart(submitButtonName, submitButtonLabel));
			}

			Part[] partArray = (Part[]) partList.toArray(new Part[0]);
			
			postMethod.setRequestEntity(new MultipartRequestEntity(partArray, postMethod.getParams()));
		}
		
		return postMethod;	
	}	
}
