/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */


package org.netbeans.modules.bpel.debugger.test;

import junit.framework.TestCase;

import org.netbeans.modules.bpel.debugger.api.Position;
import org.netbeans.modules.bpel.debugger.api.ProcessInstance;
import org.netbeans.modules.bpel.debugger.BreakPosition;
import org.netbeans.modules.bpel.debugger.bdiclient.impl.BDIDebugController;
import org.netbeans.modules.bpel.debugger.bdiclient.impl.BDIDebugFrame;
import org.netbeans.modules.bpel.debugger.test.stub.SimulationBreakpointThread;
import org.netbeans.modules.bpel.debugger.test.stub.StubBreakViewController;


public class BDIDebugControllerMultiThreaded_Test extends TestCase {
    
    
    /**
     * This is a basic multi-threaded break-handling test.
     * 1. Three breakpoints are created for two different processes (hotels and air).
     * 2. Four threads are then executed: hotels1, air, hotels2, train.
     *    - Thread train encounters no breakpoints, and runs until completion.
     *    - The other three threads encounter breakpoints and suspend.
     * 3. Thread hotels1 is continued, and because it encounters no further breakpoints,
     *    it executes to completion.
     * 4. Thread air is continued, and it encounters its second and last breakpoint,
     *    upon which it suspends again.
     * 5. The debugger is detached, and thus all suspended threads (hotels2 and air)
     *    are immediately resumed and they will run to completion.
     */
    public void testGeneralBreaking() {
        StubBreakViewController stubView = new StubBreakViewController();
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_HOTELS, SimulationBreakpointThread.LINE_BREAK_HOTELS);
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_AIR,    SimulationBreakpointThread.LINE_BREAK_AIR1);
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_AIR,    SimulationBreakpointThread.LINE_BREAK_AIR2);
        BDIDebugController controller = new BDIDebugController(stubView);
        controller.setAttached(true);
        
        SimulationBreakpointThread[] threads = SimulationBreakpointThread.createSimulationThreads(this, controller);
        SimulationBreakpointThread threadHotels1 = threads[0];
        SimulationBreakpointThread threadAir     = threads[1];
        SimulationBreakpointThread threadHotels2 = threads[2];
        SimulationBreakpointThread threadTrain   = threads[3];

        assertTrue(controller.isAttached());
        
        // start all threads, wait until all are either suspended or complete
        SimulationBreakpointThread.startThreads(this, threads);
        BDIDebugFrame frameHotels1 = (BDIDebugFrame) threadHotels1.getFrame();
        BDIDebugFrame frameAir     = (BDIDebugFrame) threadAir.getFrame();
        BDIDebugFrame frameHotels2 = (BDIDebugFrame) threadHotels2.getFrame();
        BDIDebugFrame frameTrain   = (BDIDebugFrame) threadTrain.getFrame();

        assertTrue(threadHotels1.isAlive());
        assertTrue(threadAir.isAlive());
        assertTrue(threadHotels2.isAlive());
        assertFalse(threadTrain.isAlive());
        
        assertNotNull(stubView.getActivePosition());
        assertNull(stubView.getErrorMessages());
        
        assertNotNull(threadHotels1.getFrame());
        assertNotNull(threadAir.getFrame());
        assertNotNull(threadHotels2.getFrame());
        assertNotNull(threadTrain.getFrame());
        
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
        
        Position positionHotels1 = frameHotels1.getCurrentPosition();
        Position positionAir     = frameAir.getCurrentPosition();
        Position positionHotels2 = frameHotels2.getCurrentPosition();
        Position positionTrain   = frameTrain.getCurrentPosition();
        
        assertNotNull(positionHotels1);
        assertNotNull(positionAir);
        assertNotNull(positionHotels2);
        assertNull(positionTrain);
        
        assertEquals(SimulationBreakpointThread.URL_NB_HOTELS, positionHotels1.getURL());
        assertEquals(SimulationBreakpointThread.URL_NB_AIR,    positionAir.getURL());
        assertEquals(SimulationBreakpointThread.URL_NB_HOTELS, positionHotels2.getURL());
        
        assertEquals(SimulationBreakpointThread.LINE_BREAK_HOTELS, positionHotels1.getLineNumber());
        assertEquals(SimulationBreakpointThread.LINE_BREAK_AIR1,   positionAir.getLineNumber());
        assertEquals(SimulationBreakpointThread.LINE_BREAK_HOTELS, positionHotels2.getLineNumber());
        
        // STEP-CONTINUE: resume hotels1 reservations thread, this causes it to complete
        frameHotels1.resume();
        try {
            Thread.sleep(500);
        } catch (Exception e) {
            assertNull(e);
        }
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
        positionHotels1 = frameHotels1.getCurrentPosition();
        assertNull(positionHotels1);
        
        // STEP-CONTINUE: move air reservations from the first to the second breakpoint
        stubView.setActivePositionClearedFlag(false); // reset position-cleared flag
        frameAir.resume();
        try {
            Thread.sleep(500);
        } catch (Exception e) {
            assertNull(e);
        }
        assertTrue(stubView.getActivePositionClearedFlag()); // check position-cleared flag
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
        positionAir = frameAir.getCurrentPosition();
        assertEquals(SimulationBreakpointThread.LINE_BREAK_AIR2, positionAir.getLineNumber());
        
        // DETACH from debugging: this resumes all threads, which should eventually complete
        controller.detach();
        try {
            Thread.sleep(500);
        } catch (Exception e) {
            assertNull(e);
        }
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
    }
    
    
    /**
     * This tests multiple breakpoint stepping via Step-Into.
     * 1. Three breakpoints are created for two different processes (hotels and air).
     * 2. Four threads are then executed: hotels1, air, hotels2, train.
     *    - Thread train encounters no breakpoints, and runs until completion.
     *    - The other three threads encounter breakpoints and suspend.
     * 3. Thread air is stepped-into 20 times. This causes it to break 20 times
     *    and its current line number to increment each time.
     *    Meanwhile, Thread hotels1 is resumed, and it runs to completion.
     * 4. The debugger is detached, which causes all suspended threads (air and hotels2)
     *    to immediately resume and run to completion. Thread air has one remaining
     *    unreached breakpoint, but because the debugger was detached, that breakpoint
     *    is passed without blocking.
     */
    public void testStepping() {
        StubBreakViewController stubView = new StubBreakViewController();
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_HOTELS, SimulationBreakpointThread.LINE_BREAK_HOTELS);
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_AIR,    SimulationBreakpointThread.LINE_BREAK_AIR1);
        stubView.addStubBreakpoint(SimulationBreakpointThread.URL_NB_AIR,    SimulationBreakpointThread.LINE_BREAK_AIR2);
        BDIDebugController controller = new BDIDebugController(stubView);
        controller.setAttached(true);

        SimulationBreakpointThread[] threads = SimulationBreakpointThread.createSimulationThreads(this, controller);
        SimulationBreakpointThread threadHotels1 = threads[0];
        SimulationBreakpointThread threadAir     = threads[1];
        SimulationBreakpointThread threadHotels2 = threads[2];
        SimulationBreakpointThread threadTrain   = threads[3];

        // Start all threads, wait until all are either suspended or complete.
        // Ensure the air reservation thread starts first, and sets the position first.
        SimulationBreakpointThread.startThreads(this, threads, threadAir);
        BDIDebugFrame frameHotels1 = (BDIDebugFrame) threadHotels1.getFrame();
        BDIDebugFrame frameAir     = (BDIDebugFrame) threadAir.getFrame();
        BDIDebugFrame frameHotels2 = (BDIDebugFrame) threadHotels2.getFrame();
        BDIDebugFrame frameTrain   = (BDIDebugFrame) threadTrain.getFrame();
        
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
        
        BreakPosition positionAir = (BreakPosition) frameAir.getCurrentPosition();
        assertNotNull(positionAir);
        
        int expectedBreakLine = SimulationBreakpointThread.LINE_BREAK_AIR1;
        assertEquals(expectedBreakLine, positionAir.getLineNumber());
        
        // Step one line, wait for the thread to continue and then break, then verify results.
        for (int i=0; i < 20; i++) {
            stepOneLine(controller, positionAir, frameAir, expectedBreakLine);
            if (i==10) {
                frameHotels1.resume();
            } else if (i==15) {
                assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels1.getState());
            }
            expectedBreakLine++;
        }
        
        // DETACH from debugging: this resumes all threads, which should eventually complete
        controller.detach();
        try {
            Thread.sleep(500);
        } catch (Exception e) {
            assertNull(e);
        }
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels1.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameAir.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameHotels2.getState());
        assertEquals(ProcessInstance.STATE_COMPLETED, frameTrain.getState());
    }
    
    
    private void stepOneLine(BDIDebugController controller, 
                             BreakPosition position,
                             BDIDebugFrame frame,
                             int expectedBreakLine) {
        // Step one line and wait.
        System.out.println("Stepping to line: " + (expectedBreakLine + 1));
        controller.stepInto(position);
        try {
            Thread.sleep(500);
        } catch (Exception e) {
            assertNull(e);
        }
        position = (BreakPosition) frame.getCurrentPosition();
        assertEquals(expectedBreakLine + 1, position.getLineNumber());
        assertEquals(ProcessInstance.STATE_SUSPENDED, frame.getState());
    }

}
