// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/jit.cpp,v 1.6 2001/11/23 02:12:44 xli18 Exp $
//


#include "defines.h"
#include <assert.h>
#include <iostream.h>
#include "jit_intf.h"
#include "orp_types.h"
#include "jit_runtime_support.h"
#include "jit.h"
#include "code_gen.h"
#include "x86.h"
#include "gc_tags.h"
#include "regalloc.h"
#include "register_allocator.h"

//
// just for testing purpose (remove the include later)
//
#include "o1_debugging_support.h"

void *L1a_FixHandlerContext(
	Frame_Context	*context,
	void			*methodStart,
	void			*methodInfoPtr,
	void	*thrownObj);

#ifdef MEASURE_JIT_TIME

#define rdtsc __asm _emit 0x0F __asm _emit 0x31

#pragma warning(disable: 4035)


#ifdef ORP_POSIX
static __int64 readTimeStampCounter()
{
    assert(0);
}

#else

static __int64 readTimeStampCounter() {
	_asm {
		rdtsc     
	}
} //readTimeStampCounter
#endif  // ORP_POSIX


#pragma warning(default: 4035)
#endif //MEASURE_JIT_TIME

//
// main entry point for jit dll
//
JIT_Result L1a_compile(
    Compile_Handle   compilationHandle,
    Class_Handle  classHandle,
    Method_Handle methodHandle,
    const BYTE *  byteCodeAddr,
    size_t        byteCodeSize,
    
    unsigned      maxLocals,
    unsigned      maxStack,
    JIT_Flags     flags,
    Dbg           *dbg_support,
    CODE_MI       code_mi) {

#ifdef MEASURE_JIT_TIME
	static unsigned __int64 cumulative_JIT_time = 0;
	unsigned __int64 start = readTimeStampCounter();
	JIT_Result result = O1a_gen_code(compilationHandle,classHandle,methodHandle,byteCodeAddr,
									byteCodeSize,maxLocals,maxStack,flags,dbg_support,code_mi);
	unsigned __int64 end = readTimeStampCounter();
	unsigned __int64 d = end - start;
	cumulative_JIT_time += d;
	double secs = ((double)(__int64)cumulative_JIT_time) / (double)200000000;
	cout << "Cumulative JIT time is " << secs << " s" << endl;
	return result;
#else // MEASURE_JIT_TIME
	return O1a_gen_code(compilationHandle,classHandle,methodHandle,byteCodeAddr,
						byteCodeSize,maxLocals,maxStack,flags,dbg_support,code_mi);
#endif // MEASURE_JIT_TIME

}

/////////////////////////////////////////////////////////////////////
//
// Beginning of caching code copied and modified from O3.
//
/////////////////////////////////////////////////////////////////////

#define UCACHE_SIZE 1024
static struct {
    uint32 eip;
    unsigned num_out_args;
    unsigned restore_mask;
	Boolean is_first; //::
} unwind_cache[UCACHE_SIZE];
static CRITICAL_SECTION ucache_lock;
static bool ucache_lock_initialized = false;

// Parameters needed: context, num_incoming, num_spill, num_callee, callee_restore_mask
static bool unwind_with_cache(Frame_Context *context, Jit_Method_Info *method_info, Boolean is_first) //::
{
#ifdef DISABLE_CACHING
    return false;
#endif // DISABLE_CACHING
    if (!ucache_lock_initialized)
    {
        InitializeCriticalSection(&ucache_lock);
        ucache_lock_initialized = true;
    }
    // Try to acquire a lock.  If it's already locked, return false.
    if (!TryEnterCriticalSection(&ucache_lock))
        return false;

    bool retval = false;
    unsigned i = *context->p_eip % UCACHE_SIZE;
	if (i==0xa4){
		int ii =1;
	}
    if (unwind_cache[i].eip == *context->p_eip && unwind_cache[i].is_first == is_first)
    {
        retval = true;
        unsigned restore_mask = unwind_cache[i].restore_mask;
        unsigned num_out_args = unwind_cache[i].num_out_args;
        ESP_Frame	esp_frame(method_info->num_in_args,
            method_info->num_vars,
            method_info->num_gc_tag_words,
            method_info->num_spills,
            method_info->num_callee_saved_regs);
        
        EBP_Frame	ebp_frame(method_info->num_in_args,
            method_info->num_vars,
            method_info->num_gc_tag_words,
            method_info->num_spills,
            method_info->num_callee_saved_regs);
        Frame *frame;
        unsigned base_addr;
        if (method_info->is_esp_based)
        {
            frame = &esp_frame;
            base_addr = context->esp;
        }
        else
        {
            frame = &ebp_frame;
            base_addr = *context->p_ebp;
        }
        unsigned frame_size = (method_info->frame_size - method_info->num_in_args + num_out_args) * 4;
        frame->push(num_out_args);
        unsigned esp = context->esp;
        unsigned old_eip = *context->p_eip;
        context->esp += frame_size;
        context->p_eip = (uint32 *)(context->esp - 4);
        if ((restore_mask & (1u << ebx_reg)))
            context->p_ebx = (uint32 *)(base_addr + frame->callee_offset(ebx_reg));
        if ((restore_mask & (1u << ebp_reg)))
            context->p_ebp = (uint32 *)(base_addr + frame->callee_offset(ebp_reg));
        if ((restore_mask & (1u << esi_reg)))
            context->p_esi = (uint32 *)(base_addr + frame->callee_offset(esi_reg));
        if ((restore_mask & (1u << edi_reg)))
            context->p_edi = (uint32 *)(base_addr + frame->callee_offset(edi_reg));
    }
    // Release the lock before returning.
    LeaveCriticalSection(&ucache_lock);
    return retval;
}

static void unwind_add_to_cache(uint32 eip, unsigned num_out_args, unsigned restore_mask)
{
#ifdef DISABLE_CACHING
    return;
#endif // DISABLE_CACHING
    // Try to acquire a lock.  If it's already locked, just return without doing anything.
    if (!TryEnterCriticalSection(&ucache_lock))
        return;

    unsigned i = eip % UCACHE_SIZE;

    unwind_cache[i].eip = eip;
    unwind_cache[i].num_out_args = num_out_args;
    unwind_cache[i].restore_mask = restore_mask;

    // Release the lock before returning.
    LeaveCriticalSection(&ucache_lock);
}

#define TCACHE_SIZE 16
static struct {
    uint32 eip;
    Boolean is_first;
    int this_offset;
} thisaddr_cache[TCACHE_SIZE];
static CRITICAL_SECTION tcache_lock;
static bool tcache_lock_initialized = false;

static bool thisaddr_with_cache(Frame_Context *context, Boolean isFirst, int &this_offset)
{
#ifdef DISABLE_CACHING
    return false;
#endif // DISABLE_CACHING
    if (!tcache_lock_initialized)
    {
        InitializeCriticalSection(&tcache_lock);
        tcache_lock_initialized = true;
    }
    // Try to acquire a lock.  If it's already locked, return false.
    if (!TryEnterCriticalSection(&tcache_lock))
        return false;

    bool retval = false;
    unsigned i = *context->p_eip % TCACHE_SIZE;
    if (thisaddr_cache[i].eip == *context->p_eip &&
        thisaddr_cache[i].is_first == isFirst)
    {
        this_offset = thisaddr_cache[i].this_offset;
        retval = true;
    }
    // Release the lock before returning.
    LeaveCriticalSection(&tcache_lock);
    return retval;
}

static void thisaddr_add_to_cache(Frame_Context *context, Boolean isFirst,
                                  int this_offset)
{
#ifdef DISABLE_CACHING
    return;
#endif // DISABLE_CACHING
    // Try to acquire a lock.  If it's already locked, just return without doing anything.
    if (!TryEnterCriticalSection(&tcache_lock))
        return;

    unsigned i = *context->p_eip % TCACHE_SIZE;

    thisaddr_cache[i].eip = *context->p_eip;
    thisaddr_cache[i].is_first = isFirst;
    thisaddr_cache[i].this_offset = this_offset;

    // Release the lock before returning.
    LeaveCriticalSection(&tcache_lock);
}

/////////////////////////////////////////////////////////////////////
//
// End of caching code copied and modified from O3.
//
/////////////////////////////////////////////////////////////////////

//
// restore callee-save  register 
//
void restore_callee(uint32*& pEcallee, char *callee,
					unsigned *base_addr, int frame_off) {
#ifdef TRACE_LEVEL1A
	cout << "Restoring " << callee << " from " 
		 << (void *)*(uint32 *)pEcallee << " @ " << (void *)pEcallee << endl;
	cout << "              to " << (void *)*(uint32 *)(base_addr + (frame_off >> 2)) 
		 << " @ " << (void *)(base_addr + (frame_off >> 2)) << endl;
#endif // TRACE_LEVEL1A
		pEcallee = (uint32 *)(base_addr + (frame_off >> 2));
}

void enumerate_ref_var_in_callee(Frame *frame, uint32*& pEcallee, char *callee,
					  GC_Enumeration_Handle hCallback,
					  unsigned word_no, unsigned mask, 
					  unsigned var_no,  unsigned *base_addr, int frame_off) {
	//
	// compute address of variable in stack frame by adding
	// offset of variable to frame base address
	//
	int offset = frame->extra_offset(word_no);
	offset = offset >> 2;
	unsigned value = *(base_addr + offset);
	if (value & mask) {
#ifdef TRACE_LEVEL1A
		cout << "Enumerating " << callee << " with local var #" << var_no
			 << " ref " << (void *)*pEcallee << " @ " << (void *)pEcallee << endl;
#endif // TRACE_LEVEL1A
		orp_enumerate_root_reference(hCallback, (void **)pEcallee);
	}
}
//
//	linear_search_call_site()
//
//	used by EnumerateGcRefs() to find which call site the method is at
//	so we know (1) how many outgoing arguments are on the stack (2) what
//	the types of these arguments are (3) how many live entries are on the
//	mimic stack and (4) what the types of those entries are.
//
int linear_search_call_site(Jit_Method_Info *method_info,DWORD ret_IP) {
	for (unsigned i = 0; i < method_info->num_call_sites; i++) {
		if (((DWORD)method_info->cs_info[i].ret_IP) == ret_IP)
			return (int)i;
	}

	return -1;
}
static int linear_search_precall_site(Jit_Method_Info *method_info,DWORD precall_IP) {
	for (unsigned i = 0; i < method_info->num_call_sites; i++) {
		if (((DWORD)method_info->cs_info[i].precall_IP) == precall_IP)
			return (int)i;
	}

	return -1;
}
//
// use binary search to find the corresponding call site info 
//
int binary_search_call_site(Jit_Method_Info *method_info, DWORD ret_IP) {
	int num_call_sites = method_info->num_call_sites;
	Call_Site_Info *cs_info = method_info->cs_info;
	int low = 0, up = num_call_sites - 1;
	while (low <= up) {
		unsigned mid = (low + up) / 2;
		if (cs_info[mid].ret_IP > ret_IP) 
			up = mid - 1;
		else if (cs_info[mid].ret_IP == ret_IP)
			return mid;
		else  // bvle->id <  call_bc
			low = mid + 1;
	}
	return -1;
}

unsigned enumerate_locals(Jit_Method_Info *mi, 
					  Frame *frame,
					  unsigned *base_addr,
				 	  Frame_Context *context,
					  GC_Enumeration_Handle	hCallback,
                      Call_Site_Info *csi,
                      int dbg_this) {
    unsigned result = 0;
#ifdef VAR_CLONING
	if(!IS_INVALID_RV_BITMAP(mi->ref_var_bitmap)) {
		unsigned n_locals = mi->num_in_args + mi->num_vars;
		Ref_Var_Bitmap_Type mmask = 1;
		Ref_Var_Bitmap_Type rvb = mi->ref_var_bitmap;
		for (unsigned k = 0; k < n_locals; k++, mmask <<= 1) {
			if(mmask & rvb) {
				int offset = frame->var_offset(k);
				offset = offset >> 2;
				unsigned *ref_addr = base_addr + offset;
// k is the variable number, and ref_addr is currently its location
// on the stack.  If k is not live at this point, we should continue.
                X86_Reg_No reg;
                int live;
                mi->rra.get_register_mapping_and_liveness(k,csi,mi, reg,live);
                if (!live)
                    continue;
                switch (reg)
                {
                case esi_reg:
                    ref_addr = (unsigned *) pContext->pEsi;
                    result |= (1 << reg);
                    break;
                case edi_reg:
                    ref_addr = (unsigned *) pContext->pEdi;
                    result |= (1 << reg);
                    break;
                case ebx_reg:
                    ref_addr = (unsigned *) pContext->pEbx;
                    result |= (1 << reg);
                    break;
                case ebp_reg:
                    ref_addr = (unsigned *) pContext->pEbp;
                    result |= (1 << reg);
                    break;
                default:
                    break;
                }
                //call_some_function();
                // decide whether "k" is in a register, and which register,
                // and whether "k" is in fact live at this point.
#ifdef TRACE_LEVEL1A
				cout << "\tenumerating var # " << k
					 << " ref " << (void*)*ref_addr
					 << " @ " << (void*)ref_addr
					 << " base_reg " << base_addr
					 << " offset " << offset
					 << endl;
#endif // TRACE_LEVEL1A
				
				pCallback(hCallback, (void *)ref_addr);
			}
		}
		return result;
	}
#endif //VAR_CLONING
	if (mi->num_gc_tag_words > 0) {
		unsigned n_locals = mi->num_in_args + mi->num_vars;
		for (unsigned k = 0; k < n_locals; k++) {
			unsigned word_no = GC_Tags_word_no(k);
			unsigned mask = GC_Tags_mask(k);
            X86_Reg_No reg;
            int live;
            mi->rra.get_register_mapping_and_liveness(k,csi,mi, reg,live);
            if (!live)
                continue;
            switch (reg)
            {
            case esi_reg:
                enumerate_ref_var_in_callee(frame,context->p_esi,"esi",hCallback,
                    word_no,mask,k,base_addr,frame->callee_offset(esi_reg));
                result |= (1<<esi_reg);
                continue;
                break;
            case edi_reg:
                enumerate_ref_var_in_callee(frame,context->p_edi,"edi",hCallback,
                    word_no,mask,k,base_addr,frame->callee_offset(edi_reg));
                result |= (1<<edi_reg);
                continue;
                break;
            case ebx_reg:
                enumerate_ref_var_in_callee(frame,context->p_ebx,"ebx",hCallback,
                    word_no,mask,k,base_addr,frame->callee_offset(ebx_reg));
                result |= (1<<ebx_reg);
                continue;
                break;
            case ebp_reg:
				enumerate_ref_var_in_callee(frame,context->p_ebp,"ebp",hCallback,
							   word_no,mask,k,base_addr,frame->callee_offset(ebp_reg));
                result |= (1<<ebp_reg);
				continue;
                break;
            default:
                break;
            }
            //call_some_function();
            // decide whether "k" is in a register, and which register,
            // and whether "k" is in fact live at this point.
			//
			// compute address of variable in stack frame by adding
			// offset of variable to frame base address
			//
			int offset = frame->extra_offset(word_no);
			offset = offset >> 2;
			unsigned *gc_tag_addr = base_addr + offset;
			unsigned value = *gc_tag_addr;
			if (value & mask) {
				//
				// enumerate this local
				//
				offset = frame->var_offset(k);
				offset = offset >> 2;
//				offset += csi->num_out_args;
				unsigned *ref_addr = base_addr + offset;
#ifdef TRACE_LEVEL1A
				if(dbg_this) {
				cout << "\tenumerating var # " << k
					 << " ref " << (void*)*ref_addr
					 << " @ " << (void*)ref_addr
					 << " base_reg " << base_addr
					 << " offset " << offset
					 << " tag word value " << (void*)value
					 << " @ " << (void*)gc_tag_addr
					 << " mask " << (void*)mask
					 << endl;
				cout << " gc_tag_addr     = " << (void*)*(gc_tag_addr) << endl;
				}
#endif // TRACE_LEVEL1A
				
				orp_enumerate_root_reference(hCallback, (void **)ref_addr);
			}
		}
	}
    return result;
}

void L1a_EnumerateGcRefs(
	Frame_Context	*context,
	void			*methodStart,
	void			*methodInfoPtr,
	GC_Enumeration_Handle			hCallback,
    Boolean          is_first) {

#if  0  // KEN! testing jvmdi (remove them later)
    int   ival;
    float fval;
    void  *ref;
    jvmdiError err;
    context->is_first = is_first;
    err = GetLocalObject(context, 0, &ref);
    cout << "Var0 = " << ref << endl;
    err = GetLocalObject(context, 1, &ref);
    cout << "Var1 = " << ref << endl;
    err = GetLocalInt(context, 6, &ival);
    cout << "Var6 = " << ival << endl;
    err = GetLocalInt(context, 7, &ival);
    cout << "Var7 = " << ival << endl;
    err = GetLocalInt(context, 8, &ival);
    cout << "Var8 = " << ival << endl;
    err = GetLocalInt(context, 9, &ival);
    cout << "Var9 = " << ival << endl;
    err = SetLocalInt(context, 9, 4);
    err = GetLocalInt(context, 9, &ival);
    cout << "new Var9 = " << ival << endl;
#endif


    unsigned regs_enumerated; // references cannot be enumerated twice.

	// fetch method/call-site specific info
	Jit_Method_Info *mi = (Jit_Method_Info *)methodInfoPtr;

	//
	// pop the stack frame
	//
	uint32 Esp = context->esp;
    bool enumerate_outargs = false;
	int cur_call_site;
    if (is_first)
    {
        enumerate_outargs = true;
        cur_call_site = linear_search_precall_site(mi, *context->p_eip);
    }
    else
    {
        if (mi->num_call_sites > BINARY_CALL_SITE)
            cur_call_site = binary_search_call_site(mi, *context->p_eip);
        else		
            cur_call_site = linear_search_call_site(mi, *context->p_eip);
    }
    assert(cur_call_site != -1);
    Call_Site_Info *csi = &mi->cs_info[cur_call_site];
	int dbg_this = 1;
#ifdef TRACE_LEVEL1A
	if (dbg_this) {
	cout << "L1a_EnumerateGcRefs\n";
	cout << "--------------------------------------------------------" << endl;
	cout << "\tmethod " << (char *)mi->name << endl;
	cout << "\tis_esp_based " << (int)mi->is_esp_based << endl;
	cout << "\tnum_out_args " << csi->num_out_args << endl;
	cout << "\tn_spills " << mi->num_spills << endl;
	cout << "\tn_vars " << mi->num_vars << endl;
	cout << "\tn_gc_tag_words " << mi->num_gc_tag_words << endl;
	cout << "\tn_args " << mi->num_in_args <<  endl;
	cout << "\tpPC " << (unsigned*)*context->p_eip << endl;
		if (cur_call_site < 0 || mi->cnt != mi->num_call_sites) 
			cout << " Assertion failure " << endl;
		assert(cur_call_site >= 0);
		cout << "\tnum_out_args " << csi->num_out_args << endl;
	cout << endl;
	cout << "old Ebp = " << (void *)*context->p_ebp << endl;
	cout << "old Esp = " << (void *)context->esp << endl;
	cout << "old Eip = " << (void *)*context->p_eip << endl;
	}
#endif // TRACE_LEVEL1A

	ESP_Frame	esp_frame(mi->num_in_args,mi->num_vars,
        mi->num_gc_tag_words,
		                  mi->num_spills,mi->num_callee_saved_regs);

	EBP_Frame	ebp_frame(mi->num_in_args,mi->num_vars,
        mi->num_gc_tag_words,
		                  mi->num_spills,mi->num_callee_saved_regs);

	Frame *frame;
	//
	// The frame base address is the address in the base register
	// all offset are relative to this base address
	//
	unsigned *base_addr;
	unsigned is_esp_based = mi->is_esp_based;
	if (is_esp_based) {
		frame = &esp_frame;
		base_addr = (unsigned*)context->esp;
	} else {
		frame = &ebp_frame;
		base_addr = *(unsigned **)context->p_ebp;
	}
    {
		//
		// adjust ESP for the outgoing arguments that have been pushed
		//
		frame->push(csi->num_out_args);
		//
		// enumerate locals that are references
		//
		regs_enumerated = enumerate_locals(mi,frame,base_addr,context,hCallback,csi,
            dbg_this);
		//
		// enumerate stack locations that contain object/array references
		//
		for (unsigned k = 0; k < csi->stack_depth; k++) {
			if (GET_TV_STACK(csi, k) == REF_TYPE_CHAR) {	// always live
				int offset = frame->spill_offset(k);
				offset = offset >> 2;

				unsigned *ref_addr = base_addr + offset;
#ifdef TRACE_LEVEL1A
				if (dbg_this)
				cout << "\tenumerating spill # " << k
					 << " ref " << (void*)*ref_addr
					 << " @ " << (void*)ref_addr
					 << " base_reg " << base_addr
					 << " offset " << offset
					 << endl;
#endif // TRACE_LEVEL1A
				
				orp_enumerate_root_reference(hCallback, (void **)ref_addr);
			}
		}
        // Enumerate outargs if necessary.
        if (enumerate_outargs)
        {
            if (csi->outarg_bv != 0)
            {
                for (unsigned bit=0; bit<8; bit++)
                {
                    if (csi->outarg_bv & (1 << bit))
                        orp_enumerate_root_reference(hCallback,
                        (void **)(base_addr + (frame->outarg_offset(bit) >> 2)));
                }
            }
            if (csi->m_handle != NULL)
            {
                // Use the method's signature to decide which outargs are refs.
                // XXX- this has to be changed if passing args in registers.
                unsigned cur_outarg = 0;
                // enumerate outgoing "this" pointer
                if (!method_is_static(csi->m_handle))
                {
                    orp_enumerate_root_reference(hCallback,
                        (void **)(base_addr + (frame->outarg_offset(cur_outarg) >> 2)));
                    cur_outarg ++;
                }
                Arg_List_Iterator iter = method_get_argument_list(csi->m_handle);
                Java_Type typ;
                while((typ = curr_arg(iter)) != JAVA_TYPE_END)
                {
                    if (typ == JAVA_TYPE_CLASS || typ == JAVA_TYPE_ARRAY)
                    {
                        orp_enumerate_root_reference(hCallback,
                            (void **)(base_addr + (frame->outarg_offset(cur_outarg) >> 2)));
                    }
                    cur_outarg ++;
                    if (typ == JAVA_TYPE_LONG || typ == JAVA_TYPE_DOUBLE)
                        cur_outarg ++;
                    iter = advance_arg_iterator(iter);
                }
            }
        }

#ifdef LOCAL_CALLEE
		// enumerate stack ref that are in local callee registers
		for (int i = ebx_reg; i < n_reg; i++) 
			if (!(regs_enumerated & (1<<i)) && // don't enumerate a register location twice
                (csi->stack_ref_in_local_callee & (1<<i))) {
				if (i == ebx_reg)     pCallback(hCallback, (void *)pContext->pEbx);
				else if(i == ebp_reg) pCallback(hCallback, (void *)pContext->pEbp);
				else if(i == esi_reg) pCallback(hCallback, (void *)pContext->pEsi);
				else if(i == edi_reg) pCallback(hCallback, (void *)pContext->pEdi);
			}
				
#endif // LOCAL_CALLEE

                if (mi->rra.should_restore_callee(mi,ebx_reg))
                    restore_callee(context->p_ebx,"ebx",base_addr,frame->callee_offset(ebx_reg));
                if (mi->rra.should_restore_callee(mi,ebp_reg))
                    restore_callee(context->p_ebp,"ebp",base_addr,frame->callee_offset(ebp_reg));
                if (mi->rra.should_restore_callee(mi,esi_reg))
                    restore_callee(context->p_esi,"esi",base_addr,frame->callee_offset(esi_reg));
                if (mi->rra.should_restore_callee(mi,edi_reg))
                    restore_callee(context->p_edi,"edi",base_addr,frame->callee_offset(edi_reg));


		// pop outgoing args and register
		Esp += csi->num_out_args << 2;

		// pop/restore callee_saved_regs (not used yet)
		Esp += mi->num_callee_saved_regs << 2;

		// pop spill stack references and register them
		Esp += mi->num_spills << 2;

		// pop local variables (not including incoming args) and register them
		Esp += mi->num_vars << 2;

		// pop extras
		Esp += mi->num_gc_tag_words << 2;

		// for ebp-based frames, we must restore saved ebp from stack
		if (!is_esp_based) {
			context->p_ebp = (uint32 *)Esp;
			// pop Ebp
			Esp += 4;
		}
	}

	// update return IP
	context->p_eip = (uint32 *)Esp;

	// pop return IP
	Esp += 4;

	// update ESP
	context->esp = Esp;

#ifdef TRACE_LEVEL1A
	cout << endl;
	cout << "new Ebp = " << (void *)*context->p_ebp << endl;
	cout << "new Esp = " << (void *)context->esp << endl;
	cout << "new Eip = " << (void *)*context->p_eip << endl;
	cout << "\tmethod " << (char *)mi->name << " end" << endl;
#endif // TRACE_LEVEL1A
}

//
//	compute_call_site_offset()
//	
//	returns the number of arguments on the stack (real esp offset, not depth of
//	mimic stack) based upon the current location (cur_IP) within a method.
//
unsigned compute_call_site_offset(Jit_Method_Info *method_info,
                                  DWORD cur_IP,
                                  Boolean isFirst)
{
	// isFirst == is the the first method on the stack ==> can be anywhere within the method
	if (isFirst) {
		// check if we are pushing args, doing the call, or popping args
		for (unsigned i = 0; i < method_info->num_call_sites; i++) {
			Call_Site_Info *csi = &method_info->cs_info[i];
			for (unsigned j = 0; j < csi->num_records; j++) {
				if (csi->esp_record[j].offset == cur_IP) {
#ifdef TRACE_LEVEL1A
					cout << "@@@ call site #   = " << (void *)i << endl;
					cout << "@@@ record #      = " << (void *)j << endl;
					cout << "@@@ offset        = " << (void *)cur_IP << endl;
					cout << "@@@ args on stack = " << (void *)csi->esp_record[j].args_on_stack << endl;
#endif // TRACE_LEVEL1A
					return csi->esp_record[j].args_on_stack;
				}
			}
		}

		// not doing any of the above, so there are no args on the stack
		return 0;
	} else {
        // could be anywhere for NullPointer and Arithmeticm return 0
        // if not found
		// ??definitely at a call site, and args are still on the stack
		for (unsigned i = 0; i < method_info->num_call_sites; i++) {
			Call_Site_Info *csi = &method_info->cs_info[i];
			if (csi->ret_IP == cur_IP)
				return csi->num_out_args;
		}
	}

	// default
	return 0;
}



void *L1a_FixHandlerContext(
	Frame_Context	*context,
	void		*methodStart,
	void		*methodInfoPtr,
	void *thrownObj) {
 	Jit_Method_Info *method_info = (Jit_Method_Info*)methodInfoPtr;
    // compute frame size
	DWORD frame_size = method_info->frame_size;
	frame_size -= method_info->num_in_args;		// don't pop incoming arguments
	frame_size--;
	if (method_info->is_esp_based == 0) frame_size--;
    context->esp = *context->p_ebp - (frame_size << 2);

#ifdef TRACE_LEVEL1A
	cout << "L1a_FixHandlerContext " << method_info->name << endl;
    cout << hex <<
    "Eip = " << *context->p_eip <<
	", Ebx = " << *context->p_ebx <<
	", Esi = " << *context->p_esi <<
	", Edi = " << *context->p_edi <<
	", Ebp = " << *context->p_ebp <<
	", Esp = " << context->esp << dec << " // " << method_info->name << endl;
#endif // TRACE_LEVEL1A

	return NULL;
}

void L1a_UnwindStackFrame_Full(
	Frame_Context *context,         /*INOUT*/
    void *      methodInfoPtr,    /* IN  */
    Boolean     isFirst,          /* IN  */
    void *      methodStart)      /* IN  */
{
	Jit_Method_Info *method_info = (Jit_Method_Info*)methodInfoPtr;
    if (unwind_with_cache(context, method_info, isFirst)) //::
        return;

    unsigned old_eip = *context->p_eip;

	ESP_Frame	esp_frame(method_info->num_in_args,method_info->num_vars,
        method_info->num_gc_tag_words,
		                  method_info->num_spills,method_info->num_callee_saved_regs);

	EBP_Frame	ebp_frame(method_info->num_in_args,method_info->num_vars,
        method_info->num_gc_tag_words,
		                  method_info->num_spills,method_info->num_callee_saved_regs);

    Frame *frame;
	//
	// The frame base address is the address in the base register
	// all offset are relative to this base address
	//
	unsigned *base_addr;
	unsigned is_esp_based = method_info->is_esp_based;
	if (is_esp_based) {
		frame = &esp_frame;
		base_addr = (unsigned *)context->esp;
	} else {
		frame = &ebp_frame;
		base_addr = *(unsigned **)context->p_ebp;
	}
	int out_args = compute_call_site_offset(method_info, *context->p_eip, isFirst);
    frame->push(out_args);

    unsigned restore_mask = 0;
    if (method_info->rra.should_restore_callee(method_info,ebx_reg))
    {
        restore_callee(context->p_ebx,"ebx",base_addr,frame->callee_offset(ebx_reg));
        restore_mask |= (1u << ebx_reg);
    }
    if (method_info->rra.should_restore_callee(method_info,ebp_reg))
    {
        restore_callee(context->p_ebp,"ebp",base_addr,frame->callee_offset(ebp_reg));
        restore_mask |= (1u << ebp_reg);
    }
    if (method_info->rra.should_restore_callee(method_info,esi_reg))
    {
        restore_callee(context->p_esi,"esi",base_addr,frame->callee_offset(esi_reg));
        restore_mask |= (1u << esi_reg);
    }
    if (method_info->rra.should_restore_callee(method_info,edi_reg))
    {
        restore_callee(context->p_edi,"edi",base_addr,frame->callee_offset(edi_reg));
        restore_mask |= (1u << edi_reg);
    }


#ifdef TRACE_LEVEL1A
	cout << "UnwindStackFrame_Full context " << context << " methodInfoPtr " << methodInfoPtr
		 << " isFirst " << (int)isFirst << " methodStart " << methodStart << endl;
    cout << "name is " << ((method_info)? method_info->name:"(null)") << endl;

	cout << "name                   = " << method_info->name << endl;
	cout << "num_spills             = " << (void *)method_info->num_spills << endl;
	cout << "num_in_args            = " << (void *)method_info->num_in_args << endl;
	cout << "num_vars               = " << (void *)method_info->num_vars << endl;
	cout << "num_callee_saves       = " << (void *)method_info->num_callee_saved_regs << endl;
	cout << "num_call_sites         = " << (void *)method_info->num_call_sites << endl;
	for (unsigned j = 0; j < method_info->num_call_sites; j++) {
		cout << "call site " << j << " info:" << endl;
		cout << "  ret_IP             = " << (void *)method_info->cs_info[j].ret_IP << endl;
		cout << "  precall_IP         = " << (void *)method_info->cs_info[j].precall_IP << endl;
		cout << "  num_out_args       = " << (void *)method_info->cs_info[j].num_out_args << endl;
	}
	cout << "old pPC          = " << (void *)context->p_eip << endl;
	cout << "old *pPC         = " << (void *)*context->p_eip << endl;
	cout << "old DynaLink     = " << (void *)*context->p_ebp << endl;
	cout << "old StackPointer = " << (void *)context->esp << endl;

#endif // TRACE_LEVEL1A

	// compute frame size
	DWORD frame_size = method_info->frame_size - method_info->num_in_args; // don't pop incoming args
	// add on the number of args currently on the stack
	frame_size += out_args;

#ifdef TRACE_LEVEL1A
	cout << "out_args   = " << out_args << endl;
	cout << "frame size = " << (void *)frame_size << endl << endl;
#endif // TRACE_LEVEL1A

	// unwind stack
	DWORD oldSP = context->esp;
	// set new Esp
	context->esp += frame_size << 2;

	// set new return IP
	context->p_eip = (uint32 *)(context->esp - 4);

#ifdef TRACE_LEVEL1A_1
	cout << "new pPC          = " << (void *)pMC->pPC << endl;
	cout << "new *pPC         = " << (void *)pMC->pPC[0] << endl;
	cout << "new StackPointer = " << (void *)pMC->StackPointer << endl;
#endif

    unwind_add_to_cache(old_eip, out_args, restore_mask);
} //L1a_UnwindStackFrame_Full

unsigned l1a_num_breakpoints(Method_Handle method, uint32 eip)
{
	Jit_Method_Info *method_info = o1_method_get_info_block(method);
    return method_info->num_call_sites;
}

void l1a_get_breakpoints(Method_Handle method, uint32 *bp, Frame_Context *context)
{
	Jit_Method_Info *method_info = o1_method_get_info_block(method);
    for (unsigned i=0; i<method_info->num_call_sites; i++)
    {
        bp[i] = method_info->cs_info[i].precall_IP;
    }
    L1a_UnwindStackFrame_Full(context, method_info, true, NULL);
}

void * l1a_get_address_of_this(Method_Handle meth,
                               Frame_Context *context,
                               Boolean is_first)
{
    int esp_offset;
    if (thisaddr_with_cache(context, is_first, esp_offset))
        return (void *)(context->esp + esp_offset);
    unsigned orig_esp = context->esp;

    // Assume that we can always find the "this" pointer by looking at var0.
	Jit_Method_Info *method_info = o1_method_get_info_block(meth);

    ESP_Frame	esp_frame(method_info->num_in_args,
        method_info->num_vars,
        method_info->num_gc_tag_words,
        method_info->num_spills,
        method_info->num_callee_saved_regs);

	EBP_Frame	ebp_frame(method_info->num_in_args,
        method_info->num_vars,
        method_info->num_gc_tag_words,
        method_info->num_spills,
        method_info->num_callee_saved_regs);

    Frame *frame;

    unsigned *base_addr;
	unsigned is_esp_based = method_info->is_esp_based;
	if (is_esp_based) {
		frame = &esp_frame;
		base_addr = (unsigned*)context->esp;
	} else {
		frame = &ebp_frame;
		base_addr = *(unsigned **)context->p_ebp;
	}
	int out_args = compute_call_site_offset(method_info, *context->p_eip, is_first);
    frame->push(out_args);

    uint32 result = (uint32) base_addr + frame->var_offset(0);
    esp_offset = result - orig_esp;
    thisaddr_add_to_cache(context, is_first, esp_offset);
    return (void *)result;
}

Boolean l1a_call_returns_a_reference(Method_Handle         method,              // in
                                     const Frame_Context  *context              // in
                                     )
{
	Jit_Method_Info *mi = o1_method_get_info_block(method);
	int cur_call_site;
    if (mi->num_call_sites > BINARY_CALL_SITE)
        cur_call_site = binary_search_call_site(mi, *context->p_eip);
    else		
        cur_call_site = linear_search_call_site(mi, *context->p_eip);
    assert(cur_call_site != -1);
    Call_Site_Info *csi = &mi->cs_info[cur_call_site];

    if (csi->m_handle == NULL)
        return csi->returns_ref;

    Java_Type type = method_get_return_type(csi->m_handle);
#if 0 
    //
    // Ken !!
    // the following code is wrong. 
    // method contains the call site pointed by eip.
    // We want to know if the the call site returns a reference.
    // Hence, using method to get the return type to check reference type
    // is not correct.
    //
    Java_Type type = method_get_return_type(method);
#endif
    if (type == JAVA_TYPE_CLASS || type == JAVA_TYPE_ARRAY)
        return TRUE;
    return FALSE;
}

int32 l1a_get_break_point_offset(Compile_Handle compilation,
                                 Method_Handle  meth,
                                 JIT_Flags      flags,
                                 unsigned       bc_location)
{
    Class_Handle c      = method_get_class(meth);
    const Byte  *bc     = method_get_byte_code_addr(meth);
    size_t       size   = method_get_byte_code_size(meth);
    unsigned max_stack  = method_get_max_stack(meth);
    unsigned max_locals = method_get_max_locals(meth);
    Dbg dbg;
    dbg.set_bc_location(bc_location);

    JIT_Result res = L1a_compile(
        compilation,
        c,
        meth,
        bc,
        size,
        max_locals,
        max_stack,
        flags,
        &dbg,
		cm_gen_method);

    assert(dbg.get_code_offset() != DBG_LOC_UNKNOWN);

    return dbg.get_code_offset();
} 

//
// return the address of variable var_no
//
void *l1a_get_address_of_var(Frame_Context *context,
                             Boolean       is_first,
                             unsigned      var_no)
{
    Method_Handle mh = frame_get_method(context);
    //
    // determine if the method uses esp or ebp frame
    //
	Jit_Method_Info *mi = o1_method_get_info_block(mh);

    ESP_Frame	esp_frame(mi->num_in_args,
                          mi->num_vars,
                          mi->num_gc_tag_words,
                          mi->num_spills,
                          mi->num_callee_saved_regs);

	EBP_Frame	ebp_frame(mi->num_in_args,
                          mi->num_vars,
                          mi->num_gc_tag_words,
                          mi->num_spills,
                          mi->num_callee_saved_regs);

    Frame *frame;

    unsigned *base_addr;
	unsigned is_esp_based = mi->is_esp_based;
	if (is_esp_based) 
    {
		frame = &esp_frame;
		base_addr = (unsigned*)context->esp;
	} 
    else 
    {
		frame = &ebp_frame;
		base_addr = *(unsigned **)context->p_ebp;
	}
    //
    // determine how many arguments have been pushed on the stack frame
    //
	int out_args = compute_call_site_offset(mi, *context->p_eip, is_first);
    frame->push(out_args);

    //
    // if the variable is in register, then we return the address of 
    // the register
    //
    unsigned n_locals = mi->num_in_args + mi->num_vars;
    assert(var_no < n_locals);
    X86_Reg_No reg;
    int live;
    mi->rra.get_register_mapping_and_liveness(var_no, NULL, mi, reg, live);
    assert(live);  // O1 JIT always returns true
    switch (reg)
    {
    case esi_reg:
        return context->p_esi;  break;
    case edi_reg:
        return context->p_edi;  break;
    case ebx_reg:
        return context->p_ebx;  break;
    case ebp_reg:
        return context->p_ebp;  break;
    default:
        break;
    }
    //
    // the variable is not in register
    //
    int offset = frame->var_offset(var_no);
    offset = offset >> 2;
    return base_addr + offset;
}

JIT_Result 
l1a_compile_method(Compile_Handle  compilation,              // in
                   Method_Handle   meth,                     // in
                   JIT_Flags       flags,                    // in
                   CODE_MI         code_mi) 
{
    Class_Handle c      = method_get_class(meth);
    const Byte  *bc     = method_get_byte_code_addr(meth);
    size_t       size   = method_get_byte_code_size(meth);
    unsigned max_stack  = method_get_max_stack(meth);
    unsigned max_locals = method_get_max_locals(meth);


    JIT_Result res = L1a_compile(
        compilation,
        c,
        meth,
        bc,
        size,
        max_locals,
        max_stack,
        flags,
        NULL, // not for debugging
        code_mi
		);

    return res;
}
