// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/operand.h,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//



#ifndef _OPERAND_H_
#define _OPERAND_H_

#include <assert.h>
#include "register_manager.h"
#include "data_emitter.h"

class Code_Emitter;
extern X86_Reg_No choose_a_callee_reg(X86_Reg_No r);
extern X86_Reg_No choose_a_callee_reg(X86_Reg_No r1,X86_Reg_No r2);

class Operand {
public:
	enum Kind {Imm,Reg,Var,Field,Static,Array,Stk,Fp};
	const Kind kind;
	Operand(Kind k) : kind(k) {}
	enum Type {T32bit,T64bit_hi,T64bit_lo};
	unsigned is_reg() {return kind == Reg;}
	//
	// emit a mov of this operand into a register
	//
	virtual void emit_mov_to_reg(Code_Emitter& emitter,const R_Opnd *const dst) = 0;
	virtual void emit_push(Code_Emitter& emitter,unsigned n_out_args = 0) = 0;
	virtual void emit_mov_to_mem(Code_Emitter& emitter,const M_Opnd *const dst,
								  X86_Opnd_Size sz=opnd_32) {
		//
		// should never be called for memory operands!
		//
		assert(0);
	}
	virtual void emit_shift(Code_Emitter& emitter,X86_Shift_Opcode opc, 
							 const Operand *const shift) {
		assert(0); // should never be called from imm
	}
	virtual void emit_alu_inst(Code_Emitter& emitter,const R_Opnd *const dst,X86_ALU_Opcode opc) = 0;
	virtual void emit_imul_inst(Code_Emitter& emitter,const R_Opnd *const dst) = 0;
	virtual void emit_idiv_inst(Code_Emitter& emitter) {assert(0);}
	virtual void emit_widen(Code_Emitter& emitter,const R_Opnd *const dst,
		unsigned is_signed,unsigned is_half) {assert(0);}

	virtual void free_opnd(Register_Manager *const reg_manager) {}
	virtual int  is_aliased_across_call() {return 0;}
	virtual int  is_ref_not_killed_by_call() {return 0;}
	virtual int  has_scratch_reg() {return 0;}
	virtual int  hold_local_reg(unsigned local_regs) {return 0;}
	virtual int  is_scratch_reg()  {return 0;}
	virtual int  is_callee_reg()   {return 0;}
	virtual int  is_mem()   {return 1;}
	virtual int  can_trap() {return 0;}
	virtual int  reg_usage(unsigned& usage) {return 0;}
	virtual int  spill32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {return 0;}
	virtual int  spill64_lo(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {return 0;}
	virtual int  spill64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {return 0;}
	virtual void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) = 0;
	virtual void home64_lo(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth){
		home32(emitter,rm,frame,depth);
	}
	virtual void home64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth){
		home32(emitter,rm,frame,depth);
	}
	virtual int  contain(X86_Reg_No r) {return 0;}
	static unsigned estimate_mem_size(unsigned byteCodeSize,unsigned maxStack) {
		return 28 + (8 * byteCodeSize) + (maxStack * 20);
	}
	Type ty;
private:
};

class Imm_Operand : public Operand {
public:
	Imm_Operand(unsigned val) : Operand(Imm), imm_opnd(val) {}
	void emit_mov_to_reg(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_mov(dst,&imm_opnd);
	}
	void emit_mov_to_mem(Code_Emitter& emitter,const M_Opnd *const dst,
						  X86_Opnd_Size sz=opnd_32) {
		emitter.emit_mov(dst,&imm_opnd,sz);
	}
	void emit_alu_inst(Code_Emitter& emitter,const R_Opnd *const dst,X86_ALU_Opcode opc) {
		emitter.emit_alu(opc, dst, &imm_opnd);
	}
	void emit_imul_inst(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_imul(dst,&imm_opnd);
	}
	void emit_push(Code_Emitter& emitter,unsigned n_out_args) {
		emitter.emit_push(&imm_opnd);
	}
	int is_mem() {return 0;}
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {
		emitter.emit_mov(&M_Spill_Opnd(frame,depth),&imm_opnd);
	}
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	Imm_Opnd imm_opnd;
private:
};


class Reg_Operand : public Operand {
public:
	Reg_Operand(X86_Reg_No r) : Operand(Reg), opnd(r), _is_ref(0) {}
	void emit_mov_to_reg(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_mov(dst,&opnd);
	}
	void emit_mov_to_mem(Code_Emitter& emitter,const M_Opnd *const dst,
						  X86_Opnd_Size sz=opnd_32) {
		emitter.emit_mov(dst,&opnd,sz);
	}
	void emit_shift(Code_Emitter& emitter,X86_Shift_Opcode opc,const Operand *const shift) {
		if (shift->kind == Imm) 
			emitter.emit_shift(opc,&opnd,&((Imm_Operand*)shift)->imm_opnd);
		else
			emitter.emit_shift(opc,&opnd);
	}
	void emit_alu_inst(Code_Emitter& emitter,const R_Opnd *const dst,X86_ALU_Opcode opc) {
		emitter.emit_alu(opc, dst, &opnd);
	}
	void emit_imul_inst(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_imul(dst,&opnd);
	}
	void emit_idiv_inst(Code_Emitter& emitter) {
		emitter.emit_div(&opnd,1);
	}
	void emit_widen(Code_Emitter& emitter,const R_Opnd *const dst,
		unsigned is_signed,unsigned is_half) {
		emitter.emit_widen(dst,&opnd,is_signed,is_half);
	}
	void emit_push(Code_Emitter& emitter,unsigned n_out_args) {
		emitter.emit_push(&opnd);
	}
	void free_opnd(Register_Manager *const reg_manager) {
		reg_manager->free_reg(opnd.reg_no());
	}
	int is_aliased_across_call() {
		return is_scratch_x86reg(opnd.reg_no());
	}
	int has_scratch_reg() {return is_scratch_x86reg(opnd.reg_no());}
	int is_scratch_reg()  {return is_scratch_x86reg(opnd.reg_no());}
	int is_callee_reg()   {return is_callee_x86reg(opnd.reg_no());}
	int is_byte_reg()     {
		X86_Reg_No no = opnd.reg_no();
		return is_scratch_x86reg(no) || no == ebx_reg;
	}
	int  hold_local_reg(unsigned local_regs) {return local_regs & (1<<opnd.reg_no());}
	int is_mem()   {return 0;}
	int reg_usage(unsigned& usage) {
		usage |= (1 << opnd.reg_no()); 
		return is_scratch_x86reg(opnd.reg_no());
	}
	int spill32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	int spill64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {
		return spill32(emitter,rm,frame,depth);
	}
	int spill64_lo(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {
		return spill32(emitter,rm,frame,depth);
	}
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {
		emitter.emit_mov(&M_Spill_Opnd(frame,depth),&opnd);
		free_opnd(&rm);
	}
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	int  contain(X86_Reg_No r) {return opnd.reg_no() == r;}
	int is_ref_not_killed_by_call() {return _is_ref;}
	void set_ref()   {_is_ref = 1;}
	void clear_ref() {_is_ref = 0;}

	R_Opnd opnd;
private:
	unsigned short _is_ref;
};
class Mem_Operand : public Operand {
public:
	Mem_Operand(Kind k) : Operand(k) {}

	virtual M_Opnd *mem_opnd() = 0;
	virtual int is_aliased_across_call() {return 1;}
	virtual X86_Reg_No pick_a_callee_reg() {return esi_reg;}
	virtual void emit_push_with_adjust_off(Code_Emitter& emitter,int adjust_off) {
		assert(0);
	}
	void emit_fild(Code_Emitter& emitter,unsigned dbl) {
		emitter.emit_fild(mem_opnd(),dbl);
	}
	void emit_fld(Code_Emitter& emitter,unsigned dbl) {
		emitter.emit_fld(mem_opnd(),dbl);
	}
	void emit_shift(Code_Emitter& emitter,X86_Shift_Opcode opc,const Operand *const shift) {
		if (shift->kind == Imm) 
			emitter.emit_shift(opc,mem_opnd(),&((Imm_Operand*)shift)->imm_opnd);
		else
			emitter.emit_shift(opc,mem_opnd());
	}
	void emit_alu_inst(Code_Emitter& emitter,
						const R_Opnd *const dst,
						X86_ALU_Opcode opc) {
		emitter.emit_alu(opc,dst,mem_opnd());
	}
	void emit_imul_inst(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_imul(dst,mem_opnd());
	}
	void emit_idiv_inst(Code_Emitter& emitter) {
		emitter.emit_div(mem_opnd(),1);
	}
	void emit_widen(Code_Emitter& emitter,const R_Opnd *const dst,
		unsigned is_signed,unsigned is_half) {
		emitter.emit_widen(dst,mem_opnd(),is_signed,is_half);
	}
	void emit_mov_to_reg(Code_Emitter& emitter,const R_Opnd *const dst) {
		emitter.emit_mov(dst,mem_opnd());
	}
	void emit_push(Code_Emitter& emitter,unsigned n_out_args) {
		// if mem operand is not esp based, there is no need to adjust offset
		if (n_out_args && contain(esp_reg)) 
			emit_push_with_adjust_off(emitter,n_out_args << 2);
		else
			emitter.emit_push(mem_opnd());
	}
	void emit_lea(Code_Emitter& emitter,const R_Opnd* const dst) {
		emitter.emit_lea(dst,mem_opnd());
	}
};

// for {put,get}field
class Field_Operand : public Mem_Operand {
public:
    Field_Operand(X86_Reg_No r,unsigned offset) 
        : Mem_Operand(Field), opnd(r,offset), _throw_null_ptr_excp(true) {}
    void free_opnd(Register_Manager *const reg_manager) {
        reg_manager->free_reg(opnd.base_reg);
    }
    X86_Reg_No pick_a_callee_reg() {return choose_a_callee_reg(opnd.base_reg);}
    void emit_push_with_adjust_off(Code_Emitter& emitter,int adjust_off) {
        emitter.emit_push(&M_Base_Opnd(esp_reg,opnd.off() + adjust_off));
    }
    M_Opnd *mem_opnd()	{return &opnd;}
    void *operator new(size_t sz,Mem_Manager& m) {
        return m.alloc(sz);
    }
    int reg_usage(unsigned& usage) {
        usage |= (1 << opnd.base_reg); 
        return is_scratch_x86reg(opnd.base_reg);
    }
    int can_trap() {return 1;}
    int has_scratch_reg() {return is_scratch_x86reg(opnd.base_reg);}
    int hold_local_reg(unsigned local_regs) {return local_regs & (1<<opnd.base_reg);}
    int spill32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
    void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
    void home64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
    int  contain(X86_Reg_No r) {return opnd.base_reg == r;}
    void base_is_not_null() {_throw_null_ptr_excp = false;}
    bool may_throw_null_ptr_excp() {return _throw_null_ptr_excp;}
    
    M_Base_Opnd	opnd;
private:
    bool _throw_null_ptr_excp;
};

//
//  a memory operand with displacement
//
class M_Patch_Opnd : public M_Opnd {
public:
	M_Patch_Opnd(Data_Label *l, Data_Emitter *de) : M_Opnd(0), _label(l), _de(de), _hack(&_label) {}
	M_Patch_Opnd(unsigned i) : M_Opnd(i), _label(NULL), _de(NULL), _hack(&_label) {}
	virtual char *emit(char *inst,unsigned r) const {
        inst = M_Opnd::emit(inst,r);
		if (_label != NULL) {
            if (_label->code != NULL)
            {
                Data_Label *dl = _de->make_label();
                dl->offset = _label->offset;
                *_hack = dl;
            }
            //
			// fill in patch
			//
			_label->code = inst-4;
		}
		return inst;
	}
    Data_Label *label() { return _label; }
protected:
	Data_Label *_label;
    Data_Label **_hack;
    Data_Emitter *_de;
};

//
//  a memory operand with displacement
//
class M_Base_Patch_Opnd : public M_Base_Opnd {
public:
	M_Base_Patch_Opnd(X86_Reg_No r,Data_Label *l) : M_Base_Opnd(r,0x12345678), _label(l) {}
	virtual char *emit(char *inst,unsigned r) const {
		inst = M_Base_Opnd::emit(inst,r);
		if (_label != NULL) {
           assert(_label->code == NULL);
			//
			// fill in patch
			//
			_label->code = inst-4;
		}
		return inst;
	}
protected:
	Data_Label *_label;
};

//
// for {put,get}static, and loads of fp constant
//
class Static_Operand : public Mem_Operand {
public:
	Static_Operand(void *addr,unsigned is_cont) 
		: Mem_Operand(Static), opnd((unsigned)addr), is_const(is_cont) {}
	Static_Operand(Data_Label *label, Data_Emitter *de) 
		: Mem_Operand(Static), opnd(label, de), is_const(1) {}
	M_Opnd *mem_opnd()	{return &opnd;}
	int is_aliased_across_call() {return !is_const;}
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	M_Patch_Opnd opnd;
	const char is_const;
private:
};

class Array_Operand : public Mem_Operand {
public:
	Array_Operand(X86_Reg_No b,X86_Reg_No i,int off, unsigned s) :
	  Mem_Operand(Array), opnd(b,i,off,s) {}
	void free_opnd(Register_Manager *const reg_manager) {
		reg_manager->free_reg(opnd.base_reg);
		reg_manager->free_reg(opnd.index_reg);
	}
	X86_Reg_No pick_a_callee_reg() {
		return choose_a_callee_reg(opnd.base_reg,opnd.index_reg);
	}
	int has_scratch_reg() {
		return is_scratch_x86reg(opnd.base_reg) || is_scratch_x86reg(opnd.index_reg);
	}
	int hold_local_reg(unsigned local_regs) {
		return (local_regs & (1<<opnd.base_reg)) | (local_regs & (1<<opnd.index_reg));
	}
	int spill32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	int spill64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	int spill64_lo(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	void home64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	int  can_trap() {return 1;}
	int reg_usage(unsigned& usage) {
		usage |= ((1 << opnd.base_reg) | (1 << opnd.index_reg));
		return is_scratch_x86reg(opnd.base_reg) + is_scratch_x86reg(opnd.index_reg);
	}
	int  contain(X86_Reg_No r) {return (opnd.base_reg == r) || (opnd.index_reg == r);}
	M_Opnd *mem_opnd()	{return &opnd;}
	M_Index_Opnd opnd;
private:
};

class Mem_Var_Operand : public Mem_Operand {
public:
	Mem_Var_Operand(Frame& frame,unsigned v,unsigned ref) 
		: Mem_Operand(Var), opnd(frame,v), var_no(v), is_ref(ref) {}

	void emit_push_with_adjust_off(Code_Emitter& emitter,int adjust_off) {
		emitter.emit_push(&M_Base_Opnd(esp_reg,opnd.off() + adjust_off));
	}
	M_Opnd *mem_opnd()	{return &opnd;}
	int is_aliased_across_call() {return is_ref;}
	int is_ref_not_killed_by_call() {return is_ref;}
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth);
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	int reg_usage(unsigned& usage) {
		usage |= (1 << opnd.base_reg); 
		return is_scratch_x86reg(opnd.base_reg);
	}
	int  contain(X86_Reg_No r) {return opnd.base_reg == r;}
	const unsigned short var_no;
	const unsigned short is_ref;
	M_Var_Opnd opnd;
private:
};

class Stack_Operand : public Mem_Operand {
public:
	Stack_Operand(Frame& f,unsigned d) 
		: Mem_Operand(Stk), depth(d),opnd(f,d) {}

	void emit_push_with_adjust_off(Code_Emitter& emitter,int adjust_off) {
		emitter.emit_push(&M_Base_Opnd(esp_reg,opnd.off() + adjust_off));
	}
	M_Opnd *mem_opnd()	{return &opnd;}
	int is_aliased_across_call() {return 0;}
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {}
	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}
	int reg_usage(unsigned& usage) {
		usage |= (1 << opnd.base_reg); 
		return 0; // esp is not a scratch reg
	}
	int  contain(X86_Reg_No r) {return opnd.base_reg == r;}
	const unsigned depth;
	M_Spill_Opnd	opnd;
};

class Fp_Operand : public Operand {
public:
	Fp_Operand(unsigned d) : Operand(Fp), is_double(d) { 
            fpstack_cnt = 0;
        }
	int is_mem()  {return 0;}      

	void emit_mov_to_mem(Code_Emitter& emitter,const M_Opnd *const dst,
							  X86_Opnd_Size sz=opnd_32) {
	     fpstack_cnt = 0;
             if (is_double) {
                 if (ty == T64bit_lo) {
	             emitter.emit_fst(dst,is_double,1);
                 }
             }
             else {
	         emitter.emit_fst(dst,is_double,1);
             }
	}

	void emit_mov_to_reg(Code_Emitter& emitter,const R_Opnd *const dst) {
	    //
	    // should never be called for floating point operands!
	    //
	    assert(0);
        }

	void emit_push(Code_Emitter& emitter,unsigned n_out_args = 0) {
            fpstack_cnt = 0;
	    if (is_double) {
	        if (ty == T64bit_hi) {
	            emitter.emit_alu(sub_opc, &esp_opnd,&Imm_Opnd(8));
                }
            }
	    else {
                emitter.emit_alu(sub_opc, &esp_opnd,&Imm_Opnd(4));
            }
	    emitter.emit_fst(&M_Base_Opnd(esp_reg,0),is_double,1);
        }
	void emit_alu_inst(Code_Emitter& emitter,const R_Opnd *const dst,X86_ALU_Opcode opc) {
            assert(0);
        }
	void emit_imul_inst(Code_Emitter& emitter,const R_Opnd *const dst) {
            assert(0);

        }
	void home32(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth) {
	     fpstack_cnt = 0;
	     emitter.emit_fst(&M_Spill_Opnd(frame,depth),0,1);
        }

	void home64_lo(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth){
	        fpstack_cnt = 0;
		emitter.emit_fst(&M_Spill_Opnd(frame,depth),1,1);
	}

	void home64_hi(Code_Emitter& emitter,Register_Manager& rm,Frame& frame,unsigned depth){
		return;
	}

	void *operator new(size_t sz,Mem_Manager& m) {
		return m.alloc(sz);
	}

	int  is_aliased_across_call() {return 1;}
        const unsigned is_double;
	unsigned fpstack_cnt;
};

class Pre_Alloc_Operand_Pool {
public:
	Pre_Alloc_Operand_Pool(Frame& f,unsigned sz,Mem_Manager& m) : _size(sz) {
		// allocate stack operands because we know the maximum stack depth
		_s_operands = (Stack_Operand**)m.alloc(sizeof(Stack_Operand*)*sz);
		for (unsigned i = 0; i < sz; i++) 
			_s_operands[i] = new(m) Stack_Operand(f,i);

		// allocate array for imm operands
		_imms = (Imm_Operand**)m.alloc(sizeof(Imm_Operand*)*7);
		_imms++; // offset to zero
		for (int j = -1; j < 6; j++)
			_imms[j] = new(m) Imm_Operand(j);
	}
	~Pre_Alloc_Operand_Pool() {}
	Stack_Operand *nth_stack(unsigned i) {
		assert(i < _size);
		return _s_operands[i];
	}
	//  must be within the range -2 < i < 6
	Imm_Operand *imm(int i) {return _imms[i];}
private:
	// small imm constants
	Stack_Operand **_s_operands;
	Imm_Operand **_imms;
	unsigned _size;
};

#endif // _OPERAND_H_
