#include "assert.h"
#include "stdio.h"
#include "stdlib.h"
#include "string.h"

#include "platform.h"

//#ifdef INLINE_NATIVE
#include "ir.h"
#include "expression.h"
#include "nativeparser.h"


const NativeInfo::Info NativeInfo::_info[NativeInfo::n_native] = {
//	 "Name",	Inst,   n_opnd, n_target, n_dest    
	{"Null",	Null,	0,		0,		0},
    {"Checkcast", Checkcast, 2, 0,      1},     //checkcast
	{"Mov",		Mov,	1,		0,		1},		//Assign_Inst
    {"Add",		Add,	2,		0,		1},		//Add_Inst
	{"Adc",		Adc,	2,		0,		1},
	{"Mul",		Mul,	2,		0,		1},		//Mul_Inst
	{"Smul",	Smul,	2,		0,		1},
	{"Cmp",		Cmp,	2,		0,		1},		//Compare_Inst
	{"Test",	Test,	1,		0,		1},
	{"Sub",		Sub,	2,		0,		1},		//Sub_Inst
	{"Sbb",		Sbb,	2,		0,		1},
	{"Shld",	Shld,	2,		0,		0},		//Shld, Shrd
	{"Shrd",	Shrd,	2,		0,		0},
	{"Shl",		Shl,	2,		0,		1},		//Bitwise_Inst
	{"Shr",		Shr,	2,		0,		1},		
	{"Sar",		Sar,	2,		0,		1},
	{"Xor",		Xor,	2,		0,		1},
	{"And",		And,	2,		0,		1},
	{"Or",		Or,		2,		0,		1},
	{"Hi",		Hi,		1,		0,		1},		//Hi, Lo
	{"Lo",		Lo,		1,		0,		1},
	{"Div",		Div,	2,		0,		1},		//Div_Inst
	{"Jp",		Jp,     0,		1,		0},		//Jm
	{"Beq",		Beq,    0,		1,		0},		//Branch_Inst
	{"Bne",		Bne,    0,		1,		0},
	{"Blt",		Blt,    0,		1,		0},
	{"Bge",		Bge,    0,		1,		0},
	{"Bgt",		Bgt,    0,		1,		0},
	{"Ble",		Ble,    0,		1,		0},
	{"Convt",	Convt,	1,		0,		1}		//Conver_Inst
};

void NativeInstParser::get_one_opnd(char*& curr, NativeOpndInfo* opnd_info)
{
	assert( *curr);
	skip_blanks(curr);
	char c = *curr;
	char buf[MAX_NATIVE_OPCODE_LENGTH];
	if (*curr >= '0' && *curr <= '9'){
		copy_string(curr, buf);
		opnd_info->tag = NativeOpndInfo::imm;
		opnd_info->val = atoi(buf);
	}
	else{
		switch	(c){
		case 't':				// temp argument
			skip_char(curr, 't');
			copy_string(curr, buf);
			opnd_info->tag = NativeOpndInfo::temp;
			opnd_info->num = atoi(buf);
			break;
		case '$':				// in argument
			skip_char(curr, '$');
			copy_string(curr, buf);
			opnd_info->tag = NativeOpndInfo::in_arg;
			opnd_info->num = atoi(buf);
			break;
		case '#':				// out argument
			skip_char(curr, '#');
			copy_string(curr, buf);
			opnd_info->tag = NativeOpndInfo::out_arg;
			opnd_info->num = atoi(buf);
			break;
		case '[':{				// filed operand sign
			skip_char(curr, '[');
			copy_string(curr, buf);
			unsigned field = 0, offset = 0;
			get_field_offset(buf, &field, &offset);

			opnd_info->tag = NativeOpndInfo::field;
			opnd_info->num = field;
			opnd_info->off = offset;
			break;
				 }
		case '@':				// src operand
			skip_char(curr, '@');
			copy_string(curr, buf);
			opnd_info->tag = NativeOpndInfo::src_arg;
			opnd_info->num = atoi(buf);
			break;
		default:
			assert(0);
		}
	}

	// decide the operand is hi_opnd, lo_opnd or the whole opnd
	// "opnd.Hi" is hi_opnd
	// "opnd.Lo" is lo_opnd
	// "opnd" is whole opnd
	if (*curr++ == '.'){ // 
		copy_string(curr, buf);
		if (strcmp(buf, "Hi") == 0) 
			opnd_info->hi_tag = NativeOpndInfo::hi;
		else if (strcmp(buf, "Lo") == 0)
			opnd_info->hi_tag = NativeOpndInfo::lo;
	}
	else
		opnd_info->hi_tag = NativeOpndInfo::whole;
}

void NativeInstParser::get_field_offset(char* buf, unsigned* field, unsigned* offset)
{
	replace_char(buf, ']', ' ');			// "[base+off]" ==> "base+off"

	int i = replace_char(buf, '+', ' ');	// "base+off" ==> "base off"
	char buf_field[MAX_NATIVE_LABEL_LENGTH];
	char *p = buf_field;
	char buf_offset[MAX_NATIVE_LABEL_LENGTH];

	// get base opnd sign
	copy_string(buf, buf_field);			// buf <== base
	assert(*p == 't');
	skip_char(p, 't');
	*field = atoi(p);
	
	if ( i > 0) {		// has offset
		skip_blanks(buf);
		copy_string(buf, buf_offset);		// buf <== off
		*offset = atoi(buf_offset);
	}
}

int NativeInstParser::replace_char(char* buf, char oldc, char newc)
{
	char* c = strchr( buf, oldc);
	if ( c ) { 
		buf[c - buf] = newc;
		return c - buf;
	}
	else 
		return -1;
}

void NativeInstParser::parse_label(char*& curr, NativeInfo* info) 
{
	char label[MAX_NATIVE_LABEL_LENGTH];

	skip_blanks(curr);
	if ( *curr == '.'){					// has label
		skip_char(curr, '.');
		copy_string(curr, label);		
		info->set_label(label);
	}
	else								// has no label
		info->set_label("");
}

NativeInfo::Opcode NativeInstParser::get_opcode(char* curr) 
{
	for (int i = 0; i < NativeInfo::n_native; i++){
		if ( strcmp( curr, NativeInfo::_info[i].name) == 0)
			return NativeInfo::_info[i].opcode;
	}
	assert(0);
	return NativeInfo::n_native;
}

// parse op_code
void NativeInstParser::parse_opcode(char*& curr, NativeInfo* info) 
{
	char opcodestr[MAX_NATIVE_OPCODE_LENGTH];
	skip_blanks(curr);
	if (*curr){
		copy_string(curr, opcodestr);
		info->set_opcode( get_opcode(opcodestr));
	}
	else
		info->set_opcode( NativeInfo::Null);
}

// pares type
void NativeInstParser::parse_type(char*& curr, NativeInfo* info) 
{
	if (*curr){
		assert(*curr == '.');
		skip_char(curr, '.');
		info->set_type(*curr);
		skip_char(curr, *curr);
	}
}

// parse target
void NativeInstParser::parse_target(char*& curr, NativeInfo* info)
{
	char target[MAX_NATIVE_LABEL_LENGTH];
	if(*curr && info->has_target()){
			skip_blanks(curr);
			copy_string(curr, target);
			info->set_target(target);
	}
}

// parse dest opnds
void NativeInstParser::parse_dst(char*& curr, NativeInfo* info)
{
	NativeOpndInfo temp_opnd;
	if(*curr && info->has_dst()){
		skip_blanks(curr);
		assert( *curr == 't' || *curr == '[' || *curr == '$' || *curr == '#' || *curr == '@');
		get_one_opnd(curr, &temp_opnd);
		info->set_opnd_info(temp_opnd, 0);
	}
}

// parse src opnds
void NativeInstParser::parse_srcs(char*& curr, NativeInfo* info)
{
	if (*curr){
		for (unsigned i = 0; i < info->src_opnd_num(); i++){
			NativeOpndInfo opnd_info;
			skip_blanks(curr);
			assert( *curr);
			get_one_opnd(curr, &opnd_info);
			info->set_opnd_info(opnd_info, i+1);
		}
	}
}

void NativeInstParser::parse_inst(char* inst_str, NativeInfo* info)
{
	char* curr = inst_str;
	parse_label(curr,info);
	parse_opcode(curr,info);
	parse_type(curr,info);
	parse_target(curr, info);
	parse_dst(curr,info);
	parse_srcs(curr,info);
};

//#endif
