-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002,2003  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.Command_Line;
with Ada.Exceptions;
with Ada.Text_IO;
with Configuration;
with Echo;
with Externals;
with Externals.Simple;
with Invocation;
with Globals;               use Globals;
with Help;
with Menus;
with Misc;                  use Misc;
with Receiving;
with Sending;
with Workaround;

procedure Topal is
   use Invocation;
begin
   -- Get our PID.
   Our_PID := Externals.C_Get_Process_ID;
   -- Set defaults.
   Topal_Directory := ToUBS(Externals.Get_Env("HOME")
                            & "/.topal");
   Configuration.Default_Configuration(Config);
   -- .topal had better exist if we are going to read the configuration file.
   if Externals.Simple.Test_R(ToStr(Topal_Directory) & "/config") then
      Configuration.Read_Config_File;
   end if;
   -- If the topal directory exists, we chmod it appropriately.
   if Externals.Simple.Test_D(ToStr(Topal_Directory)) then
      Externals.Simple.Chmod("700", ToStr(Topal_Directory));
      -- And do the same to the cache directory.
      if Externals.Simple.Test_D(ToStr(Topal_Directory) & "/cache") then
         Externals.Simple.Chmod("700", ToStr(Topal_Directory) & "/cache");
      end if;
   end if;
   -- Sort out the command-line.
   Parse_Command_Line;
   case Run_Mode is
      when Help_Text =>
         Disclaimer;
         Help.Message;
      when Inline_Display =>
         Externals.Simple.Clear;
         Disclaimer;
         Open_Result_File(Resultfile => ToStr(Resultfile));
         Receiving.Display(Tmpfile => ToStr(Tmpfile));
      when Mime_Display =>
         Disclaimer;
         Receiving.Mime_Display(Infile       => ToStr(Infile),
                                Content_Type => ToStr(Content_Type));
      when Old_Mime_Display =>
         Disclaimer;
         Receiving.Mime_Display_APGP(Infile       => ToStr(Infile),
                                     Content_Type => ToStr(Content_Type));
      when Inline_Send =>
         Externals.Simple.Clear;
         Disclaimer;
         Open_Result_File(Resultfile => ToStr(Resultfile));
         Sending.Send(Tmpfile    => ToStr(Tmpfile),
                      Recipients => Recipients.all);
      when Mime_Send =>
         Externals.Simple.Clear;
         Disclaimer;
         Open_Result_File(Resultfile => ToStr(Resultfile));
         Sending.Send(Tmpfile    => ToStr(Tmpfile),
                      Mime       => True,
                      Mimefile   => ToStr(Mimefile),
                      Recipients => Recipients.all);
      when Nonpine_Send =>
         Disclaimer;
         -- Does the first object exist as a readable file?
         if not Externals.Simple.Test_R(ToStr(Tmpfile)) then
            -- We should be able to guarantee that Tmpfile is at least one
            -- character long, otherwise it couldn't have been recognised
            -- as an argument on the command line.
            if ToStr(Tmpfile)(1) = '-' then
               -- Perhaps they thought it was a command line option.
               Ada.Text_IO.Put_Line("`" & ToStr(Tmpfile) & "' is not recognised as either a command-line option nor a readable file.");
               Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
               -- Fall out normally.
            else
               Error("File `" & ToStr(Tmpfile) & "' is not a readable file.");
            end if;
         else
            -- The file exists....
            if ToStr(Tmpfile)(1) = '-' then
               -- Perhaps they thought it was a command line option.
               Ada.Text_IO.Put_Line("`" & ToStr(Tmpfile) & "' is not recognised as a command-line option, but it is a readable file.  Proceeding anyway -- select `abort' if that's not what you intended.");
               Ada.Text_IO.New_Line;
            end if;
            Sending.Send(Tmpfile    => ToStr(Tmpfile),
                         Recipients => Recipients.all,
                         Non_Pine   => True);
         end if;
      when Clear_Temp =>
         Disclaimer;
         Ada.Text_IO.Put_Line("Clearing temporary files...");
         Externals.Simple.Rm_Tempfiles;
      when Clear_Cache =>
         Disclaimer;
         Ada.Text_IO.Put_Line("Clearing cache files...");
         Externals.Simple.Rm_Cachefiles;
      when Clear_All =>
         Disclaimer;
         Ada.Text_IO.Put_Line("Clearing temporary and cache files...");
         Externals.Simple.Rm_Tempfiles;
         Externals.Simple.Rm_Cachefiles;
      when Dump_Default_Config =>
         Configuration.Default_Configuration(Config);
         Configuration.Dump;
      when Dump_Current_Config =>
         Configuration.Dump;
      when Interactive_Config =>
         Disclaimer;
         -- Interactive configuration setting.
         Configuration.Edit_Configuration;
      when Fix_Email =>
         Workaround.Fix_Email;
      when Fix_Folders =>
         Disclaimer;
         Workaround.Fix_Folders(Folders => Folders.all);
   end case;
   if not Config.No_Clean then
      Externals.Simple.Rm_Tempfiles_PID;
   end if;
   if Run_Mode /= Fix_Email then
      -- Only do this with a real console.  This is nasty.
      Echo.Set_Echo;
      Externals.Simple.Stty_Sane;
   end if;
   Close_Result_File;
exception
   when Externals.System_Abort_By_User =>
      -- Just clean up.
      Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
      Echo.Set_Echo;
      Externals.Simple.Stty_Sane;
   when The_Exception : others =>
      -- Dump info.
      if Ada.Text_IO.Is_Open(Result_File) then
         Ada.Text_IO.Put_Line(Result_File,
                              "Exception raised: "
                              & Ada.Exceptions.Exception_Name(The_Exception));
         Ada.Text_IO.Close(Result_File);
      end if;
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "Exception raised: "
                           & Ada.Exceptions.Exception_Name(The_Exception));
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           Ada.Exceptions.Exception_Information(The_Exception));
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "This is not good: please send a bug report with the information above to");
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "pjb@lothlann.freeserve.co.uk (unless it was caused by ctrl-c when GPG");
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "was running).  Thank you!");
      Ada.Command_Line.Set_Exit_Status(Ada.Command_Line.Failure);
      -- Try to pause....
      Menus.Wait;
      -- Clean up terminal settings.
      Echo.Set_Echo;
      Externals.Simple.Stty_Sane;
end Topal;
