/*
Copyright (c) 1998, 1999 Wabasoft  All rights reserved.

This software is furnished under a license and may be used only in accordance
with the terms of that license. This software and documentation, and its
copyrights are owned by Wabasoft and are protected by copyright law.

THIS SOFTWARE AND REFERENCE MATERIALS ARE PROVIDED "AS IS" WITHOUT WARRANTY
AS TO THEIR PERFORMANCE, MERCHANTABILITY, FITNESS FOR ANY PARTICULAR PURPOSE,
OR AGAINST INFRINGEMENT. WABASOFT ASSUMES NO RESPONSIBILITY FOR THE USE OR
INABILITY TO USE THIS SOFTWARE. WABASOFT SHALL NOT BE LIABLE FOR INDIRECT,
SPECIAL OR CONSEQUENTIAL DAMAGES RESULTING FROM THE USE OF THIS PRODUCT.

WABASOFT SHALL HAVE NO LIABILITY OR RESPONSIBILITY FOR SOFTWARE ALTERED,
MODIFIED, OR CONVERTED BY YOU OR A THIRD PARTY, DAMAGES RESULTING FROM
ACCIDENT, ABUSE OR MISAPPLICATION, OR FOR PROBLEMS DUE TO THE MALFUNCTION OF
YOUR EQUIPMENT OR SOFTWARE NOT SUPPLIED BY WABASOFT.
*/

// NOTE: To build, define either WIN32 or UNIX by uncommenting one
// of the following lines (best: add in the Makefile)

//#define WIN32 1
//#define UNIX 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef WIN32
#include <windows.h>
#define unlink(path) _unlink(path)
#define WRITE_FLAGS "wb"
#define READ_FLAGS "rb"
#endif

#ifdef UNIX
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef __FreeBSD__
#include <sys/syslimits.h>
#endif

/* Comment of NAME_MAX from /usr/include/limits.h of Solaris 7 :
 * -------------------------------------------------------------
 *
 * POSIX 1003.1a, section 2.9.5, table 2-5 contains [NAME_MAX] and the
 * related text states:
 *
 * A definition of one of the values from Table 2-5 shall be omitted from the
 * <limits.h> on specific implementations where the corresponding value is
 * equal to or greater than the stated minimum, but where the value can vary
 * depending on the file to which it is applied. The actual value supported for
 * a specific pathname shall be provided by the pathconf() (5.7.1) function.
 *
 * This is clear that any machine supporting multiple file system types
 * and/or a network can not include this define, regardless of protection
 * by the _POSIX_SOURCE and _POSIX_C_SOURCE flags.
 *
 * #define      NAME_MAX        14
 */

/* any way, as we need NAME_MAX, we define it if it's not allready done */
#ifndef NAME_MAX
#define NAME_MAX 512
#endif

#define MAX_PATH NAME_MAX
#define WRITE_FLAGS "w"
#define READ_FLAGS "r"
#endif

#define CMD_CREATE 1
#define CMD_LIST 2

#define MAX_WARP_FILES 1024

static void addInputFile(char *path);
static void addInputDir(char *path);
static void genName(char *warpFile, char *name);
static void genCreator(char *name, char *creator);
static void copyright();
static void usage();
static char *convertPathSlashes();
static int pathCompare(const void *arg1, const void *arg2);
static void findDirPath(char *path, char *dirpath);
static void PdbCreate();
static void PdbList();
static void WrpCreate();
static void WrpList();

typedef struct
	{
	char *path;
	int size;
	} InputFile;

typedef struct
	{
	char name[32];
	char creator[5];
	int version;
	int quiet;
	char *warpFile;
	InputFile inputFiles[MAX_WARP_FILES];
	int numInputFiles;
	} Options;

Options options;

int main(int argc, char *argv[])
	{
	char *arg, **files;
	char *creatorOverride;
	int i, n, numFiles, cmd;

	if (argc < 2)
		usage();

	// sanity check
	if (sizeof(int) != 4)
		{
		printf("ERROR: int bad size\n");
		exit(-1);
		}

	cmd = 0;
	switch (argv[1][0])
		{
		case 'c':
		case 'C':
			cmd = CMD_CREATE;
			break;
		case 'l':
		case 'L':
			cmd = CMD_LIST;
			break;
		default:
			printf("ERROR: no command specified\n");
			exit(-1);
		}

	// parse command line options
	options.quiet = 0;
	creatorOverride = NULL;
	for (i = 2; i < argc; i++)
		{
		arg = argv[i];
		if (arg[0] != '/' || strlen(arg) < 2)
			break;
		switch(arg[1])
			{
			case '?':
				usage();
				break;
			case 'c':
			case 'C':
				if (++i == argc)
					{
					printf("ERROR: no creator specified\n");
					exit(-1);
					}
				arg = argv[i];
				if (strlen(arg) != 4)
					{
					printf("ERROR: creator must be 4 charaters\n");
					exit(-1);
					}
				creatorOverride = arg;
				break;
			case 'q':
			case 'Q':
				options.quiet = 1;
				break;
			default:
				printf("ERROR: unknown option %s\n", argv[i]);
				exit(-1);
			}
		}
	if (i == argc)
		{
		printf("ERROR: no warp file specified\n");
		exit(-1);
		}
	options.warpFile = argv[i];
	n = strlen(options.warpFile);
	if (cmd == CMD_CREATE && n > 4 && options.warpFile[n - 4] == '.')
		{
		printf("ERROR: when creating, don't specify an extension such as %s\n",
			&options.warpFile[n - 4]);
		exit(-1);
		}
	i++;
	options.version = 1;
	genName(options.warpFile, options.name);
	genCreator(options.name, options.creator);

	// override generated creator if specified
	if (creatorOverride)
		strcpy(options.creator, creatorOverride);

	// generate input file list - expand wildcards and get full paths
	numFiles = argc - i;
	files = &argv[i];
	options.numInputFiles = 0;
	for (i = 0; i < numFiles; i++)
		addInputFile(files[i]);
	if (cmd == CMD_CREATE)
		{
		if (numFiles == 0)
			{
			printf("ERROR: no input files specified\n");
			exit(-1);
			}
		if (!options.quiet)
			{
			copyright();
			printf("warp files: %s.pdb %s.wrp\n", options.warpFile, options.warpFile); 
			printf("PalmOS PDB name: %s\n", options.name);
			printf("PalmOS PDB creator: %s\n", options.creator); 
			printf("PalmOS PDB version: %d\n", options.version);
			}
		// sort path names
		qsort(options.inputFiles, options.numInputFiles, sizeof(InputFile), pathCompare);
		PdbCreate();
		WrpCreate();
		}
	else if (cmd == CMD_LIST)
		{
		int pathLen;

		pathLen = strlen(options.warpFile);
		if (pathLen > 4 && !strncmp(&options.warpFile[pathLen - 4], ".pdb", 4))
			PdbList(options.warpFile);
		else if (pathLen > 4 && !strncmp(&options.warpFile[pathLen - 4], ".wrp", 4))
			WrpList(options.warpFile);
		else
			{
			printf("ERROR: file does not have a warp file extension\n");
			exit(-1);
			}
		}
	return 0;
	}

#ifdef WIN32
static void findDirPath(char *path, char *dirpath)
	{
	int i, len;

	// The Win32 FindNextFile() function returns only the filename and
	// not the full path to the file. There doesn't appear to be a way
	// to get the full path out of the find file functions so we need to
	// parse out the directory we are looking in here.
	strcpy(dirpath, path);
	len = strlen(dirpath);
	for (i = len - 1; i >= 0; i--)
		if (dirpath[i] == '/' || dirpath[i] == '\\')
			{
			dirpath[i + 1] = 0;
			return;
			}
	dirpath[0] = 0;
	return;
	}

static void addInputFile(char *path)
	{
	WIN32_FIND_DATA fileData;
	HANDLE h;
	char fullpath[MAX_PATH], dirpath[MAX_PATH];
	int n, done;

	findDirPath(path, dirpath);
	h = FindFirstFile(path, &fileData);
	if (h == INVALID_HANDLE_VALUE)
		{
		printf("ERROR: could not find file %s\n", path);
		exit(-1);
		}
	done = 0;
	while (!done)
		{
		if (fileData.cFileName[0] == '.')
			;
		else if (fileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
			sprintf(fullpath, "%s%s\\*", dirpath, fileData.cFileName);
			addInputFile(fullpath);
			}
		else
			{
			sprintf(fullpath, "%s%s", dirpath, fileData.cFileName);
			n = options.numInputFiles;
			if (n == MAX_WARP_FILES)
				{
				printf("ERROR: too many files\n");
				exit(-1);
				}
			options.inputFiles[n].path = convertPathSlashes(fullpath);
			options.inputFiles[n].size = fileData.nFileSizeLow;
			options.numInputFiles++;
			}
		if (FindNextFile(h, &fileData) == 0)
			done = 1;
		}
	FindClose(h);
	}
#endif

#ifdef UNIX
static void addInputFile(char *path)
	{
	struct stat entstat;
	int n;

	if (stat(path, &entstat) != 0)
		{
		printf("ERROR: can't stat file %s\n", path);
		exit(-1);
		}
	if (S_ISDIR(entstat.st_mode))
		{
		addInputDir(path);
		return;
		}
	n = options.numInputFiles;
	if (n == MAX_WARP_FILES)
		{
		printf("ERROR: too many files\n");
		exit(-1);
		}
	options.inputFiles[n].path = convertPathSlashes(path);
	options.inputFiles[n].size = entstat.st_size;
	options.numInputFiles++;
	}

static void addInputDir(char *path)
	{
	DIR *dir;
	struct dirent *ent;
	char dirpath[MAX_PATH], fullpath[MAX_PATH];
	int n;

	dir = opendir(path);
	if (dir == NULL)
		{
		printf("ERROR: could not open directoy %s\n", path);
		exit(-1);
		}
	while (ent = readdir(dir))
		{
		if (ent->d_name[0] == '.')
			continue;
		sprintf(fullpath, "%s/%s", path, ent->d_name);
		addInputFile(fullpath);
		}
	closedir(dir);
	}
#endif

//
// Utility Methods
//

static void genName(char *warpFile, char *name)
	{
	int i, n;

	// get the tail end of the warp file to get its name
	i = strlen(warpFile) - 1;
	while (i > 0 && warpFile[i] != '/' && warpFile[i] != '\\')
		i--;
	if (i)
		i++; // skip slash
	if (warpFile[i] == '"')
		i++; // skip start quote
	n = strlen(warpFile) - i;
	if (warpFile[strlen(warpFile) - 1] == '"')
		n--; // skip end quote
	if (!n || n > 30)
		{
		printf("ERROR: warp file name must be less than 31 characters\n");
		exit(-1);
		}
	strncpy(name, &warpFile[i], n);
	name[n] = 0;
	}

static void genCreator(char *name, char *creator)
	{
	int i, n, hash;

	n = strlen(name);
	hash = 0;
	for (i = 0; i < n; i++)
		hash += name[i];
	for (i = 0; i < 4; i++)
		{
		creator[i] = (hash % 26) + 'a';
		if (hash & 64)
			creator[i] += ('A' - 'a');
		hash = hash / 2;
		}
	creator[4] = 0;
	}

static void copyright()
	{
	printf("Wabasoft (TM) Application Resource Packager   Version 1.40.0\n");
	printf("Copyright (C) Wabasoft 1998, 1999. All rights reserved\n");
	printf("\n");
	}

static void usage()
	{
	copyright();
	printf("Usage: warp command [options] warpfile [files]\n");
	printf("\n");
	printf("Commands:\n");
	printf("   c   Create new warp file\n");
	printf("   l   List contents of a warp file\n");
	printf("\n");
	printf("Options:\n");
	printf("  /?   Displays usage text\n");
	printf("  /c   Override and assign PDB database creator (e.g. /c CrTr)\n");
	printf("  /q   Quiet mode (no output except for errors)\n");
	printf("\n");
	printf("This program creates both a WindowsCE .wrp warp file and a PalmOS .pdb\n");
	printf("warp file. For PalmOS, a PDB database name and PDB creator will be\n");
	printf("generated automatically from the name of the warp file.\n");
	printf("\n");
	printf("Examples:\n");
	printf("   warp c helloApp *.class util\\*.class\n");
	printf("   warp l helloApp.wrp\n");
	printf("   warp l helloApp.pdb\n");
	exit(0);
	}

static char *convertPathSlashes(char *path)
	{
	int i, pathLen;
	char *slashPath;

	// make path name with forward slashes instead of backwards
	pathLen = strlen(path);
	slashPath = malloc(pathLen + 1);
	strcpy(slashPath, path);
	for (i = 0; i < pathLen; i++)
		if (slashPath[i] == '\\')
			slashPath[i] = '/';
	return slashPath;
	}

static int pathCompare(const void *arg1, const void *arg2)
	{
	const InputFile *file1, *file2;
	unsigned char *s1, *s2;
	unsigned int i, len1, len2, minLen;

	// NOTE: here we do our own ISO strcmp() instead of calling the
	// library function strcmp(). We do this because the implementation of
	// strcmp() varies between systems and we need a string compare that
	// gives the same results across platforms.
	file1 = (const InputFile *)arg1;
	file2 = (const InputFile *)arg2;
	s1 = file1->path;
	s2 = file2->path;
	len1 = strlen(s1);
	len2 = strlen(s2);
	if (len1 > len2)
		minLen = len2;
	else
		minLen = len1;
	for (i = 0; i < minLen; i++)
		{
		if (s1[i] == s2[i])
			;
		else if (s1[i] < s2[i])
			return -1;
		else
			return 1;
		}
	if (len1 < len2)
		return -1;
	else if (len1 > len2)
		return 1;
	return 0;
	}

static unsigned char *readFileIntoMemory(char *path)
	{
	FILE *file;
	char *buf;
	int size;

	file = fopen(path, READ_FLAGS);
	if (!file)
		return NULL;
	fseek(file, 0L, SEEK_END); 
	size = (int)ftell(file);
	buf = (char *)malloc(size);
	if (buf)
		{
		fseek(file, 0L, SEEK_SET);
		if (fread(buf, 1, size, file) != (size_t)size)
			{
			free(buf);
			buf = NULL;
			}
		}
	fclose(file);
	return buf;
	}

//
// PDB FILE
//

typedef struct
	{
	FILE *file;
	char path[MAX_PATH];
	int recOffset;
	} Pdb;

Pdb pdb;

static void PdbExitError(int num);
static void PdbWriteHeader();
static void PdbWriteOffset();
static void PdbWriteRecord(char *path, int size);

static void PdbCreate()
	{
	int i;

	sprintf(pdb.path, "%s.pdb", options.warpFile);
	if (!options.quiet)
		printf("...writing %s\n", pdb.path);
	pdb.file = fopen(pdb.path, WRITE_FLAGS);
	if (!pdb.file)
		{
		printf("ERROR: can't create file %s\n", pdb.path);
		PdbExitError(-1);
		}

	PdbWriteHeader();

	pdb.recOffset = 78 + (options.numInputFiles * 8) + 2;
	for (i = 0; i < options.numInputFiles; i++)
		{
		int pathLen, size;

		PdbWriteOffset();
		pathLen = strlen(options.inputFiles[i].path);
		size = options.inputFiles[i].size;
		pdb.recOffset += 2 + pathLen + size;
		}

	// write 2 byte filler
		{
		unsigned char filler[2];

		filler[0] = 0x00;
		filler[1] = 0x00;
		fwrite(filler, 1, 2, pdb.file);
		}

	for (i = 0; i < options.numInputFiles; i++)
		PdbWriteRecord(options.inputFiles[i].path, options.inputFiles[i].size);

	fclose(pdb.file);
	if (!options.quiet)
		printf("...done\n");
	}

static void PdbExitError(int num)
	{
	if (pdb.file)
		{
		fclose(pdb.file);
		unlink(pdb.path);
		}
	exit(num);
	}

static void PdbWriteHeader()
	{
	int i;
	unsigned int timeX;
	char header[78];

	for (i = 0; i < 78; i++)
		header[i] = 0x00;

	// copy name
	strcpy(header, options.name);

	// we append a ! to the end of the name. Under PalmOS, if we have two
	// databases with the same name, they overwrite each other. The PRC file
	// has the same name as options.name so we append a ! to the warp resource
	// PDB name to make it different
	i = strlen(options.name);
	header[i] = '!';
	header[i + 1] = 0;

	// attributes
	// NOTE: tried to set 0x10 to overwrite existing but that flag
	// is apparently documented incorrectly and is actually the reset
	// when load flag. In any case 0x00 allows overwrite of existing.
	header[32] = 0x00;
	header[33] = 0x00;

	// set version
	header[34] = options.version >> 8; // version
	header[35] = options.version & 0xff;

	timeX = time(NULL); // since 1970
	timeX += 66 * 365 * 24 * 60 * 60; // rough add of 66 years

	// creation time
	header[36] = timeX >> 24;
	header[37] = (timeX >> 16) & 0xFF;
	header[38] = (timeX >> 8) & 0xFF;
	header[39] = timeX & 0xFF;

	// modification time
	header[40] = timeX >> 24;
	header[41] = (timeX >> 16) & 0xFF;
	header[42] = (timeX >> 8) & 0xFF;
	header[43] = timeX & 0xFF;

	// database type
	strncpy(&header[60], "Wrp1", 4);

	// creator
	strncpy(&header[64], options.creator, 4);

	// # records
	header[76] = options.numInputFiles >> 8;
	header[77] = options.numInputFiles & 0xFF;

	fwrite(header, 1, 78, pdb.file);
	}

static void PdbWriteOffset()
	{
	int i;
	unsigned char recHeader[8];

	for (i = 0; i < 8; i++)
		recHeader[i] = 0x00;
	recHeader[0] = pdb.recOffset >> 24;
	recHeader[1] = (pdb.recOffset >> 16) & 0xFF;
	recHeader[2] = (pdb.recOffset >> 8) & 0xFF;
	recHeader[3] = pdb.recOffset & 0xFF;
	fwrite(recHeader, 1, 8, pdb.file);
	}

static void PdbWriteRecord(char *path, int size)
	{
	unsigned char *p;
	int pathLen;
	unsigned char lenBuf[2];

	p = readFileIntoMemory(path);
	if (p == NULL)
		{
		printf("ERROR: can't read file %s\n", path);
		PdbExitError(-1);
		}
	if (!options.quiet)
		printf("...adding: %s\n", path);
	pathLen = strlen(path);
	lenBuf[0] = pathLen >> 8;
	lenBuf[1] = pathLen & 0xFF;
	fwrite(lenBuf, 1, 2, pdb.file);
	fwrite(path, 1, pathLen, pdb.file);
	fwrite(p, 1, size, pdb.file); 
	free(p);
	}

static void PdbList(char *path)
	{
	unsigned char *p;
	int i, numFiles;

	p = readFileIntoMemory(path);
	if (p == NULL)
		{
		printf("ERROR: can't open file %s\n", path);
		exit(-1);
		}
	if (strncmp(&p[60], "Wrp1", 4))
		{
		printf("ERROR: bad magic - file not a warp file\n", path);
		exit(-1);
		}
	if (!options.quiet)
		{
		copyright();
		printf("file: %s\n", path);
		}
	numFiles = (p[76] << 8) + p[77];
	printf("record count: %d\n", numFiles);
	printf("contents:\n");
	for (i = 0; i < numFiles; i++)
		{
		unsigned char *rh;
		int j, off, pathLen;

		rh = &p[78 + (i * 8)];
		off = (rh[0] << 24) + (rh[1] << 16) + (rh[2] << 8) + rh[3];
		pathLen = (p[off] << 8) + p[off + 1];
		printf("  ");
		for (j = 0; j < pathLen; j++)
			printf("%c", p[off + 2 + j]);
		printf("\n");
		}
	free(p);
	}

//
// WRP FILE
//

typedef struct
	{
	FILE *file;
	char path[MAX_PATH];
	int recOffset;
	} Wrp;

Wrp wrp;

static void WrpExitError(int num);
static void WrpWriteHeader();
static void WrpWriteOffset();
static void WrpWriteRecord(char *path, int size);

void WrpCreate()
	{
	int i;

	sprintf(wrp.path, "%s.wrp", options.warpFile);
	if (!options.quiet)
		printf("...writing %s\n", wrp.path);
	wrp.file = fopen(wrp.path, WRITE_FLAGS);
	if (!wrp.file)
		{
		printf("ERROR: can't create file %s\n", wrp.path);
		WrpExitError(-1);
		}

	WrpWriteHeader();

	wrp.recOffset = 8 + (options.numInputFiles + 1) * 4;
	for (i = 0; i < options.numInputFiles; i++)
		{
		int pathLen, size;

		WrpWriteOffset();
		pathLen = strlen(options.inputFiles[i].path);
		size = options.inputFiles[i].size;
		wrp.recOffset += 2 + pathLen + size;
		}
	WrpWriteOffset();

	for (i = 0; i < options.numInputFiles; i++)
		WrpWriteRecord(options.inputFiles[i].path, options.inputFiles[i].size);

	fclose(wrp.file);
	if (!options.quiet)
		printf("...done\n");
	}

static void WrpExitError(int num)
	{
	if (wrp.file)
		{
		fclose(wrp.file);
		unlink(wrp.path);
		}
	exit(num);
	}

static void WrpWriteHeader()
	{
	unsigned char header[8];

	strncpy(header, "Wrp1", 4);
	header[4] = options.numInputFiles >> 24;
	header[5] = (options.numInputFiles >> 16) & 0xFF;
	header[6] = (options.numInputFiles >> 8) & 0xFF;
	header[7] = options.numInputFiles & 0xFF;
	fwrite(header, 1, 8, wrp.file);
	}

static void WrpWriteOffset()
	{
	unsigned char recHeader[4];

	recHeader[0] = wrp.recOffset >> 24;
	recHeader[1] = (wrp.recOffset >> 16) & 0xFF;
	recHeader[2] = (wrp.recOffset >> 8) & 0xFF;
	recHeader[3] = wrp.recOffset & 0xFF;

	fwrite(recHeader, 1, 4, wrp.file);
	}

static void WrpWriteRecord(char *path, int size)
	{
	unsigned char *p;
	int pathLen;
	unsigned char lenBuf[2];

	p = readFileIntoMemory(path);
	if (p == NULL)
		{
		printf("ERROR: can't load file %s\n", path);
		WrpExitError(-1);
		}
	if (!options.quiet)
		printf("...adding: %s\n", path);
	pathLen = strlen(path);
	lenBuf[0] = pathLen >> 8;
	lenBuf[1] = pathLen & 0xFF;
	fwrite(lenBuf, 1, 2, wrp.file);
	fwrite(path, 1, pathLen, wrp.file);
	fwrite(p, 1, size, wrp.file); 
	free(p);
	}

static void WrpList(char *path)
	{
	unsigned char *p;
	int i, numFiles;

	p = readFileIntoMemory(path);
	if (p == NULL)
		{
		printf("ERROR: can't open file %s\n", path);
		exit(-1);
		}
	if (strncmp(p, "Wrp1", 4))
		{
		printf("ERROR: bad magic - file not a warp file\n", path);
		exit(-1);
		}
	if (!options.quiet)
		{
		copyright();
		printf("file: %s\n", path);
		}
	numFiles = (p[4] << 24) + (p[5] << 16) + (p[6] << 8) + p[7];
	printf("record count: %d\n", numFiles);
	printf("contents:\n");
	for (i = 0; i < numFiles; i++)
		{
		unsigned char *rh;
		int j, off, off2, pathLen, size;

		rh = &p[8 + (i * 4)];
		off = (rh[0] << 24) + (rh[1] << 16) + (rh[2] << 8) + rh[3];
		rh += 4;
		off2 = (rh[0] << 24) + (rh[1] << 16) + (rh[2] << 8) + rh[3];
		pathLen = (p[off] << 8) + p[off + 1];
		size = off2 - off - pathLen - 2;
		printf("  ");
		for (j = 0; j < pathLen; j++)
			printf("%c", p[off + 2 + j]);
		printf(" (%d)\n", size);
		}
	free(p);
	}
