/* ------------------------------------------------------------------------
 * $Id: client.cc,v 1.34 2001/08/21 21:37:38 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000, 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <stdlib.h>
#include <unistd.h>
#include <vector>

// -- 3Dwm Includes
#include <Nobel++/Nobel.hh>
#include <Nobel/Solid.hh>
#include <Nobel/SolidKit.hh>
#include <Nobel/Shape.hh>
#include <Nobel/NodeKit.hh>
#include <Nobel/Transform.hh>
#include <Nobel/Appearance.hh>
#include <Nobel/GeometryKit.hh>
#include <Nobel/LineGeometry.hh>
#include <Nobel/TriangleGeometry.hh>

using namespace Nobel;

// -- Local Includes
#include "meshio.hh"

// -- Struct Definitions

struct Tri {
    int material;
    Triangle face;
    Triangle texCoordFace;
};

// -- Local Function Prototypes

void usage(const char *binary, const char *msg);

void loadModel(const char *filename, Node_ptr root, NodeKit_ptr node_kit,
	       GeometryKit_ptr geo_kit, SolidKit_ptr csg_kit);

void sendGeometry(TriangleGeometry_ptr geo,
		  const std::vector<Tri> &triList,
		  const std::vector<Vertex3D> &vertexList,
		  const std::vector<Vertex3D> &normalList,
		  const std::vector<TexCoord> &texCoordList);

void sendGeometry(LineGeometry_ptr geo,
		  const std::vector<Tri> &triList,
		  const std::vector<Vertex3D> &vertexList);

void sortGeometry(MeshIO::MeshSet *mset,
		  std::vector< std::vector<Tri> > &triangleListSet,
		  std::vector< std::vector<Vertex3D> > &vertexListSet,
		  std::vector< std::vector<Vertex3D> > &normalListSet,
		  std::vector< std::vector<TexCoord> > &texCoordListSet);

static std::ostream & operator << (std::ostream &os, TriangleGeometry_ptr geo);
static std::ostream & operator << (std::ostream &os, LineGeometry_ptr geo);

// -- Code Segment

int main(int argc, char *argv[]) {
    
    try {

	// Get the filename from the command line
	if (argc < 2)
	    usage(argv[0], "geometry file missing");

	// Initialize the connection
	Nobelxx::Client client("Geometry Client", argc, argv);

	// Resolve the necessary kits
	NodeKit_var node_kit =
	    client.resolve<NodeKit>(Nobelxx::name<NodeKit>());
	GeometryKit_var geo_kit =
	    client.resolve<GeometryKit>(Nobelxx::name<GeometryKit>());
	SolidKit_var solid_kit =
	    client.resolve<SolidKit>(Nobelxx::name<SolidKit>());
	
	// And here's the scene graph root (local to the client)
	Node_var root = client.root();
	
	// Initialize and load the mesh set
	loadModel(argv[1], root, node_kit, geo_kit, solid_kit);
    }
    catch (const CORBA::Exception &e) {

	// The ORB threw an exception
	std::cerr << "Uncaught CORBA exception: " << e << std::endl;
	return EXIT_FAILURE;
    }
    catch (...) {
	
	// Another exception was raised
	std::cerr << "Exception thrown. Exiting." << std::endl;
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}

void usage(const char *binary, const char *msg)
{
    std::cerr << "Error: " << msg << std::endl;
    std::cerr << "USAGE: " << binary << " <model file>" << std::endl;
    throw 0;
}

void copy(Color &color, float f[3])
{
    color.red   = f[0];
    color.green = f[1];
    color.blue  = f[2];
    color.alpha = 1.0f;
}

CORBA::Long &index(Triangle &f, int ndx) 
{
    return ndx == 0 ? f.a : ndx == 1 ? f.b : f.c;
}

void loadModel(const char *filename, Node_ptr root, NodeKit_ptr node_kit,
	       GeometryKit_ptr geo_kit, SolidKit_ptr csg_kit)
{
    // Load the geometry data
    MeshIO::MeshSet *mset = (MeshIO::MeshSet *)
	malloc(sizeof(MeshIO::MeshSet));
    MeshIO::mioMSetInit(mset);
    if (MeshIO::mioLoad(filename, mset) != MeshIO::MIO_NO_ERROR)
	throw 0;

    // Sort the geometry according to material
    std::vector< std::vector<Tri> > triangleListSet;
    std::vector< std::vector<Vertex3D> > vertexListSet;
    std::vector< std::vector<Vertex3D> > normalListSet;
    std::vector< std::vector<TexCoord> > texCoordListSet;

    std::cerr << "sorting geometry... "; 
    sortGeometry(mset, triangleListSet, vertexListSet, 
		 normalListSet, texCoordListSet);
    std::cerr << "done." << std::endl;

    // Create shape nodes for all materials
    for (unsigned int i = 0; i < triangleListSet.size(); i++) {

	// Create the shape
	Shape_var shape = node_kit->createShape();
	
	// Create the material
	MaterialAttributes material;
	
	float emissive[3] = { 0.0f, 0.0f, 0.0f };
	
	copy(material.ambient, mset->materialList[i].ambient);
	copy(material.specular, mset->materialList[i].specular);
	copy(material.diffuse, mset->materialList[i].diffuse);
	copy(material.emissive, emissive);
	material.shininess = 10.0f;
	
	Appearance_var app = geo_kit->createAppearance();
	
	app->material(material);
	
	shape->setAppearance(app);
	
#ifdef USE_LINES
	// Create and define the triangle geometry
	LineGeometry_var geo = geo_kit->createLineGeometry();
	sendGeometry(geo, triangleListSet[i], vertexListSet[i]);
#else
	// Create and define the triangle geometry
	TriangleGeometry_var geo = geo_kit->createTriangleGeometry();
	
	sendGeometry(geo, triangleListSet[i], vertexListSet[i],
		     normalListSet[i], texCoordListSet[i]);
#endif
	
	// Add the geometry to the shape
	shape->addGeometry(geo); 
	std::cerr << geo << std::endl;

#if 0
	std::cerr << "\tCreating solid geometry with BSP tree... ";
	Solid::Geometry_var csg = csg_kit->createGeometry(geo);
	std::cerr << "done." << std::endl;
#endif
	
	// Add shape to client root
	root->insert(shape);
    }

    // Clear and deallocate mesh set
    MeshIO::mioMSetClear(mset);
    free(mset);
}

void storeTriangle(IndexSeq &vs, Triangle &t, int index)
{
    vs[index * 3] = t.a;
    vs[index * 3 + 1] = t.b;
    vs[index * 3 + 2] = t.c;
}

void sendGeometry(TriangleGeometry_ptr geo,
		  const std::vector<Tri> &triList,
		  const std::vector<Vertex3D> &vertexList,
		  const std::vector<Vertex3D> &normalList,
		  const std::vector<TexCoord> &texCoordList)
{
#if 1
    geo->setVertexNumber(vertexList.size());
    for (unsigned int i = 0; i < vertexList.size(); i++)
	geo->setVertex(i, vertexList[i]);

    geo->setFaceNumber(triList.size());
    for (unsigned int i = 0; i < triList.size(); i++) {
	geo->setVertexIndex(i, triList[i].face);
	geo->setNormalIndex(i, triList[i].face);
	geo->setTexCoordIndex(i, triList[i].texCoordFace);
    }
    
    geo->setNormalNumber(normalList.size());
    for (unsigned int i = 0; i < normalList.size(); i++)
	geo->setNormal(i, normalList[i]);
    
    geo->setTexCoordNumber(texCoordList.size());
    for (unsigned int i = 0; i < texCoordList.size(); i++)
	geo->setTexCoord(i, texCoordList[i]);
#else
    TriangleMesh mesh;

    mesh.vertexList.length(vertexList.size());
    for (unsigned int i = 0; i < vertexList.size(); i++)
	mesh.vertexList[i] = vertexList[i];

    mesh.vertexIndexList.length(triList.size() * 3);
    mesh.normalIndexList.length(triList.size() * 3);
    mesh.texCoordIndexList.length(triList.size() * 3);
    for (unsigned int i = 0; i < triList.size(); i++) {
	Triangle t = triList[i].face;
	Triangle tc = triList[i].texCoordFace;
	storeTriangle(mesh.vertexIndexList, t, i);
	storeTriangle(mesh.normalIndexList, t, i);
	storeTriangle(mesh.texCoordIndexList, tc, i);
    }
    
    mesh.normalList.length(normalList.size());
    for (unsigned int i = 0; i < normalList.size(); i++)
	mesh.normalList[i] = normalList[i];
    
    mesh.texCoordList.length(texCoordList.size());
    for (unsigned int i = 0; i < texCoordList.size(); i++)
	mesh.texCoordList[i] = texCoordList[i];
    
    geo->setMesh(mesh);
#endif
}

void sendGeometry(LineGeometry_ptr geo,
		  const std::vector<Tri> &triList,
		  const std::vector<Vertex3D> &vertexList)
{
#if 1
    geo->setVertexNumber(vertexList.size());
    for (unsigned int i = 0; i < vertexList.size(); i++)
	geo->setVertex(i, vertexList[i]);

    geo->setLineNumber(triList.size() * 3);
    for (unsigned int i = 0; i < triList.size(); i++) {
	
	Line line;
	
	line.a = triList[i].face.a;
	line.b = triList[i].face.b;
	geo->setVertexIndex(i * 3, line);

	line.a = triList[i].face.b;
	line.b = triList[i].face.c;
	geo->setVertexIndex(i * 3 + 1, line);

	line.a = triList[i].face.c;
	line.b = triList[i].face.a;
	geo->setVertexIndex(i * 3 + 2, line);
    }
#else
    
#endif
}


void sortGeometry(MeshIO::MeshSet *mset,
		  std::vector< std::vector<Tri> > &triangleListSet,
		  std::vector< std::vector<Vertex3D> > &vertexListSet,
		  std::vector< std::vector<Vertex3D> > &normalListSet,
		  std::vector< std::vector<TexCoord> > &texCoordListSet)
{
    std::vector<Vertex3D> vertexList;
    std::vector<Vertex3D> normalList;
    std::vector<Tri> triangleList;
    std::vector<TexCoord> texCoordList;
 
    for (MeshIO::MeshNode *node = mset->list; node != 0; node = node->next) {
	
	int vertex_offset = vertexList.size();
	int texcoord_offset = texCoordList.size();
	MeshIO::Mesh *mesh = node->mesh;
	
	// Add all vertices to the main list
	for (int i = 0; i < mesh->vertexNum; i++) {
	    Vertex3D vertex;
	    
	    vertex.x = mesh->vertexList[i].coord.x;
	    vertex.y = mesh->vertexList[i].coord.y;
	    vertex.z = mesh->vertexList[i].coord.z;
	    
	    vertexList.push_back(vertex);
	}

	// Add all normals to the main list
	for (int i = 0; i < mesh->vertexNum; i++) {
	    Vertex3D normal;
	    
	    normal.x = mesh->normalList[i].coord.x;
	    normal.y = mesh->normalList[i].coord.y;
	    normal.z = mesh->normalList[i].coord.z;
	    
	    normalList.push_back(normal);
	}

	// Add all texture coordinates to the main list
	for (int i = 0; i < mesh->mappingNum; i++) {
	    TexCoord tc;
	    
	    tc.u = mesh->mappingList[i].coord.x;
	    tc.v = mesh->mappingList[i].coord.y;
	    
	    texCoordList.push_back(tc);
	}
	
	// Add all triangles to the main list
	for (int i = 0; i < mesh->faceNum; i++) {
	    Tri tri;
	    
	    tri.face.a = mesh->faceList[i].ndxVertex.ndx.a + vertex_offset;
	    tri.face.b = mesh->faceList[i].ndxVertex.ndx.b + vertex_offset;
	    tri.face.c = mesh->faceList[i].ndxVertex.ndx.c + vertex_offset;
	    
	    tri.texCoordFace.a = mesh->faceList[i].ndxMapping.ndx.a + 
		texcoord_offset;
	    tri.texCoordFace.b = mesh->faceList[i].ndxMapping.ndx.b +
		texcoord_offset;
	    tri.texCoordFace.c = mesh->faceList[i].ndxMapping.ndx.c +
		texcoord_offset;

	    tri.material = mesh->faceList[i].material;
	    
	    triangleList.push_back(tri);
	}
    }
    
    // Split geometry data in different sets depending on material
    for (int current_material = 0; current_material < mset->materialNum;
	 current_material++) {
	
	// Temporary lists
	std::vector<Vertex3D> vList;
	std::vector<Vertex3D> nList;
	std::vector<Tri> tList;
	std::vector<TexCoord> tcList;
	
	// Get the affected triangles
	std::vector<Tri>::iterator it;
 	for (it = triangleList.begin(); it != triangleList.end(); it++) 
	    if (it->material == current_material)
		tList.push_back(*it);

	// Copy the triangle list and flag all vertices as unused
	std::vector<Tri> tNewList = tList;
	std::vector<bool> vtxUsed, tcUsed;
	for (unsigned int i = 0; i < vertexList.size(); i++)
	    vtxUsed.push_back(false);
	for (unsigned int i = 0; i < texCoordList.size(); i++)
	    tcUsed.push_back(false);

	// Now, add vertices, normals and texture coordinates
	for (it = tList.begin(); it != tList.end(); it++) {
	    
	    // We'll step through all three indices for each triangle
	    for (int ndx = 0; ndx < 3; ndx++) {
		
		unsigned int old_index;
		
		// Has this vertex index been translated?
		old_index = index(it->face, ndx);
		if (vtxUsed[old_index] == false) {
		    
		    // Yes. That means we haven't acquired it yet. Do so.
		    unsigned int new_index = vList.size();

		    vList.push_back(vertexList[old_index]);
		    nList.push_back(normalList[old_index]);
		    
		    vtxUsed[old_index] = true;
		    
		    // Update all indices
		    for (unsigned int j = 0; j < tList.size(); j++) {
			for (int t = 0; t < 3; t++)
			    if (index(tList[j].face, t) == 
				CORBA::Long(old_index))
				index(tNewList[j].face, t) = new_index;
		    }
		}
		
		// Do the same for texture coordinates (if any)
		if (texCoordList.size() == 0)
		    continue;
		/*
		old_index = index(it->texCoordFace, ndx);
		if (tcUsed[old_index] == false) {
		    
		    // Yes. That means we haven't acquired it yet. Do so.
		    unsigned int new_index = tcList.size();
		    tcList.push_back(texCoordList[old_index]);
		    
		    tcUsed[old_index] = true;
		    
		    // Update all indices
		    for (unsigned int j = 0; j < tList.size(); j++) {
			for (int t = 0; t < 3; t++)
			    if (index(tList[j].texCoordFace, t) == 
				CORBA::Long(old_index))
				index(tNewList[j].texCoordFace, t) = new_index;
		    }
		}
		*/
	    }
	}

	// Add the lists to the sets
	triangleListSet.push_back(tNewList);
	vertexListSet.push_back(vList);
	normalListSet.push_back(nList);
	texCoordListSet.push_back(tcList);
    }
}

std::ostream & operator << (std::ostream &os, TriangleGeometry_ptr geo)
{
    os << "Number of vertices: " << geo->getVertexNumber() << ", "
       << "Number of faces: " << geo->getFaceNumber() << ", " 
       << "Number of texture coordinates: " << geo->getTexCoordNumber();
    return os;
}

std::ostream & operator << (std::ostream &os, LineGeometry_ptr geo)
{
    os << "Number of vertices: " << geo->getVertexNumber() << ", "
       << "Number of faces: " << geo->getLineNumber();
    return os;
}
