/* ------------------------------------------------------------------------
 * $Id: SolidCache.hh,v 1.1 2001/08/09 14:19:32 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _SolidCache_hh_
#define _SolidCache_hh_

// -- System Includes
#include <hash_map>

// -- 3Dwm Includes
#include "Nobel/CORBA.hh"
#include "Nobel/Solid.hh"

// -- Forward Declarations
class BSPTree;

// -- Class Declarations

/**
 * Solid caching class. Used for caching intermediate representations
 * of solids in a solid tree (BSP trees, in this case). @@@ There is
 * an obvious generalization (using templates) to be done for this
 * class if it is ever needed.
 **/
class SolidCache {
public:

    /**
     * Cache key struct.
     **/
    struct cache_key_t {

	/// Object reference key
	Nobel::Solid::Node_var key;

	/**
	 * Constructor.
	 **/
	cache_key_t(Nobel::Solid::Node_ptr _key) 
	    : key(Nobel::Solid::Node::_duplicate(_key)) { }

	/**
	 * Equivalence operator. 
	 **/
	bool operator == (const cache_key_t &ctk) const {
	    return key->_is_equivalent(ctk.key);
	}
    };

    /**
     * Cache key hash function object.
     **/
    struct hash<cache_key_t> {

	/// Arbitrary number of partitions for IORs
	static const unsigned int hash_max = 64;

	/**
	 * Function operator call. Used to compute the hash value for
	 * the given key.
	 **/
	size_t operator() (const cache_key_t &t) const {
	    return (size_t) t.key->_hash(hash_max);
	}
    };
    
    /**
     * Constructor.
     **/
    SolidCache();

    /**
     * Destructor.
     **/ 
    virtual ~SolidCache();

    /**
     * Lookup a solid node in the cache and return the cached BSP tree
     * if found. If the node is not cached, this call will return a
     * null (0) pointer.
     *
     * @param key solid node reference associated with the cached tree.
     * @return pointer to the cached BSP tree if it exists in the
     *         cache, 0 if not.
     **/
    BSPTree *lookup(Nobel::Solid::Node_ptr key) const;

    /**
     * Query whether a given solid node exists in the cache.
     *
     * @param key solid node reference to query for a cached version of.
     * @return true if a cached tree exists, false otherwise.
     **/
    bool exists(Nobel::Solid::Node_ptr key) const;

    /**
     * Invalidate a cached version of a given solid node. This will
     * remove the cached version of the solid node from the cache (if
     * there is no cached version, this call will silently return
     * without doing anything).
     *
     * @param key solid node reference to invalidate in the cache.
     **/
    void invalidate(Nobel::Solid::Node_ptr key);
    
    /**
     * Cache the BSP tree of a given solid node. The solid node
     * reference will be used as a key. If the node has already been
     * cached, the old cache entry will be deallocated and replaced by
     * the new.
     *
     * @param key solid node reference to use as a key.
     * @param tree pointer to the BSP tree to cache. 
     **/
    void cache(Nobel::Solid::Node_ptr key, BSPTree *tree);

    /**
     * Clear the cache. This will remove (and deallocate) all the
     * cached BSP trees in the cache.
     **/
    void clear();

private:    
    std::hash_map<cache_key_t, BSPTree *>  _cache;
};

#endif /* SolidCache.hh */
