/* ------------------------------------------------------------------------
 * $Id: SolidContainerImpl.cc,v 1.3 2001/08/28 13:18:52 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <map>

// -- 3Dwm Includes
#include "Nobel/Renderer.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Polhem/SolidEvaluator.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/SolidContainerImpl.hh"

using namespace Nobel;

// -- Code Segment

SolidListener::~SolidListener()
{
    // Unregister interest in the solid node
    if (!CORBA::is_nil(_tree)) _tree->removeListener(Listener_var(_this()));
}

void SolidListener::setTree(Nobel::Solid::Node_ptr tree)
{
    // Unregister the listener, update tree reference, and set the
    // listener in the new tree.
    if (!CORBA::is_nil(_tree)) _tree->removeListener(Listener_var(_this()));
    _tree = Solid::Node::_duplicate(tree);
    _tree->addListener(Listener_var(_this()));
}

void SolidListener::receive(const Event &e)
{
    // We only act on child change events
    if (e.type != SubjectChanged) return;
    
    // Try to narrow the reference to a solid node
    try {
	// Narrow the reference
	Solid::Node_var node = Solid::Node::_narrow(e.obj);

	// It succeeded if we got here, so invalidate the reference in
	// the cache.
	_cache->invalidate(node);
    }
    catch (const CORBA::Exception &e) { }
}

SolidContainerImpl::SolidContainerImpl()
    : _bounds(new VolumeImpl),
      _listener(new SolidListener(&_cache)),
      _evaluator(new SolidEvaluator(&_cache))
{
    // empty
}

void SolidContainerImpl::clearMeshList()
{
    // Deallocate the triangle meshes in the mesh list
    for (std::vector<TriMeshShape>::iterator i = _mesh_list.begin();
	 i != _mesh_list.end(); i++)
	delete i->mesh;
    
    // Clear the list itself
    _mesh_list.clear();
}

SolidContainerImpl::~SolidContainerImpl()
{
    // Delete geometry data
    clearMeshList();
    
    // Deactivate CORBA entities
    deactivate(_bounds);
    deactivate(_listener);
    deactivate(_evaluator);

    // Delete entities
    delete _bounds;
    delete _listener;
    delete _evaluator;
}

void SolidContainerImpl::render(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);
    
    // Render the triangle meshes in the mesh list
    for (std::vector<TriMeshShape>::iterator i = _mesh_list.begin();
	 i != _mesh_list.end(); i++) {
	
	// First apply the appearance
	if (!CORBA::is_nil(i->appearance))
	    i->appearance->apply(r);

	// Then render the triangle mesh
	r->renderTriangles(*(i->mesh));
    }
}

void SolidContainerImpl::pick(Picker_ptr p)
{
    // @@@ should not be empty!!!
}

Volume_ptr SolidContainerImpl::getBoundingVolume()
{
    // Return the new volume
    return _bounds->_this();
}

void SolidContainerImpl::setTree(Solid::Node_ptr tree)
{
    Guard<Mutex> guard(_mutex);

    // Update reference and the listener
    _tree = Solid::Node::_duplicate(tree);
    _listener->setTree(_tree);

    // Clear the cache
    _cache.clear();
}

Solid::Node_ptr SolidContainerImpl::getTree()
{    
    // Return a duplicate to the solid tree pointer
    Guard<Mutex> guard(_mutex);
    return Solid::Node::_duplicate(_tree);
}

void SolidContainerImpl::evaluate()
{
    // Sanity check 
    if (CORBA::is_nil(_tree)) return;
    
    // Use the solid evaluator to generate the triangles for the solid tree
    std::vector<TriMeshShape> temp_mesh_list = _evaluator->evaluate(_tree);

    // If the mesh list is empty, that means that there was an error,
    // or the meshes did not change. In any case, we just silently
    // ignore this.
    if (temp_mesh_list.empty() == true) return;

    {
        Guard<Mutex> guard(_mutex);

	// Now deallocate the mesh list
	clearMeshList();
	
	// Use the other mesh list
	_mesh_list = temp_mesh_list;
    }
}

void SolidContainerImpl::recomputeBoundingVolume()
{
    // @@@ empty at the moment!
}
