/* ------------------------------------------------------------------------
 * $Id: SolidEvaluator.cc,v 1.10 2001/08/27 15:10:28 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <map>
#include <algorithm>

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Solid/Face.hh"
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"
#include "Solid/BSPTreeBuilder.hh"
#include "Nobel/Primitive.hh"
#include "Nobel/Appearance.hh"
#include "Nobel/TriangleGeometry.hh"
#include "Polhem/SolidCache.hh"
#include "Polhem/SolidEvaluator.hh"

using namespace Nobel;

// -- Code Segment

std::ostream &operator << (std::ostream &f, const BSPTree &tree) 
{
    f << "Vertices:" << std::endl;
    for (int i = 0; i < tree.getVertexCount(); i++) 
	f << "#" << i << ": " << tree.getVertex(i) << std::endl;
    f << "Faces:" << std::endl;
    std::vector<TriangleFace> face_list;
    tree.getTree()->buildFaceList(face_list);
    for (unsigned int i = 0; i < face_list.size(); i++) 
	f << "#" << i << ": " 
	  << " a: " << face_list[i].v.a()
	  << " b: " << face_list[i].v.b()
	  << " c: " << face_list[i].v.c()
	  << std::endl;
    return f;
}

Vertex3D fromVector3D(const Vector3D &v)
{
    Vertex3D vtx = { v.x(), v.y(), v.z() };
    return vtx;
}

TexCoord fromVector2D(const Vector2D &v)
{
    TexCoord tc = { v.x(), v.y() };
    return tc;
}

SolidEvaluator::SolidEvaluator(SolidCache *cache) 
    : _cache(cache)
{
    // empty
}

std::vector<TriMeshShape> SolidEvaluator::evaluate(Solid::Node_ptr tree)
{
    std::vector<TriMeshShape> mesh_list;

    // Sanity check
    if (CORBA::is_nil(tree)) return mesh_list;

    // Initialize the material appearance change flag
    _appChanged = false;

    // Evaluate the solid tree
    tree->accept(Solid::Visitor_var(_this()));
    
    // Retrieve the resulting BSP tree from the top of the stack
    TreeItem item = popTree();

    // If the BSP didn't change, we don't want to create new meshes
    if (item.changed == false || item.tree == 0) 
	return mesh_list;
    
    // Separate triangles into meshes with the same material
    mesh_list = buildMeshFromBSP(item.tree);

    return mesh_list;
}

void SolidEvaluator::visitComplement(Solid::Unary_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->body()) == false) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }
    
    // First, evaluate the body of the node
    node->body()->accept(Solid::Visitor_var(_this()));        
    
    // Pop the topmost tree off the stack
    TreeItem item = popTree();

    // If we didn't get a valid tree pointer, just return
    if (item.tree == 0)
	pushTree(0);
    else {

	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item.changed && _cache->exists(node)) 
	    pushTree(_cache->lookup(node));

	// Okay, it didn't exist or the children changed, recompute
	else {
	    
	    // Complement the tree and push the result onto the
	    // evaluation stack
	    BSPTree *result = BSPTreeBuilder::complement(item.tree);
	    pushTree(result, true);

	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, result);
	}
    }
}

void SolidEvaluator::visitUnion(Solid::Binary_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->left()) || CORBA::is_nil(node->right())) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }
    
    // First, evaluate left and right subtrees
    node->left()->accept(Solid::Visitor_var(_this()));
    node->right()->accept(Solid::Visitor_var(_this()));
    
    // Pop the operand trees off the evaluation stack
    TreeItem item2 = popTree();
    TreeItem item1 = popTree();

    // If we didn't get valid tree pointers, just return
    if (item1.tree == 0 || item2.tree == 0)
	pushTree(0);
    else {
	
	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item1.changed && !item2.changed && _cache->exists(node)) 
	    pushTree(_cache->lookup(node));

	// Okay, it didn't exist or the children changed, recompute!
	else {
	    
	    // Perform a union between the operand trees and store result
	    BSPTree *r = BSPTreeBuilder::combine(item1.tree, item2.tree,
						 BSPTreeBuilder::Union);
	    pushTree(r, true);
	    
	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, r);
	}
    }
}

void SolidEvaluator::visitIntersection(Solid::Binary_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->left()) || CORBA::is_nil(node->right())) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }
    
    // First, evaluate left and right subtrees
    node->left()->accept(Solid::Visitor_var(_this()));    
    node->right()->accept(Solid::Visitor_var(_this()));    
    
    // Pop the operand trees off the evaluation stack
    TreeItem item2 = popTree();
    TreeItem item1 = popTree();
    
    // If we didn't get valid tree pointers, just return
    if (item1.tree == 0 || item2.tree == 0)
	pushTree(0);
    else {
	
	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item1.changed && !item2.changed && _cache->exists(node)) 
	    pushTree(_cache->lookup(node));

	// Okay, it didn't exist or the children changed, recompute!
	else {
	    
	    // Perform a union between the operand trees and store result
	    BSPTree *r = BSPTreeBuilder::combine(item1.tree, item2.tree,
						 BSPTreeBuilder::Intersection);
	    pushTree(r, true);
	    
	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, r);
	}
    }
}

void SolidEvaluator::visitSubtraction(Solid::Binary_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->left()) || CORBA::is_nil(node->right())) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }
    
    // First, evaluate left and right subtrees
    node->left()->accept(Solid::Visitor_var(_this()));    
    node->right()->accept(Solid::Visitor_var(_this()));    
    
    // Pop the operand trees off the evaluation stack
    TreeItem item2 = popTree();
    TreeItem item1 = popTree();
    
    // If we didn't get valid tree pointers, just return
    if (item1.tree == 0 || item2.tree == 0)
	pushTree(0);
    else {
	
	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item1.changed && !item2.changed && _cache->exists(node)) 
	    pushTree(_cache->lookup(node));

	// Okay, it didn't exist or the children changed, recompute!
	else {
	    
	    // First, complement the second operand
	    BSPTree *t2_compl = BSPTreeBuilder::complement(item2.tree);
	    
	    // Then, do an intersection, which together makes a subtraction
	    BSPTree *r = BSPTreeBuilder::combine(item1.tree, t2_compl,
						 BSPTreeBuilder::Intersection);
	    pushTree(r, true);
	    
	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, r);
	    
	    // Delete the temporary tree
	    delete t2_compl;
	}
    }
}

void SolidEvaluator::visitAppearance(Solid::Appearance_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->body())) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }

    // Retrieve the appearance of the node
    Appearance_var appearance = node->app();
    
    // See if it has been cached already...
    unsigned int index = 0;
    for (std::vector<Appearance_var>::iterator i = _appearances.begin();
	 i != _appearances.end(); i++, index++) 
	if ((*i)->_is_equivalent(appearance)) break;
    
    // If the index is too high, we didn't find it, so we add it
    if (index == _appearances.size())
	_appearances.push_back(appearance);

    // Store the material index on the material stack
    pushMaterial(index);

    // If the appearance has changed, we want to invalidate the whole
    // subtree.
    bool save_appChanged = _appChanged;
    if (!_cache->exists(node))
	_appChanged = true;
    
    // Visit the body of the node
    node->body()->accept(Solid::Visitor_var(_this()));
    
    // Don't forget to remove our material and reset appearance
    // changed flag
    popMaterial();
    _appChanged = save_appChanged;

    // Retrieve the current tree
    TreeItem item = popTree();
    
    // If we didn't get a valid tree pointer, just return
    if (item.tree == 0)
	pushTree(0);
    else {
	
	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item.changed && _cache->exists(node))
	    pushTree(_cache->lookup(node));
	
	// Okay, it didn't exist or the children changed, recompute
	else {
	    
	    // Clone the tree
	    BSPTree *result = new BSPTree(*item.tree);
	    
	    // Store the result on the evaluation stack
	    pushTree(result, true);
	    
	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, result);
	}
    }
}

void SolidEvaluator::visitTransform(Solid::Transform_ptr node)
{
    // Make sure this is not a dead end
    if (CORBA::is_nil(node->body())) {
	// It is, push an empty tree pointer onto the stack
	pushTree(0);
	return;
    }
    
    // Visit the body of the node
    node->body()->accept(Solid::Visitor_var(_this()));
    
    // Retrieve the current tree
    TreeItem item = popTree();
    
    // If we didn't get a valid tree pointer, just return
    if (item.tree == 0)
	pushTree(0);
    else {
	
	// If the children didn't change and this node exists in the
	// cache, we just use the cached version.
	if (!item.changed && _cache->exists(node)) 
	    pushTree(_cache->lookup(node));
	
	// Okay, it didn't exist or the children changed, recompute
	else {
	    
	    // Clone the tree
	    BSPTree *result = new BSPTree(*item.tree);
	    
	    // Now, we retrieve the transformation data
	    Matrix m;
	    node->getTransformationMatrix(m);
	    
	    // Apply the transformation to the resulting tree
	    Matrix3D trafo(m);
	    result->transform(trafo);
	    
	    // Store the result on the evaluation stack
	    pushTree(result, true);

	    // Update cache with the new BSP tree for this node
	    _cache->cache(node, result);
	}
    }
}

void SolidEvaluator::visitGeometry(Solid::Geometry_ptr node)
{
    // Does this node exist in the cache? In that case, use the
    // existing BSP tree instead of computing a new one. Also, we need
    // to make sure that a parent appearance hasn't changed, or we
    // need to rebuild anyway.
    if (_cache->exists(node) && !_appChanged)
	pushTree(_cache->lookup(node));
    else {
	
	// Retrieve geometry data from the geometry
	TriangleGeometry_var geo = node->geo();
	TriangleMesh *mesh = geo->getMesh();
	
	// Build the BSP tree and add it to the evaluation stack
	BSPTree *result = buildBSPFromMesh(mesh);
	pushTree(result, true);

	// Update cache with the new BSP tree for this node
	_cache->cache(node, result);
	
	// Delete the triangle mesh 
	delete mesh;
    }
}

void SolidEvaluator::visitPrimitive(Solid::Primitive_ptr node)
{
    // Does this node exist in the cache? In that case, use the
    // existing BSP tree instead of computing a new one. Also, we need
    // to make sure that a parent appearance hasn't changed, or we
    // need to rebuild anyway.
    if (_cache->exists(node) && !_appChanged) 
	pushTree(_cache->lookup(node));
    else {
	
	// Retrieve geometry data from the geometry
	Primitive_var prim = node->prim();
	TriangleMesh *mesh = prim->getMesh();
	
	// Build the BSP tree and add it to the evaluation stack
	BSPTree *result = buildBSPFromMesh(mesh);
	pushTree(result, true);
	
	// Update cache with the new BSP tree for this node
	_cache->cache(node, result);
	
	// Delete the triangle mesh 
	delete mesh;
    }
}

BSPTree *SolidEvaluator::buildBSPFromMesh(TriangleMesh *mesh)
{
    // Build the vertex list...
    std::vector<Vector3D> vertex_list;
    for (unsigned int i = 0; i < mesh->vertexList.length(); i++)
	vertex_list.push_back(Vector3D(mesh->vertexList[i].x,
				       mesh->vertexList[i].y,
				       mesh->vertexList[i].z));

    // The normal list...
    std::vector<Vector3D> normal_list;
    for (unsigned int i = 0; i < mesh->normalList.length(); i++)
	normal_list.push_back(Vector3D(mesh->normalList[i].x,
				       mesh->normalList[i].y,
				       mesh->normalList[i].z));

    // The texture coordinate list...
    std::vector<Vector2D> texcoord_list;
    for (unsigned int i = 0; i < mesh->texCoordList.length(); i++)
	texcoord_list.push_back(Vector2D(mesh->texCoordList[i].u,
					 mesh->texCoordList[i].v));
    
    // And the face list!
    std::vector<TriangleFace> face_list;
    for (unsigned int i = 0; i < mesh->vertexIndexList.length() / 3; i++) {
	TriangleFace t;
	
	// Initialize the triangle components
	t.v = TriangleIndex(mesh->vertexIndexList[i * 3],
			    mesh->vertexIndexList[i * 3 + 1],
			    mesh->vertexIndexList[i * 3 + 2]);
	t.n = TriangleIndex(mesh->normalIndexList[i * 3],
			    mesh->normalIndexList[i * 3 + 1],
			    mesh->normalIndexList[i * 3 + 2]);
	t.tc = TriangleIndex(mesh->texCoordIndexList[i * 3],
			     mesh->texCoordIndexList[i * 3 + 1],
			     mesh->texCoordIndexList[i * 3 + 2]);
	
	// Get the material index from the material stack
	t.material = getCurrentMaterial();
	
	// Add the triangle to the face list
	face_list.push_back(t);
    }
    
    // Now, build the friggin' BSP tree and return it :)
    return BSPTreeBuilder::build(face_list, vertex_list, normal_list,
				 texcoord_list);
}

bool orderFaces(const TriangleFace &f1, const TriangleFace &f2)
{
    return f1.material < f2.material;
}

TriangleIndex remap(const TriangleIndex &f,
		    std::map<int, int> &index_map,
		    int &index_counter)
{
    TriangleIndex result;
    
    // Iterate through all indices in the triangle index
    for (int i = 0; i < 3; i++) {
	
	// Has the index already been remapped? 
	std::map<int, int>::iterator n = index_map.find(f[i]);
	
	// Yes, it has, so use the old index
	if (n != index_map.end())
	    result[i] = n->second;
	// No, it has not, add a new index
	else {
	    int new_index = index_counter++;
	    index_map[f[i]] = new_index;
	    result[i] = new_index;
	}
    }
    return result;
}

std::vector<TriMeshShape> SolidEvaluator::buildMeshFromBSP(BSPTree *t) const
{
    std::vector<TriMeshShape> mesh_list;
    std::vector<TriangleFace> face_list;
    
    // Build the triangle face list
    t->getTree()->buildFaceList(face_list);

    // Sort the triangle face list
    std::sort(face_list.begin(), face_list.end(), orderFaces);
    std::vector<TriangleFace>::iterator i = face_list.begin();

    // Create a mesh for every appearance
    while (i != face_list.end()) {
	
	// Create the new mesh
	TriangleMesh *mesh = new TriangleMesh; 
	
	// Retrieve the material index
	int material = i->material;

	std::map<int, int> v_index_map, n_index_map, tc_index_map;
	int v_counter = 0, n_counter = 0, tc_counter = 0;
	
	// Add triangles until we reach the next material
	std::vector<TriangleFace> tlist;
	for ( ; i != face_list.end() && i->material == material; i++) {
	    
	    TriangleFace face;
	    
	    // Remap indices for vertices, normals and texture coordinates
	    face.v = remap(i->v, v_index_map, v_counter);
	    face.n = remap(i->n, n_index_map, n_counter);
	    face.tc = remap(i->tc, tc_index_map, tc_counter);
	    
	    tlist.push_back(face);
	}
	
	// Add the vertices we need to the mesh (and in the correct
	// order).
	mesh->vertexList.length(v_counter);
	for (std::map<int, int>::iterator i = v_index_map.begin();
	     i != v_index_map.end(); i++)
	    mesh->vertexList[i->second] = fromVector3D(t->getVertex(i->first));
	
	// Then add the normals we need to the mesh (and in the
	// correct order).
	mesh->normalList.length(n_counter);
	for (std::map<int, int>::iterator i = n_index_map.begin();
	     i != n_index_map.end(); i++)
	    mesh->normalList[i->second] = fromVector3D(t->getNormal(i->first));
	
	// Finally, add the texture coordinates we need to the mesh
	// (and in the correct order).
	mesh->texCoordList.length(tc_counter);
	for (std::map<int, int>::iterator i = tc_index_map.begin();
	     i != tc_index_map.end(); i++)
 	    mesh->texCoordList[i->second] =
		fromVector2D(t->getTexCoord(i->first));

	// Then go for the triangle faces
	mesh->vertexIndexList.length(tlist.size() * 3);
	mesh->normalIndexList.length(tlist.size() * 3);
	mesh->texCoordIndexList.length(tlist.size() * 3);
	
	// Step through all faces in the list
	for (unsigned int i = 0; i < tlist.size(); i++) {
	    
	    // Three components to every face
	    for (int j = 0; j < 3; j++) {
		mesh->vertexIndexList[i * 3 + j] = tlist[i].v[j];
		mesh->normalIndexList[i * 3 + j] = tlist[i].n[j];
		mesh->texCoordIndexList[i * 3 + j] = tlist[i].tc[j];
	    }
	}
	
	// Finally, add the finished mesh to the mesh list
	mesh_list.push_back(TriMeshShape(mesh, getAppearance(material)));
    }

    return mesh_list;
}

Appearance_ptr SolidEvaluator::getAppearance(int material) const
{
    // If the material index is out of bounds, return a nil reference
    if (material < 0 || material >= (int) _appearances.size())
	return Appearance::_nil();
    return Appearance::_duplicate(_appearances[material]);
}
