/* ------------------------------------------------------------------------
 * $Id: VNCDesktop.cc,v 1.4 2001/08/28 13:18:52 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-10-25 by Steve Houston.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System includes

// -- 3Dwm includes
#include "Nobel/Mapper.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Garbo/VNCInterface.hh"
#include "Polhem/VNCDesktop.hh"
#include "Celsius/Thread.hh"

using namespace Nobel;

// -- Code Segment

int getNearestMultipleOfTwo(int value)
{
    int bits;
    value--;
    for (bits = 0; value; bits++, value >>= 1);
    return (1 << bits);
}

VNCTextureGenerator::VNCTextureGenerator(VNCInterface *vnc)
    : _vnc(vnc)
{
    // Retrieve VNC desktop dimensions
    _vncWidth = _vnc->getWidth();
    _vncHeight = _vnc->getHeight();
    
    // Calculate size of texture (must be a power of two)
    _width = getNearestMultipleOfTwo(_vncWidth);
    _height = getNearestMultipleOfTwo(_vncHeight);
    
    // Get the first update
    _vnc->updateFramebuffer(0, 0, _vncWidth, _vncHeight);
}

VNCTextureGenerator::~VNCTextureGenerator()
{
    // empty
}

CORBA::Short VNCTextureGenerator::getPixelSize() 
{
    return (CORBA::Short) _vnc->getPixelSize(); 
}

Nobel::PixelType VNCTextureGenerator::getPixelType() 
{ 
    return _vnc->getPixelType(); 
}

BinaryData *VNCTextureGenerator::buildUpdate(const Rectangle &r)
{
    // Allocate a temporary buffer for our use
    int bytesPerPixel = getPixelSize() / 8;
    int updateSize = r.w * r.h * bytesPerPixel;
    char *buf = new char [updateSize];

    // Clear the buffer
    memset(buf, 0, updateSize);
    
    // Initialize the framebuffer pointers
    const char *src = _vnc->getFramebuffer() + r.y * _vncWidth + r.x;
    char *dst = buf;

    // Make sure we don't stray outside of the framebuffer
    int copyWidth = std::min(_vncWidth - r.x, r.w);
    int copyHeight = std::min(_vncHeight - r.y, r.h);
    
    // Copy data to the temporary buffer
    for (int i = 0; i < copyHeight; i++) {
	memcpy(dst, src, copyWidth * bytesPerPixel);
	src += _vncWidth * bytesPerPixel;
	dst += r.w * bytesPerPixel;
    }

    // Create and return the CORBA binary sequence (let CORBA worry
    // about memory management).
    return new BinaryData(updateSize, updateSize, (CORBA::Octet *) buf, true);
}

BinaryData *VNCTextureGenerator::getTextureData()
{
    Guard<Mutex> guard(_mutex);
    Rectangle r; 
    
    // Initialize the update rectangle to the entire framebuffer
    r.x = 0; r.y = 0;
    r.w = _width; r.h = _height;

    // Build the update and return it
    return buildUpdate(r);
}

CORBA::Boolean VNCTextureGenerator::getUpdate(BinaryData_out texData,
					      Rectangle &rect)
{
    Guard<Mutex> guard(_mutex);
    WindowingSystemInterface::Rectangle r;
    bool update = false;
    
    // Retrieve framebuffer updates
    while (_vnc->getFramebufferUpdate(r) == true)
	update = true;
    
    if (update == false)
	return false;

    // TODO: for now we are forcing complete screen refresh
    r.x = 0;
    r.y = 0;
    r.width = _vncWidth;
    r.height = _vncHeight;

    // Transfer the rectangle info from the rect struct used in WSInterface
    // into the rect struct passed around by CORBA.
    // TODO: Unify these to eliminate this copy.
    rect.x = r.x;
    rect.y = r.y;
    rect.w = _width;
    rect.h = _height;
    
    // Build the texture update
    texData = buildUpdate(rect);

    return true; 
}

VNCDesktop::VNCDesktop(const char *host, CORBA::Short port, const char *pwd)
    : _vnc(new VNCInterface(host, port, pwd)),
      _generator(new VNCTextureGenerator(_vnc))
{
    // TODO: throw exception if unable to connect to server
    
    // Create a thread to handle the VNC connection
    _vnc_thread = new Thread(_vnc);
    _vnc_thread->start();
}

VNCDesktop::~VNCDesktop()
{
    // Kill the VNC thread
    _vnc_thread->kill();
    
    // Delete objects
    delete _vnc_thread;
    delete _vnc;
}

DesktopController_ptr VNCDesktop::createController(Node_ptr body)
{
    Guard<Mutex> guard(_mutex);
    VNCController *controller = activate(new VNCController(_vnc));
    controller->setBody(body);
    return controller->_this();
}

TextureGenerator_ptr VNCDesktop::getTextureGenerator()
{
    Guard<Mutex> guard(_mutex);
    return _generator->_this();
}

CORBA::Long VNCDesktop::getWidth()
{
    return _vnc->getWidth();    
}

CORBA::Long VNCDesktop::getHeight()
{
    return _vnc->getWidth();    
}

VNCController::VNCController(VNCInterface *vnc)
    : _vnc(vnc) { }

VNCController::~VNCController() { }

void VNCController::setMapper(Mapper_ptr m)
{
    Guard<Mutex> guard(_mutex);
    _mapper = Nobel::Mapper::_duplicate(m);
}

Mapper_ptr VNCController::getMapper()
{
    Guard<Mutex> guard(_mutex);
    return Nobel::Mapper::_duplicate(_mapper);
}

bool VNCController::handleNodeEvents(const Nobel::Event &e)
{
    static int buttonMask = 0;
    
    // If we don't have a mapper, we can't do anything
    if (CORBA::is_nil(_mapper))
	return false;

    // Now, use the mapper to convert to texture and then screen
    // coordinates.
    TexCoord t = _mapper->map(e.pos);
    int x = int(t.u * _vnc->getWidth());
    int y = int((1.0 - t.v) * _vnc->getHeight());
    
    // See if we can handle this event
    switch (e.type) {

    case ButtonPress: 
	if (e.dev == 0)
	    _vnc->keyEvent(e.key, true);
	else {
	    buttonMask |= 1 << (e.key - 1);
	    _vnc->pointerEvent(x, y, buttonMask);
	}
	return true;

    case ButtonRelease:
	if (e.dev == 0)
	    _vnc->keyEvent(e.key, false);
 	else {
	    buttonMask &= ~(1 << (e.key - 1));
	    _vnc->pointerEvent(x, y, buttonMask);
	}
	return true;

    case Movement:
	_vnc->pointerEvent(x, y, buttonMask);
	return true;
	
    default: break;
    }
    
    // No, can't handle the event
    return false;
}
    
