/* cmd-redo.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "libfsutils/rmrf.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "libarch/local-cache.h"
#include "libarch/proj-tree-lint.h"
#include "libarch/undo.h"
#include "libarch/apply-changeset.h"
#include "commands/cmd.h"
#include "commands/redo.h"
#include "commands/version.h"


/* __STDC__ prototypes for static functions */
static void redo_callback (void * ign, char * fmt, va_list ap);



static t_uchar * usage = N_("[options] [changeset]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_keep, "k", "keep", 0, \
      N_("do not delete the patch")) \
  OP (opt_quiet, "q", "quiet", 0, \
      N_("no progress reports while computing changeset")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("Operate on project tree in DIR (default `.')")) \
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))

t_uchar arch_cmd_redo_help[] = N_("redo changes in project tree\n"

				"Apply CHANGESET to the project tree and then delete CHANGESET.\n"
				"\n"
				"If CHANGESET is not specified, the highest numbered ,,undo-N directory\n"
				"in the project tree root is used.\n"
				"\n"
				"If --keep is given, the changeset directory is not deleted.\n"
				"\n"
				"See also \"baz undo --help\" and \"baz apply-changeset --help\".\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_redo (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * dir = 0;
  t_uchar * patch = 0;
  int quiet = 0;
  int keep = 0;
  int escape_classes = arch_escape_classes;
  int exit_status = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_redo_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_keep:
          {
            keep = 1;
            break;
          }

        case opt_quiet:
          {
            quiet = 1;
            break;
          }

	case opt_unescaped:
	  {
	    escape_classes = 0;
	    break;
	  }
        }
    }

  if (argc != 1 && argc != 2)
    goto usage_error;

  if (argc == 2)
    patch = str_save (0, argv[1]);

  {
    arch_project_tree_t * tree = arch_project_tree_new (talloc_context, dir);

    if (!tree->root)
      {
        safe_printfmt (2, "%s: not in a project tree\n  dir: %s\n",
                       argv[0], dir);
        exit (2);
      }

    if (!patch)
      {
        patch = arch_latest_undo_changeset (0, tree->root);
        if (!patch)
          {
            safe_printfmt (2, "%s: no changeset from undo found\n  dir: %s\n",
                           argv[0], tree->root);
            exit (2);
          }
      }


    {
      struct arch_tree_lint_result * lint = 0;

      if (!quiet)
        safe_printfmt (1, "* linting the source tree\n");
      lint = arch_tree_lint (tree);
      if ((0 > arch_print_tree_lint_report ((quiet ? 2 : 1), lint, escape_classes)))
        {
          exit (1);
        }
    }

    {
      struct arch_apply_changeset_report * report;
      arch_apply_changeset_report_callback report_callback = NULL;
      void * report_thunk = NULL;

      if (!quiet)
        {
          report_callback = redo_callback;
          report_thunk = (void *)1;
        }

      report = arch_apply_changeset (patch, talloc_context, tree, arch_unspecified_id_tagging, arch_inventory_unrecognized, 0, 0, escape_classes, report_callback, report_thunk);
      exit_status = arch_conflicts_occurred (report);

      if (!keep)
        rmrf_file (patch);

      talloc_free (report);
    }

    arch_project_tree_delete (tree);
  }


  lim_free (0, dir);
  lim_free (0, patch);

  return exit_status;
}



static void
redo_callback (void * ign, char * fmt, va_list ap)
{
  safe_printfmt_va_list (1, fmt, ap);
  safe_flush (1);
}



/* tag: Tom Lord Thu Jun  5 00:34:33 2003 (cmd-redo-changes.c)
 */
