/* cmd-uncacherev.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "libarch/pristines.h"
#include "libarch/build-revision.h"
#include "commands/uncacherev.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] revision [dir]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'"))


t_uchar arch_cmd_uncacherev_help[] = N_("remove a cached full source tree from an archive\n"
                                      "Remove the cached form of REVISION from its archive.\n"
                                      "\n"
                                      "If REVISION is not specified, but the command is run from within\n"
                                      "a project tree, uncache the latest revision in the default\n"
                                      "version of that tree.\n"
                                      "\n"
                                      "Also see \"baz cacherev -H\".\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_uncacherev (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_uncacherev_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  default_archive = arch_my_default_archive (default_archive);

  {
    t_uchar * revision_spec;
    t_uchar * archive = 0;
    t_uchar * revision = 0;
    t_uchar * package = NULL;
    struct arch_archive * arch = 0;

    revision_spec = argv[1];

    arch = arch_archive_connect_writeable_branch (revision_spec, &package);
    if (!arch_valid_package_name (package, arch_maybe_archive, arch_req_patch_level, 1))
      {
        safe_printfmt (2, "%s: invalid revision spec (%s)\n",
                       argv[0], revision_spec);
        exit (2);
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, package);
    revision = arch_parse_package_name (arch_ret_non_archive, 0, package);

    {
      t_uchar * errstr;

      if (arch_archive_delete_cached (&errstr, arch, revision))
        {
          safe_printfmt (2, "%s: unable to delete cached revision of %s/%s (%s)\n",
                         argv[0], archive, revision, errstr);
          exit (1);
        }
    }

    arch_archive_close (arch);

    lim_free (0, archive);
    lim_free (0, revision);
    lim_free (0, package);
  }

  lim_free (0, default_archive);
  return 0;
}




/* tag: Tom Lord Thu May 29 23:16:23 2003 (uncacherev.c)
 */
