/* archive-location.c:
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2005 Canonical Limited
 *      Authors: Robert Collins <robert.collins@canonical.com>
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/fmt/cvt.h"
#include "hackerlab/char/str.h"
#include "debug.h"
#include "libarch/archive-location.h"

arch_archive_location_t *
arch_archive_location_new (t_uchar const * const url)
{
    arch_archive_location_t * result = talloc (NULL, arch_archive_location_t);
    result->url = talloc_strdup (result, url);
    result->disabled = 1;
    result->master = 1;
    result->readonly = 1;
    result->priority = 100;
    return result;
}

/**
 * \brief construct a archive_location from an inivalue
 * \param inivalue the value from the inifile parser
 * \return a new arch_archive_location_t
 */
arch_archive_location_t *
arch_archive_location_new_inivalue (t_uchar const * const inivalue)
{
    int line_index;
    rel_table split_line = rel_ws_split (inivalue);
    arch_archive_location_t *result = arch_archive_location_new (split_line[0][0]);
    /* set ini file defaults */
    result->disabled = 0;
    result->master = 0;
    result->readonly = 0;
    rel_for_each (split_line, line_index)
      {
        t_uchar *key;
        if (line_index == 0)
            /* the url */
            continue;
        if (!str_chr_index (split_line[line_index][0], '='))
            key =str_save (0, split_line[line_index][0]);
        else
            key = str_save_n (0, split_line[line_index][0], str_chr_index (split_line[line_index][0], '=') - split_line[line_index][0]);
        if (!str_casecmp("disabled", key))
            result->disabled = 2;
        if (!str_casecmp("master", key))
            result->master = 2;
        else if (!str_casecmp("readonly", key))
            result->readonly = 2;
        else
          {
            if (!str_casecmp("priority", key))
              {
                t_uchar * value = split_line[line_index][0] + str_length (key) + 1;
                int ign;
                int priority;
                if (cvt_decimal_to_int (&ign, &priority, value, str_length (value)))
                  {
                    priority = result->priority;
                    debug (dbg_archives, 1, "Invalid priority specification in '%s'\n", inivalue);
                  }
                result->priority = priority;
              }
          }
        lim_free (0, key);
      }
    rel_free_table (split_line);
    return result;
}

/**
 * \brief get a suitable value string for urls in archive inifiles
 */
t_uchar *
arch_archive_location_to_ini_key(arch_archive_location_t const * const location)
{
    t_uchar * result = NULL;
    result = str_save (0, location->url);
    if (location->master == 2)
        result = str_replace (result, str_alloc_cat (0, result, " master"));
    if (location->readonly == 2)
        result = str_replace (result, str_alloc_cat (0, result, " readonly"));
    if (location->priority != 100)
      {
        t_uchar buf[64]=" priority=";
        cvt_long_to_decimal (&buf[10], location->priority);
        result = str_replace (result, str_alloc_cat (0, result, buf));
      }
    return result;
}
