#!/bin/sh
# -*- mode: sh; coding: utf-8 -*-
# Test archive registration
# Copyright © 2003 Colin Walters <walters@verbum.org>
# Copyright (C) 2005 Canonical Limited
#	Authors: Robert Collins <robert.collins@canonical.com>
#
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.

set -e

arg0="$0"
srcdir=`dirname "$arg0"`

. ${srcdir}/test-framework

setup_with_trivial_archives

test_class "Archive registration"

begin_test "Registration with no name given"
tla register-archive -d jane@example.com--2003
file_matches jane-archive ~/.arch-params/archives/jane@example.com--2003 && test_fail "jane@example.com--2003 is still in the new registry"
tla register-archive ${ARCHIVEDIR}/jane-archive
test $(tla whereis-archive jane@example.com--2003) = ${ARCHIVEDIR}/jane-archive""
file_matches "jane-archive master" ~/.arch-params/archives/jane@example.com--2003 || test_fail "jane@example.com--2003 is not in the new registry"
end_test 

begin_test "Registration with spaces is escaped in the registry"
tla register-archive -d jane@example.com--2003
file_matches jane-archive ~/.arch-params/archives/jane@example.com--2003 && test_fail "jane@example.com--2003 is still in the new registry"
cp -pR ${ARCHIVEDIR}/jane-archive "${ARCHIVEDIR}/jane archive"
tla register-archive "${ARCHIVEDIR}/jane archive"
tla whereis-archive jane@example.com--2003 > ,,whereis
file_matches "^${ARCHIVEDIR}/jane archive$" ,,whereis || test_fail "archive misregistered"
file_matches "jane%20archive" ~/.arch-params/archives/jane@example.com--2003 || test_fail "jane@example.com--2003 is not in the new registry"
end_test 

begin_test "Registration with archive name given fails"
tla register-archive -d jane@example.com--2003
tla register-archive jane@example.com--2003 ${ARCHIVEDIR}/jane-archive && test_fail "registered archive with a name parameter"
end_test 

begin_test "Archive unregistration"
tla register-archive -f -d jane@example.com--2003
tla whereis-archive jane@example.com--2003 2>/dev/null && test_fail "archive still registered"
file_matches jane-archive ~/.arch-params/archives/jane@example.com--2003 && test_fail "jane@example.com--2003 is in the new registry"
end_test

begin_test "Registration with no .arch-params"
rm -fr $HOME/.arch-params && test ! -d $HOME/.arch-params
tla register-archive ${CACHEDARCHIVEDIR}/jane-archive || test_fail "failed to register the archive"
test $(tla whereis-archive jane@example.com--2003) = ${CACHEDARCHIVEDIR}/jane-archive"" || test_fail "misregistered the archive"
file_matches jane-archive ~/.arch-params/archives/jane@example.com--2003 || test_fail "jane@example.com--2003 is not in the new registry"
end_test 

begin_test "Archive unregistration by url"
tla register-archive -d ${ARCHIVEDIR}/jane-archive
tla whereis-archive jane@example.com--2003 2>/dev/null && test_fail "jane@example.com--2003 is still registered"
end_test

begin_test "Archive unregistration by url and supplied name"
echo "url=${CACHEDARCHIVEDIR}/jane-archivemissing" > ~/.arch-params/archives/jane@example.com--2003
tla register-archive -d ${ARCHIVEDIR}/jane-archivemissing jane@example.com--2003
tla whereis-archive jane@example.com--2003 2>/dev/null && test_fail "jane@example.com--2003 is still registered"
file_matches jane-archivemissing ~/.arch-params/archives/jane@example.com--2003 && test_fail "jane@example.com--2003 is still registered"
end_test

begin_test "whereis-archive --all-locations skips disabled archives"
echo "url=${CACHEDARCHIVEDIR}/jane-archive disabled
url=${ARCHIVEDIR}/jane-archive-mirror" > ~/.arch-params/archives/jane@example.com--2003
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive" ,,locations && test_fail "jane@example.com--2003 disabled listed"
file_matches "${ARCHIVEDIR}/jane-archive-mirror" ,,locations || test_fail "jane@example.com--2003 mirror not listed"
end_test

begin_test "whereis-archive --all-locations"
echo "url=${CACHEDARCHIVEDIR}/jane-archive
url=${ARCHIVEDIR}/jane-archive-mirror" > ~/.arch-params/archives/jane@example.com--2003
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
file_matches "${ARCHIVEDIR}/jane-archive-mirror" ,,locations || test_fail "jane@example.com--2003 mirror not listed"
end_test

begin_test "archives --all-locations"
echo "url=${CACHEDARCHIVEDIR}/jane-archive
url=${ARCHIVEDIR}/jane-archive-mirror" > ~/.arch-params/archives/jane@example.com--2003
mkdir -p ~/.arch-params/\=locations
echo "${ARCHIVEDIR}/foo-archive" > ~/.arch-params/\=locations/foo@example.org--2003
tla archives --all-locations > ,,archives
file_matches '^foo@example.org--2003$' ,,archives || test_fail "foo@example.org--2003 not listed"
file_matches "^    ${ARCHIVEDIR}/foo-archive$" ,,archives || test_fail "foo-archive location not listed"
file_matches '^jane@example.com--2003$' ,,archives || test_fail "jane@example.com--2003 not listed"
file_matches "^    ${CACHEDARCHIVEDIR}/jane-archive$" ,,archives || test_fail "jane-archive location not listed"
file_matches "^    ${ARCHIVEDIR}/jane-archive-mirror$" ,,archives || test_fail "jane-archive-mirror location not listed"
end_test

begin_test "abrowse name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla abrowse jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "abrowse name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla abrowse jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "abrowse url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla abrowse ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "ancestry-graph name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla ancestry-graph jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "ancestry-graph name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla ancestry-graph jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "ancestry-graph url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla ancestry-graph ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "archive-fixup name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla archive-fixup jane@example.com--2003
end_test

begin_test "archive-fixup name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-fixup jane@example.com--2003
end_test

begin_test "archive-fixup url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-fixup ${CACHEDARCHIVEDIR}/jane-archive
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "archive-snapshot name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
mkdir snap
tla archive-snapshot snap jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "archive-snapshot name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
mkdir snap
tla archive-snapshot snap jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "archive-snapshot url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
mkdir snap
tla archive-snapshot snap ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "archive-version name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla archive-version jane@example.com--2003
end_test

begin_test "archive-version name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-version jane@example.com--2003
end_test

begin_test "archive-version url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-version ${CACHEDARCHIVEDIR}/jane-archive
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "branches name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla branches jane@example.com--2003/hello-world
end_test

begin_test "branches name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla branches jane@example.com--2003/hello-world
end_test

begin_test "branches url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla branches ${CACHEDARCHIVEDIR}/jane-archive/hello-world
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "cat-archive-log name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla cat-archive-log jane@example.com--2003/hello-world--mainline--1.0--base-0
end_test

begin_test "cat-archive-log name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla cat-archive-log jane@example.com--2003/hello-world--mainline--1.0--base-0
end_test

begin_test "cat-archive-log url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla cat-archive-log ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "diff name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd  foo
tla diff jane@example.com--2003/hello-world--mainline--1.0
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "diff name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla diff jane@example.com--2003/hello-world--mainline--1.0
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "diff url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
cd foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla diff ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "categories name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla categories jane@example.com--2003
end_test

begin_test "categories name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla categories jane@example.com--2003
end_test

begin_test "categories url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla categories ${CACHEDARCHIVEDIR}/jane-archive
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "export name (version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla export jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "export name (old) (version)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla export jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "export url (version)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla export ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "export name (revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla export jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "export name (old) (revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla export jane@example.com--2003/hello-world--mainline--1.0--base-0
end_test

begin_test "export url (revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla export ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

# TODO RBC 20050329 fix grab to just connect to to the location, rather than munging the path
##begin_test "grab name"
##echo "Archive-Name: jane@example.com--2003
##Archive-Location: ${CACHEDARCHIVEDIR}/jane-archive
##Target-Revision: hello-world--mainline--1.0" > grab-file
##echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
##rm -f ~/.arch-params/=locations/jane@example.com--2003
##tla grab grab-file
##end_test
##
##begin_test "grab name (old)"
##echo "Archive-Name: jane@example.com--2003
##Archive-Location: ${CACHEDARCHIVEDIR}/jane-archive
##Target-Revision: hello-world--mainline--1.0" > grab-file
##echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
##rm -f ~/.arch-params/archives/jane@example.com--2003
##tla grab grab-file
##end_test
##
##begin_test "grab url"
##echo "Archive-Name: jane@example.com--2003
##Archive-Location: ${CACHEDARCHIVEDIR}/jane-archive
##Target-Revision: hello-world--mainline--1.0" > grab-file
##rm -f ~/.arch-params/=locations/jane@example.com--2003
##rm -f ~/.arch-params/archives/jane@example.com--2003
##tla grab grab-file
### should have registered it
##tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
##file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
##end_test
##

begin_test "library-add name (revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
mkdir revlib
tla my-revision-library $(pwd)/revlib
tla library-add jane@example.com--2003/hello-world--mainline--1.0
tla my-revision-library -d $(pwd)/revlib
end_test

begin_test "library-add name (old) (revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
mkdir revlib
tla my-revision-library $(pwd)/revlib
tla library-add jane@example.com--2003/hello-world--mainline--1.0--base-0
tla my-revision-library -d $(pwd)/revlib
end_test

begin_test "library-add url (revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
mkdir revlib
tla my-revision-library $(pwd)/revlib
tla library-add ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
tla my-revision-library -d $(pwd)/revlib
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "merges name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd  foo
tla merges jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--mainline--1.0
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "merges name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla merges jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--mainline--1.0
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "merges url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
cd foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla merges ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "revisions name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla revisions jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "revisions name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla revisions jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "revisions url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla revisions ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "versions name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla versions jane@example.com--2003/hello-world--mainline
end_test

begin_test "versions name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla versions jane@example.com--2003/hello-world--mainline
end_test

begin_test "versions url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla versions ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "get name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "get name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "get w/default-archive name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla my-default-archive jane@example.com--2003
tla get hello-world--mainline--1.0
tla my-default-archive -d
end_test

begin_test "get url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "get url with space"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane\ archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "lock-revision name (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla my-id 'Jane Doe <jane@example.com>'
tla lock-revision jane@example.com--2003/hello-world--mainline--1.0
tla lock-revision -b jane@example.com--2003/hello-world--mainline--1.0
end_test 

begin_test "lock-revision name (old, version)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla lock-revision jane@example.com--2003/hello-world--mainline--1.0
tla lock-revision -b jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "lock-revision name (new, revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla lock-revision jane@example.com--2003/hello-world--mainline--1.0--patch-1
tla lock-revision -b jane@example.com--2003/hello-world--mainline--1.0--patch-1
end_test 

begin_test "lock-revision name (old, revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla lock-revision jane@example.com--2003/hello-world--mainline--1.0--patch-1
tla lock-revision -b jane@example.com--2003/hello-world--mainline--1.0--patch-1
end_test

begin_test "lock-revision name (url, version)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla lock-revision ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
tla lock-revision -b ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "lock-revision name (new, revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla lock-revision ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-1
tla lock-revision -b ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-1
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test 

begin_test "make-archive -m name url (new)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "make-archive -m name url (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "make-archive -m url url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "make-archive -M name url (new)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror-from jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "make-archive -M name url (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror-from jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "make-archive -M url url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create the second mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror2" ,,locations ||  test_fail "mirror2 not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
test -f ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror || test_fail "mirror2 flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror2/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror2 file has incorrect contents"
end_test

begin_test "archive_mirror url name"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror ${CACHEDARCHIVEDIR}/jane-archive jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror url url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror ${CACHEDARCHIVEDIR}/jane-archive ${ARCHIVEDIR}/jane-archive-mirror || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror name url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror jane@example.com--2003 ${ARCHIVEDIR}/jane-archive-mirror || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror name name"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror jane@example.com--2003 jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror name hello-world"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror jane@example.com--2003 hello-world || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror url hello-world"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror ${ARCHIVEDIR}/jane-archive-mirror hello-world || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror name (new)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "archive_mirror name (old, push)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive-mirror" > ~/.arch-params/=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "archive_mirror name (old, pull)"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003-SOURCE
echo "${CACHEDARCHIVEDIR}/jane-archive-mirror" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "archive_mirror name (non master pull)"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror jane@example.com--2003 || test_fail "seed mirror run failed"
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create second mirror"
echo "url=${CACHEDARCHIVEDIR}/jane-archive-mirror readonly
url=${CACHEDARCHIVEDIR}/jane-archive-mirror2" > ~/.arch-params/archives/jane@example.com--2003
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror2/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "archive_mirror name (all)"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror2 || test_fail "failed to create second mirror"
tla archive-mirror --all-mirrors jane@example.com--2003 || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror2/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
rm -f ~/.arch-params/\=locations/jane@example.com--2003-SOURCE
rm -f ~/.arch-params/\=locations/jane@example.com--2003-MIRROR
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
end_test

begin_test "archive_mirror url hello-world"
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror
rm -rf  ${ARCHIVEDIR}/jane-archive-mirror2
tla make-archive --mirror-from ${CACHEDARCHIVEDIR}/jane-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla archive-mirror ${ARCHIVEDIR}/jane-archive-mirror || test_fail "mirror run failed"
diff ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/checksum ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/checksum || test_fail "did not mirror to the local mirror"
end_test

begin_test "branch name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd  foo
tla branch jane@example.com--2003/hello-world--mainline--1.1
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
end_test

begin_test "branch name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla branch jane@example.com--2003/hello-world--mainline--1.1
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -rf ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
end_test

begin_test "branch url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
cd foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla branch ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
rm -rf ${CACHEDIR}/archives
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
# used by switch tests rm -rf ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
end_test

begin_test "branch url url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla branch ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1 ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.2
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.2 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "branch name(old) url"
rm -f ~/.arch-params/archives/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
tla branch jane@example.com--2003/hello-world--mainline--1.1 ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "branch name(new) url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
tla branch jane@example.com--2003/hello-world--mainline--1.1 ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "branch url name"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla branch ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1 jane@example.com--2003/hello-world--mainline--1.4
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.4 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "branch name(old) name(old)"
rm -f ~/.arch-params/archives/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
tla branch jane@example.com--2003/hello-world--mainline--1.1 jane@example.com--2003/hello-world--mainline--1.4
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "branch name(new) name(new)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
tla branch jane@example.com--2003/hello-world--mainline--1.1 jane@example.com--2003/hello-world--mainline--1.5
[ -d ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.3 ] || test_fail "Branch failed"
tla whereis-archive jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
end_test

begin_test "switch name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd  foo
tla switch jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "switch name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla switch jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "switch url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
cd foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla switch ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

# TODO switch-revision tests

begin_test "merge name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd  foo
tla merge jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "merge name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla merge jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "merge url"
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
cd foo
# merge requires the current branch to be registered, though the source can be a url.
tla merge ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
end_test

begin_test "sync-tree name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla branch jane@example.com--2003/hello-world--mainline--1.0--base-0 foo@example.org--2003/hello-world--mainline--1.0
tla get foo@example.org--2003/hello-world--mainline--1.0 foo
cd  foo
tla sync-tree foo@example.org--2003/hello-world--mainline--1.0--patch-1 && test_fail "synced with non existant revision"
tla sync-tree jane@example.com--2003/hello-world--mainline--1.3--patch-1 
end_test

begin_test "sync-tree name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla sync-tree jane@example.com--2003/hello-world--mainline--1.1--base-0
end_test

begin_test "sync-tree url"
tla get ${CACHEDARCHIVEDIR}/foo-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
cd foo
tla sync-tree ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.3--patch-1
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "delta tree name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla delta foo jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "delta tree name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla delta foo jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "delta tree url"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla delta foo ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "delta name tree"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla delta jane@example.com--2003/hello-world--mainline--1.1 foo
end_test

begin_test "delta name (old) tree"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla delta jane@example.com--2003/hello-world--mainline--1.1 foo
end_test

begin_test "delta url tree"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla delta ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0 foo
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "delta tree tree"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla delta foo foo
end_test

begin_test "delta name name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla delta jane@example.com--2003/hello-world--mainline--1.{0,1}
end_test

begin_test "delta url url"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla delta ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0 ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "apply-delta tree name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo foo jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "apply-delta tree name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo foo jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "apply-delta tree url"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla apply-delta -d foo foo ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "apply-delta name tree"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo jane@example.com--2003/hello-world--mainline--1.1 foo
end_test

begin_test "apply-delta name (old) tree"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo jane@example.com--2003/hello-world--mainline--1.1 foo
end_test

begin_test "apply-delta url tree"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla apply-delta -d foo ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0 foo
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "apply-delta tree tree"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo foo foo
end_test

begin_test "apply-delta name name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
tla apply-delta -d foo jane@example.com--2003/hello-world--mainline--1.{0,1}
end_test

begin_test "apply-delta url url"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla apply-delta -d foo ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.1--base-0 ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

#TODO merge name-revision, url-revision.

# Test cases to be written that involve the archive registry.
#  (not meant to test the commands behaviour, just that it works with these input commands )
# whereis-archive --all-locations --sources name
# whereis-archive --all-locations --mirrors name
# whereis-archive --all-locations --master name
# archives --sources
# archives --mirrors
# archives --masters
# build-config magicp-path, relpath, name+path, url+path
# cat-config as per build-config

# FIXME: need to check build-config with the release-id


begin_test "build-config -d foo path-in-foo (new)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
cat >> foo/config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
tla build-config -d foo config
end_test


begin_test "build-config contents: path + (new)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
cd foo
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
tla build-config config
end_test

begin_test "build-config contents: path + (url)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
cd foo
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
tla build-config config
end_test

begin_test "build-config contents: path + (old)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
cd foo
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
tla build-config config
end_test

begin_test "build-config contents: (magicp .)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
cd foo
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
tla build-config config
end_test

begin_test "build-config contents: ('config' which is in configs/)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
cd foo
mkdir configs && cd configs
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
cd ..
tla build-config config
end_test

begin_test "build-config contents: (magicp configs/ when in subdir)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
cd foo
mkdir configs && cd configs
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF
cd ..
mkdir testing && cd testing
tla build-config config
end_test

begin_test "build-config config out of project tree"
rm -rf /tmp-baztest
mkdir /tmp/baztest && cd /tmp/baztest
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003

cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF

tla build-config config
cd /
rm -rf /tmp/baztest
end_test

begin_test "build-config file://config"
rm -rf /tmp-baztest
mkdir /tmp/baztest && cd /tmp/baztest
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
cat >> config << EOF
./src ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
EOF

tla build-config file:///tmp/baztest/config
rm -rf /tmp/baztest
end_test

begin_test "commit (new, url)"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
cd foo
tla commit -s"empty" -b "${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0"
end_test

begin_test "commit (new, name)"
tla get ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
cd foo
tla commit -s"empty" -b jane@example.com--2003/hello-world--mainline--1.0 
end_test

begin_test "commit (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0 foo
cd foo
tla commit -s"empty"
tla commit -s"also empty"
end_test

begin_test "cachedrevs (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
[ "$(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0)" = "hello-world--mainline--1.0--patch-1" ] || test_fail "cachedrevs got confused (got $(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0))"
end_test

begin_test "cachedrevs (new, url)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0)" = "hello-world--mainline--1.0--patch-1" ] || test_fail "cachedrevs got confused"
end_test

begin_test "cachedrevs (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
[ "$(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0)" = "hello-world--mainline--1.0--patch-1" ] || test_fail "cachedrevs got confused"
end_test

begin_test "cacherev (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla cacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodgy
[ "$(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0 | tr '\n' ' '))"
tla uncacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
end_test

begin_test "cacherev (new, url)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla cacherev ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodgy
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')"
# manual because uncacherev is untested as yet
rm -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/patch-2/hello-world--mainline--1.0--patch-2.tar.gz
rm -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/patch-2/checksum.cacherev
end_test

begin_test "cacherev (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla cacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodgy
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')"
rm -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/patch-2/hello-world--mainline--1.0--patch-2.tar.gz
rm -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/patch-2/checksum.cacherev
end_test

begin_test "uncacherev (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla cacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodgy
[ "$(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs jane@example.com--2003/hello-world--mainline--1.0 | tr '\n' ' '))"
tla uncacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
end_test

begin_test "uncacherev (new, url)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla cacherev ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodge
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')"
# manual because uncacherev is untested as yet
tla uncacherev ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-2
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0)" = "hello-world--mainline--1.0--patch-1" ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0)"
end_test

begin_test "uncacherev (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla cacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
# FIXME, tr is dodge
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')" = "hello-world--mainline--1.0--patch-1 hello-world--mainline--1.0--patch-2 " ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0 | tr '\n' ' ')"
tla uncacherev jane@example.com--2003/hello-world--mainline--1.0--patch-2
[ "$(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0)" = "hello-world--mainline--1.0--patch-1" ] || test_fail "cachedrevs got confused (got $(tla cachedrevs ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0)"
end_test

begin_test "replay name (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay jane@example.com--2003/hello-world--mainline--1.0
end_test 

begin_test "replay name (old, version)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "replay name (url, version)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "replay name (new, revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay jane@example.com--2003/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 && test_fail "patch-1 exists when it shouldn't."
end_test 

begin_test "replay name (old, revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay jane@example.com--2003/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 && test_fail "patch-1 exists when it shouldn't."
end_test

begin_test "replay name (url, revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
cd foo
tla replay ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 && test_fail "patch-1 exists when it shouldn't."
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "update name (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla update jane@example.com--2003/hello-world--mainline--1.0
end_test 

begin_test "update name (old, version)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla update jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "update name (url, version)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
cd foo
tla update ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "update name (new, revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla update jane@example.com--2003/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 || test_fail "patch-1 is missing."
end_test 

begin_test "update name (old, revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla update jane@example.com--2003/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 || test_fail "patch-1 is missing."
end_test

begin_test "update name (url, revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
cd foo
tla update ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-2
tla cat-log patch-1 || test_fail "patch-1 is missing."
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "missing (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
expected="jane@example.com--2003/hello-world--mainline--1.1--base-0"
got=$(tla missing jane@example.com--2003/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "missing failed (expected $expected, got $got)"
end_test

begin_test "missing (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
expected="jane@example.com--2003/hello-world--mainline--1.1--base-0"
got=$(tla missing jane@example.com--2003/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "missing failed (expected $expected, got $got)"
end_test 

begin_test "missing (new, url)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
cd foo
expected="jane@example.com--2003/hello-world--mainline--1.1--base-0"
got=$(tla missing ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "missing failed (expected $expected, got $got)"
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "join-branch (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla join-branch jane@example.com--2003/hello-world--mainline--1.1
end_test

begin_test "join-branch (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo
cd foo
tla join-branch jane@example.com--2003/hello-world--mainline--1.1
end_test 

begin_test "join-branch (new, url)"
tla get ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--base-0 foo
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
cd foo
tla join-branch ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "archive-meta-info name (new, revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla archive-meta-info jane@example.com--2003/name
end_test 

begin_test "archive-meta-info name (old, revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-meta-info jane@example.com--2003/name
end_test

begin_test "archive-meta-info name (url, revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla archive-meta-info ${ARCHIVEDIR}/jane-archive/name
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "rbrowse name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla rbrowse jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "rbrowse name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla rbrowse jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "rbrowse url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla rbrowse ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "change-archive name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive master" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla change-archive --add-signatures jane@example.com--2003
tla change-archive --remove-signatures jane@example.com--2003
end_test

begin_test "change-archive name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
mkdir -p ~/.arch-params/signing
echo "true" > $HOME/.arch-params/signing/\=default.check
echo "gpg --clearsign" > $HOME/.arch-params/signing/\=default
tla change-archive --add-signatures jane@example.com--2003
tla change-archive --remove-signatures jane@example.com--2003
end_test

begin_test "change-archive url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla change-archive --add-signatures ${CACHEDARCHIVEDIR}/jane-archive
tla change-archive --remove-signatures ${CACHEDARCHIVEDIR}/jane-archive
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

begin_test "ancestry name"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla ancestry jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "ancestry name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla ancestry jane@example.com--2003/hello-world--mainline--1.0
end_test

begin_test "ancestry url"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla ancestry ${CACHEDARCHIVEDIR}/jane-archive/hello-world--mainline--1.0
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${CACHEDARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test


begin_test "parse-package-name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
expected="hello-world--mainline--1.1"
got=$(tla parse-package-name --non-arch jane@example.com--2003/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "missing failed (expected $expected, got $got)"
end_test

begin_test "parse-package-name (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
expected="hello-world--mainline--1.1"
got=$(tla parse-package-name --non-arch jane@example.com--2003/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "parse-package-name failed (expected $expected, got $got)"
end_test 

begin_test "parse-package-name (new, url)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
expected="hello-world--mainline--1.1"
got=$(tla parse-package-name --non-arch ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1)
[ "$expected" = "$got" ] || test_fail "parse-package-name failed (expected $expected, got $got)"
end_test

begin_test "valid-package-name (old)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla valid-package-name -v jane@example.com--2003/hello-world--mainline--1.1 || test_fail "Invalid version"
end_test

begin_test "valid-package-name (new, version)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla valid-package-name -v jane@example.com--2003/hello-world--mainline--1.1 || test_file "Invalid version"
end_test 

begin_test "valid-package-name (new, url)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla valid-package-name -v ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.1 || test_fail "Invalid version"
end_test

begin_test "get-changeset name (new, revision)"
echo "url=${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/archives/jane@example.com--2003
rm -f ~/.arch-params/=locations/jane@example.com--2003
tla get-changeset jane@example.com--2003/hello-world--mainline--1.0--patch-1 
end_test 

begin_test "get-changeset name (old, revision)"
echo "${CACHEDARCHIVEDIR}/jane-archive" > ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get-changeset jane@example.com--2003/hello-world--mainline--1.0--patch-1 
end_test

begin_test "get-changeset name (url, revision)"
rm -f ~/.arch-params/=locations/jane@example.com--2003
rm -f ~/.arch-params/archives/jane@example.com--2003
tla get-changeset ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0--patch-1 
# should have registered it
tla whereis-archive --all-locations jane@example.com--2003 >,,locations || test_fail "jane@example.com--2003 is not registered"
file_matches "^${ARCHIVEDIR}/jane-archive" ,,locations || test_fail "jane@example.com--2003 master not listed"
end_test

clean_workdir

# tag: Colin Walters Tue, 16 Sep 2003 20:35:26 -0400 (test-register-archive.sh)
#
