/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "books.h"
#include "directories.h"
#include "gwrappers.h"
#include "sqlite3.h"
#include "sqlite_reader.h"
#include "utilities.h"
#include <config.h>
#include "shell.h"


#define BOOKS_SQL "books.sql4"


ustring books_filename (const ustring& project)
{
  return gw_build_filename (directories_get_projects (), project, BOOKS_SQL);
}


ustring books_filename_template ()
/*
A note about this database.
The id should always be a number higher than 0, because 0 is taken for "not
found".
The id is connected to a book, and is used in the notes database too.
Therefore, ids are not supposed to be changed; new ones can be added though.
*/
{
  return gw_build_filename (PACKAGE_DATA_DIR, BOOKS_SQL);
}


void books_create (const ustring& project)
/*
This creates a books database in a project, derived from the template.
*/
{
  // Filenames.
  ustring templatefilename = books_filename_template ();
  ustring projectfilename = books_filename (project);
  // Data.
  vector <ustring> names;
  vector <ustring> sequences;
  vector <ustring> ids;
  // Read relevant data from the template.
  {
    sqlite3 *db;
    int rc;
    char *error = NULL;
    try
    {
      rc = sqlite3_open(templatefilename.c_str(), &db);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
      SqliteReader sqlitereader (0);
      char * sql;
      sql = g_strdup_printf ("select name, sequence, id from books;");
      rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      names.assign (sqlitereader.ustring0.begin(), sqlitereader.ustring0.end());
      sequences.assign (sqlitereader.ustring1.begin(), sqlitereader.ustring1.end());
      ids.assign (sqlitereader.ustring2.begin(), sqlitereader.ustring2.end());
    }
    catch (exception & ex)
    {
      cerr << ex.what () << endl;
    }
    sqlite3_close (db);
  }
  // Write this data to the project's database. 
  {
    sqlite3 *db;
    int rc;
    char *error = NULL;
    try
    {
      rc = sqlite3_open(projectfilename.c_str(), &db);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
      char * sql;
      sql = g_strdup_printf ("CREATE TABLE books (name text, namecf text, sequence integer, id integer);");
      rc = sqlite3_exec(db, sql, NULL, NULL, &error);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      for (unsigned int i = 0; i < names.size(); i++) {
        sql = g_strdup_printf ("insert into books values ('%s', '%s', '%s', '%s');", 
          names[i].c_str(), names[i].casefold().c_str(), sequences[i].c_str(), ids[i].c_str());
        rc = sqlite3_exec(db, sql, NULL, NULL, &error);
        if (rc != SQLITE_OK) {
          throw runtime_error (error);
        }
      }
      g_free (sql);
    }
    catch (exception & ex)
    {
      cerr << ex.what () << endl;
    }
    sqlite3_close (db);
  }
}


void books_order (const ustring& project, vector<ustring>& books)
// Books read from the database usually are out of order. Reorder the books
// in agreement with the user's settings.
{
  // Storage for sequence.
  vector<unsigned int> sequences;
  // Database variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Open database.
    rc = sqlite3_open(books_filename (project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Go through all the books we have.
    for (unsigned int i = 0; i < books.size(); i++) {
      // Read the sequence of the book.
      SqliteReader sqlitereader (0);
      char * sql;
      sql = g_strdup_printf ("select sequence from books where name = '%s';", books[i].c_str());
      rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      // In case the book doesn't exist, reorder it to the end of the list.
      unsigned int sequence = 1000;
      if (sqlitereader.ustring0.size() > 0)
        sequence = convert_to_int (sqlitereader.ustring0[0]);
      sequences.push_back (sequence);      
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
  // Sort books on sequence.
  quick_sort (sequences, books, 0, sequences.size());
}


unsigned int books_name_to_id (const ustring& project, const ustring& book)
/*
This returns the id of "book" in "project". 

The id is Bibledit's internal id for a given book. 
This id uniquely identifies the book. 

We could have identified books by the Paratext ID, or by the OSIS name, but as 
there aren't Paratext IDs or OSIS names for every book that Bibledit supports, 
we need to have our own id.
*/
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename (project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where namecf = '%s';", book.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


ustring books_id_to_name  (const ustring& project, unsigned int id)
// Returns the human readable bookname from the Bibledit id.
{
  ustring name;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename (project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select name from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      name = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return name;
}


ustring books_id_to_paratext (unsigned int id)
{
  ustring paratext;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select paratext from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      paratext = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return paratext;
}


unsigned int books_paratext_to_id (const ustring& paratext)
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where paratextcf = '%s';", paratext.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


ustring books_id_to_bibleworks (unsigned int id)
{
  ustring bibleworks;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select bibleworks from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      bibleworks = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return bibleworks;
}


unsigned int books_bibleworks_to_id (const ustring& bibleworks)
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where bibleworkscf = '%s';", bibleworks.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


ustring books_id_to_osis (unsigned int id)
{
  ustring osis;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select osis from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      osis = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return osis;
}


unsigned int books_osis_to_id (const ustring& osis)
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where osiscf = '%s';", osis.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


ustring books_id_to_english (unsigned int id)
{
  ustring english;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select name from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      english = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return english;
}


unsigned int books_english_to_id (const ustring& english)
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(books_filename_template().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where namecf = '%s';", english.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


void book_rename (const ustring& oldname, const ustring& newname)
{
}
