/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "stylesheetutils.h"
#include "directories.h"
#include <config.h>
#include "constants.h"
#include "xmlutils.h"
#include "gwrappers.h"
#include "gtkwrappers.h"
#include "shell.h"
#include "sqlite3.h"
#include "sqlite_reader.h"
#include "vacuum.h"


#define STYLESHEET_SUFFIX ".sql9"
char *RECOGNIZED_SUFFIXES [] = { ".sql4", ".sql5", ".sql6", ".sql7", ".sql8", ".sql9" };


ustring stylesheet_filename (const ustring& name)
// This returns the database's filename for a named stylesheet.
{
  return gw_build_filename (directories_get_stylesheets (), name + STYLESHEET_SUFFIX);
}


ustring stylesheet_template_filename ()
// This returns the database's filename of the template.
{
  return gw_build_filename (PACKAGE_DATA_DIR, "stylesheet.sql");
}


void stylesheet_get_ones_available (vector<ustring>& names)
// Gets the names of the stylesheets that are there.
{
  ReadFiles rf (directories_get_stylesheets (), "", STYLESHEET_SUFFIX);
  for (unsigned int i = 0; i < rf.files.size(); i++)
    rf.files[i].erase (rf.files[i].length() - strlen (STYLESHEET_SUFFIX), strlen (STYLESHEET_SUFFIX));
  names.assign (rf.files.begin(), rf.files.end());
}


void stylesheet_create_new (const ustring& name)
// Create a new stylesheet, named "name", from the template.
{
  ustring templatefile = stylesheet_template_filename ();
  ustring stylefile = gw_build_filename (directories_get_stylesheets (), name + STYLESHEET_SUFFIX);
  copy_file (templatefile, stylefile);  
}


void stylesheet_delete (const ustring& name)
// Deletes a stylesheet.
{
  unlink (stylesheet_filename (name).c_str());
}


void stylesheet_copy (const ustring& from_name, const ustring& to_name)
// Copies one stylesheet to another.
{
  ustring command = "cp";
  command.append (shell_quote_space (stylesheet_filename (from_name)));
  command.append (shell_quote_space (stylesheet_filename (to_name)));
  system (command.c_str());
}


ustring stylesheet_import (const ustring& filename)
// Imports a new stylesheet from "filename".
// It expects a file in the formst as it is given in the export function.
{
  // See that the sheet is a recognized one.
  // Derive the name of the new stylesheet from the filename.
  ustring name;
  for (unsigned int i = 0; i < (sizeof (RECOGNIZED_SUFFIXES) / sizeof (*RECOGNIZED_SUFFIXES)); i++) {
    if (g_str_has_suffix (filename.c_str(), RECOGNIZED_SUFFIXES[i])) {
      name = gw_path_get_basename (filename);
      name.erase (name.length() - strlen (RECOGNIZED_SUFFIXES[i]), strlen (RECOGNIZED_SUFFIXES[i]));
    }
  }
  if (name.empty()) {
    gtkw_dialog_error (NULL, filename + ": Unrecognized stylesheet");
    return "";
  }
  // Check whether it already exists.
  if (stylesheet_exists (name)) {
    gtkw_dialog_error (NULL, "Stylesheet " + name +  " already exists");
    return "";
  }
  // Get the path of the new stylesheet.
  ustring path = stylesheet_filename (name);
  // Copy the database.
  copy_file (filename, path);
  // Upgrade the stylesheet.
  stylesheets_upgrade ();
  // Return the name of the stylesheet we imported;
  return name;
}


void stylesheet_export (const ustring& name, const ustring& filename)
// Exports a stylesheet.
{
  ustring originalfile = stylesheet_filename (name);
  ustring destinationfile (filename);
  destinationfile.append (STYLESHEET_SUFFIX);
  copy_file (originalfile, destinationfile);
}


bool stylesheet_exists (const ustring& name)
// Returns whether this stylesheet exists.
{
  vector<ustring> sheets;
  stylesheet_get_ones_available (sheets);
  set<ustring> existing_sheets (sheets.begin(), sheets.end());
  return (existing_sheets.find (name) != existing_sheets.end());
}


void stylesheet_get_styles (const ustring& stylesheet, vector<Style>& styles)
// Get all data of all stylesheets.
{
  // Get available markers.
  vector<ustring> markers = stylesheet_get_markers (stylesheet, NULL);
  // Read all styles.
  for (unsigned int i = 0; i < markers.size(); i++) {
    Style style (stylesheet, markers[i], false);
    styles.push_back (style);
  }
}


vector<ustring> stylesheet_get_markers (const ustring& stylesheet, vector<ustring> * names)
/*
This function only gets the markers and the names of the styles of the stylesheet,
and is therefore faster than the similar function that gets the whole style.
It is intended to be used in such situation that only the markers and/or the
name of a style is needed, such as in the styletree.
*/
{
  // Store styles
  vector<ustring> markers;
  // Some variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Read the available markers.
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select marker, name from styles;");
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    markers.assign (reader.ustring0.begin(), reader.ustring0.end());
    if (names)
      names->assign (reader.ustring1.begin(), reader.ustring1.end());
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
  // Return result.
  return markers;
}


void stylesheet_delete_style (const ustring& stylesheet, const ustring& marker)
// Deletes a style.
{
  // Some variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Delete style.
    char * sql;
    sql = g_strdup_printf ("delete from styles where marker = '%s';", marker.c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
}


void stylesheet_new_style (const ustring& stylesheet, const ustring& marker)
// Adds a new style. Searches template for data.
{
  Style style (stylesheet, marker, true);
  style.read_template ();
}


void stylesheet_save_style (const ustring& stylesheet, const ustring& marker,
                            const ustring& name, const ustring& info,
                            StyleType type, int subtype,
                            double fontsize, int fontpercentage,
                            const ustring& italic, const ustring& bold, 
                            const ustring& underline, const ustring& smallcaps,
                            bool superscript, const ustring& justification,
                            double spacebefore, double spaceafter,
                            double leftmargin, double rightmargin,
                            double firstlineindent, bool spancolumns,
                            unsigned int color, bool print,
                            bool userbool1, bool userbool2, bool userbool3,
                            int userint1, int userint2, int userint3,
                            ustring userstring1, ustring userstring2, ustring userstring3)
// Saves style information.
{
  Style style (stylesheet, marker, true);
  style.name = name;
  style.info = info;
  style.type = type;
  style.subtype = subtype;
  style.fontsize = fontsize;
  style.fontpercentage = fontpercentage;
  style.italic = italic;
  style.bold = bold;
  style.underline = underline;
  style.smallcaps = smallcaps;
  style.superscript = superscript;
  style.justification = justification;
  style.spacebefore = spacebefore;
  style.spaceafter = spaceafter;
  style.leftmargin = leftmargin;
  style.rightmargin = rightmargin;
  style.firstlineindent = firstlineindent;
  style.spancolumns = spancolumns;
  style.color = color;
  style.print = print;
  style.userbool1 = userbool1;
  style.userbool2 = userbool2;
  style.userbool3 = userbool3;
  style.userint1 = userint1;
  style.userint2 = userint2;
  style.userint3 = userint3;
  style.userstring1 = userstring1;
  style.userstring2 = userstring2;
  style.userstring3 = userstring3;
}


int stylesheet_style_get_pointer (const vector<Style>& styles, const ustring& marker)
// Returns a pointer to "styles" which describes "marker".
// Or -1 if not found.
{
  // Search marker and return pointer.
  for (unsigned int i = 0; i < styles.size(); i++)
    if (marker == styles[i].marker)
      return i;
  // Ok, marker not found.
  return -1;
}


void stylesheets_upgrade ()
// Upgrade older stylesheets to the currently used format.
{
  // Upgrade *.sql3 stylesheets *.sql4 ones.
  {
    // The point here is to upgrade the description of a style. It gets an
    // asterisk if the closing marker is obligatory, like in \add_text\add*,
    // and it gets (*) when the closer is optional, like in footnote styles.
    char *ASTERISK [] =
    { 
      "va", "vp", "qs", "qac", "f", "fe", "fm", "fdc", "x", "xdc", "qt", "nd",
      "tl",  "dc", "bk", "sig",  "pn", "wj", "k",  "sls", "ord", "add", "lit",
      "no", "bd", "it", "bdit", "em",  "sc", "fig", "pro",  "w", "wh", "wg",
      "ndx"
    };
    char *ASTERISK_BRACKETS [] =
    { 
      "fr", "fk", "fq", "fqa", "fl", "fp", "fv", "ft", "xo", "xk", "xq",  "xt"
    };
    ReadFiles rf (directories_get_stylesheets (), "", ".sql3");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        for (unsigned int m = 0; m < (sizeof (ASTERISK) / sizeof (*ASTERISK)); m++) {
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf ("select name from styles where marker = '%s';", ASTERISK[m]);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          if (!reader.ustring0.empty()) {
            reader.ustring0[0].insert (0, "* ");
            sql = g_strdup_printf ("update styles set name = '%s' where marker = '%s';", double_apostrophy (reader.ustring0[0]).c_str(), ASTERISK[m]);
            rc = sqlite3_exec(db, sql, NULL, NULL, &error);
            g_free (sql);
          }
        }
        for (unsigned int m = 0; m < (sizeof (ASTERISK_BRACKETS) / sizeof (*ASTERISK_BRACKETS)); m++) {
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf ("select info from styles where marker = '%s';", ASTERISK_BRACKETS[m]);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          if (!reader.ustring0.empty()) {
            reader.ustring0[0].insert (0, "(*) ");
            sql = g_strdup_printf ("update styles set info = '%s' where marker = '%s';", double_apostrophy (reader.ustring0[0]).c_str(), ASTERISK_BRACKETS[m]);
            rc = sqlite3_exec(db, sql, NULL, NULL, &error);
            g_free (sql);
          }
        }
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "4");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // Upgrade *.sql4 stylesheets *.sql5 ones.
  {
    // The upgrade here is that colors are introduced for inline text, and 
    // some other things. Set \wj to red.
    ReadFiles rf (directories_get_stylesheets (), "", ".sql4");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        rc = sqlite3_exec(db, "alter table styles add column color integer;", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set color = 0;", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set color = 16711680 where marker = 'wj';", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set type = 4 where marker = 'ndx';", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set fontpercentage = 60 where superscript = 1;", NULL, NULL, &error);
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "5");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // Upgrade *.sql5 stylesheets *.sql6 ones.
  {
    // The upgrade here is that the chapter should be printed at verse one.
    ReadFiles rf (directories_get_stylesheets (), "", ".sql5");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        rc = sqlite3_exec(db, "update styles set userbool1 = 1 where marker = 'c';", NULL, NULL, &error);
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "6");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // Upgrade *.sql6 stylesheets *.sql7 ones.
  {
    // The upgrade here is that the chapter properties should be set properly.
    ReadFiles rf (directories_get_stylesheets (), "", ".sql6");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        rc = sqlite3_exec(db, "update styles set fontsize = 34 where marker = 'c';", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set spaceafter = '-3.0' where marker = 'c';", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set userint1 = 90 where marker = 'c';", NULL, NULL, &error);
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "7");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // Upgrade *.sql7 stylesheets *.sql8 ones.
  {
    // The upgrade here is that the "print" property has been introduced.
    ReadFiles rf (directories_get_stylesheets (), "", ".sql7");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        rc = sqlite3_exec(db, "alter table styles add column print integer;", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set print = 1;", NULL, NULL, &error);
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "8");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // Upgrade *.sql8 stylesheets *.sql9 ones.
  {
    // The upgrade here is that the endnotes print.
    ReadFiles rf (directories_get_stylesheets (), "", ".sql8");
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring filename = gw_build_filename (directories_get_stylesheets (), rf.files[i]);
      gw_message ("Updating stylesheet " + filename);
      sqlite3 *db;
      int rc;
      char *error = NULL;
      try
      {
        rc = sqlite3_open(filename.c_str (), &db);
        if (rc) throw runtime_error (sqlite3_errmsg(db));
        sqlite3_busy_timeout (db, 1000);
        rc = sqlite3_exec(db, "update styles set userint2 = 1 where marker = 'fe';", NULL, NULL, &error);
        rc = sqlite3_exec(db, "update styles set userstring2 = 'zendnotes' where marker = 'fe';", NULL, NULL, &error);
      }
      catch (exception & ex)
      {
        gw_critical (ex.what ());
      }
      sqlite3_close (db);
      ustring newfilename (filename);
      newfilename.replace (newfilename.length() - 1, 1, "9");
      ustring command = "mv";
      command.append (shell_quote_space (filename));
      command.append (shell_quote_space (newfilename));
      system (command.c_str());
    }
  }
  
  // At the end of everything, check that we have at least one stylesheet.  
  {
    vector<ustring> stylesheets;
    stylesheet_get_ones_available (stylesheets);
    if (stylesheets.size() == 0) {
      stylesheet_create_new (STANDARDSHEET);
    }
  }
}


vector<ustring> stylesheet_get_recently_used (const ustring& stylesheet)
{
  // Read all the markers.
  vector<ustring> recent_markers;
  // Some variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Read the styles.
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select marker, count from recent;");
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    recent_markers.assign (reader.ustring0.begin(), reader.ustring0.end());
    // Sort them on usage count. Most used first.
    vector <unsigned int> usage_count;
    for (unsigned int i = 0; i < reader.ustring1.size(); i++) {
      usage_count.push_back (G_MAXINT - convert_to_int (reader.ustring1[i]));
    }
    quick_sort (usage_count, recent_markers, 0, usage_count.size());
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
  // Take out the ones that are not in the stylesheet.
  vector<ustring> styles = stylesheet_get_markers (stylesheet, NULL);
  set<ustring> styles2 (styles.begin(), styles.end());
  vector<ustring> existing_ones;
  for (unsigned int i = 0; i < recent_markers.size(); i++) {
    if (styles2.find (recent_markers[i]) != styles2.end())
      existing_ones.push_back (recent_markers[i]);
  }
  recent_markers.clear();
  recent_markers.assign (existing_ones.begin(), existing_ones.end());
  // Return result.
  return recent_markers;  
}


void stylesheet_set_recently_used (const ustring& stylesheet, vector<ustring> styles)
{
  // Some variables.
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Variable.
    char * sql;
    // Delete all existing markers.
    sql = g_strdup_printf ("delete from recent;");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Insert all new markers.
    for (unsigned int i = 0; i < styles.size(); i++) {
      sql = g_strdup_printf("insert into recent values ('%s', %d);", styles[i].c_str(), 0);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
    }
    // Free memory.
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
}


void stylesheet_recently_used_increase_count (const ustring& stylesheet, const ustring& style)
{
  // Some variables.
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Variable.
    char * sql;
    // Read the style's usage count.
    SqliteReader reader (0);
    sql = g_strdup_printf ("select count from recent where marker = '%s';", style.c_str());
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    unsigned int count = 0;
    if (!reader.ustring0.empty()) count = convert_to_int (reader.ustring0[0]);
    // Increate the count.
    count++;
    // Delete marker's data from database.
    sql = g_strdup_printf ("delete from recent where marker = '%s';", style.c_str());
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Insert the marker and its count.
    sql = g_strdup_printf("insert into recent values ('%s', %d);", style.c_str(), count);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
}


bool stylesheet_style_has_endmarker (const Style& style)
{
  bool endmarker = false;
  if (style.type == stInlineText)
    endmarker = true;
  if (style.type == stFootEndNote) {
    if (style.subtype == fentFootnote)
      endmarker = true;
    if (style.subtype == fentEndnote)
      endmarker = true;
    if (style.subtype == fentContentWithEndmarker)
      endmarker = true;
  }
  if (style.type == stCrossreference) {
    if (style.subtype == ctCrossreference)
      endmarker = true;
    if (style.subtype == ctContentWithEndmarker)
      endmarker = true;
  }
  if (style.type == stPicture)
    endmarker = true;
  return endmarker;
}


void stylesheet_save_style (const ustring& stylesheet, const Style& style)
{
  // Some variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(stylesheet_filename (stylesheet).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Variable.
    char * sql;
    // Delete possible existing style.
    sql = g_strdup_printf ("delete from styles where marker = '%s';", style.marker.c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Save the style.
    sql = g_strdup_printf("insert into styles values ('%s', '%s', '%s', %f, %i, '%s', '%s', '%s', '%s', %i, '%s', %f, %f, %f, %f, %f, %i, %i, %i, %i, %i, %i, %i, %i, %i, '%s', '%s', '%s', %d, %d);",
      style.marker.c_str(), double_apostrophy (style.name).c_str(), double_apostrophy (style.info).c_str(),
      style.fontsize, style.fontpercentage,
      style.italic.c_str(), style.bold.c_str(), style.underline.c_str(), style.smallcaps.c_str(), style.superscript, style.justification.c_str(),
      style.spacebefore, style.spaceafter, style.leftmargin, style.rightmargin,
      style.firstlineindent, style.spancolumns, 
      style.type, style.subtype, 
      style.userbool1, style.userbool2, style.userbool3,
      style.userint1, style.userint2, style.userint3,
      style.userstring1.c_str(), style.userstring2.c_str(), style.userstring3.c_str(),
      style.color, style.print);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Free memory.
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
}


void stylesheet_load_style (const ustring& stylesheet, Style& style)
{
  // Some variables.  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // If the stylesheet is not given, we read from the template.
    ustring filename = stylesheet_filename (stylesheet);
    if (stylesheet.empty()) filename = stylesheet_template_filename ();
    // Connect to the database.
    rc = sqlite3_open(filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    // Read the style.
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select name, info, fontsize, fontpercentage, "
      "italic, bold, underline, smallcaps, superscript, justification, "
      "spacebefore, spaceafter, leftmargin, rightmargin, firstlineindent, spancolumns, "
      "type, subtype, userbool1, userbool2, userbool3, "
      "userint1, userint2, userint3, userstring1, userstring2, userstring3, "
      "color, print "
      "from styles where marker = '%s';", style.marker.c_str());
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    if (reader.ustring0.size() > 0) {
      style.name = reader.ustring0[0];
      style.info = reader.ustring1[0];
      style.fontsize = convert_to_double (reader.ustring2[0]);
      style.fontpercentage = convert_to_int (reader.ustring3[0]);
      style.italic = reader.ustring4[0];
      style.bold = reader.ustring5[0];
      style.underline = reader.ustring6[0];
      style.smallcaps = reader.ustring7[0];
      style.superscript = convert_to_bool (reader.ustring8[0]);
      style.justification = reader.ustring9[0];
      style.spacebefore = convert_to_double (reader.ustring10[0]);
      style.spaceafter = convert_to_double (reader.ustring11[0]);
      style.leftmargin = convert_to_double (reader.ustring12[0]);
      style.rightmargin = convert_to_double (reader.ustring13[0]);
      style.firstlineindent = convert_to_double (reader.ustring14[0]);
      style.spancolumns = convert_to_bool (reader.ustring15[0]);
      style.type = (StyleType) convert_to_int (reader.ustring16[0]);
      style.subtype = convert_to_int (reader.ustring17[0]);
      style.userbool1 = convert_to_bool (reader.ustring18[0]);
      style.userbool2 = convert_to_bool (reader.ustring19[0]);
      style.userbool3 = convert_to_bool (reader.ustring20[0]);
      style.userint1 = convert_to_int (reader.ustring21[0]);
      style.userint2 = convert_to_int (reader.ustring22[0]);
      style.userint3 = convert_to_int (reader.ustring23[0]);
      style.userstring1 = reader.ustring24[0];
      style.userstring2 = reader.ustring25[0];
      style.userstring3 = reader.ustring26[0];
      style.color = convert_to_int (reader.ustring27[0]);
      style.print = convert_to_int (reader.ustring28[0]);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.
  sqlite3_close (db);
}


void stylesheet_vacuum (const ustring& stylesheet, unsigned int starttime)
{
  vacuum_database (stylesheet_filename (stylesheet), starttime);
}
