#!/usr/bin/env python
#test_lyxclient.py            -*- coding: iso-8859-1 -*-
# Copyright (c) 2005 Gnter Milde
# Released under the terms of the GNU General Public License (ver. 2 or later)
"""Test functions and classes in the LyX.lyxclient module

Revision History:

2005-02-23 G. Milde first version with the unittest framework
"""


from unittest import TestCase, main 
import os, sys, types

from LyX.lyxclient import *
logger.setLevel(logging.DEBUG) 

# an absolute and a relative filename
filenames = [os.path.join(os.path.dirname(__file__), "test.lyx"),
             "test2.lyx"]


class lyxclientTests(TestCase):
    """Test public functions in lyxclient.py"""
    # Common testcases
    # a list of file name arguments
    filelist = ["file1", 'file2']
    # valid options and args: update this list for new args
    valid_options = (['-help'],
                     ['-version'],
                     ['-sysdir', 'arg1'], 
                     ['-userdir', 'arg1'], 
                     ['-dbg', 'arg1'], 
                     ['-x', 'arg1'], 
                     ['--execute', 'arg1'], 
                     ['-e', 'arg1'], 
                     ['--export', 'arg1'],
                     ['-i', 'arg1', 'arg2'], 
                     ['--import', 'arg1', 'arg2'],
                     # xfroms fronted options
                     ['-geometry', 'arg1'], 
                     ['-display', 'arg1'], 
                     ['-bw', 'arg1'], 
                     ['-visual', 'arg1'], 
                     ['-depth', 'arg1'], 
                     ['-debug', 'arg1'], 
                     ['-sync'], 
                     ['-private'], 
                     ['-shared'], 
                     ['-stdcmap'], 
                     ['-name', 'arg1']
                    )
    invalid_options = (['--help'],           # misspelling (2 dashes)
                       ['-version', 'arg1'], # spurious argument
                       ['-sysdir'],          # missing argument
                      )
    #    
    def test_filter_options_valid(self):
        """filter defined options and the correct number of arguments"""
        for options in self.valid_options:
            args = options+self.filelist
            _options, files = filter_options(args, LYX_OPTIONS)
            self.assertEqual(_options, options)
            self.assertEqual(files, self.filelist)
    #
    def test_filter_options_invalid(self):
        """do not filter invalid options or wrong number of arguments"""
        for options in self.invalid_options:
            args = options+self.filelist
            _options, files = filter_options(args, LYX_OPTIONS)
            self.assertNotEqual(_options, options)
            self.assertNotEqual(files, self.filelist)
    #
    # def test_lyx_remote(self):
    #     """lyxserver.lyx_remote should open given files"""
    #     client = lyx_remote(LYXCMD, filenames)
    #     time.sleep(0.5) # give lyx some response time (adjust to your system)
    #     for i in range(2):
    #         self.assert_(os.path.samefile(client("server-get-name"), filenames[0])
    #                      or os.path.samefile(client("server-get-name"), filenames[1]))
    #         client("buffer-close")



class LyXMessageTests(TestCase):
    """Test the lyxclient.LyXMessage class"""
    msg_strings = ["LYXCMD:clientname:function:argument",
                   "INFO:clientname:function:data",
                   "ERROR:clientname:function:error message",
                   "NOTIFY:key-sequence",
                   "LYXSRV:clientname:hello"]
    def test_init_without_args(self):
        """initializing LyXMessage without args should return empty object"""
        msg = LyXMessage()
        self.assertEqual(0, len(msg))

    def test_init_with_keyword_args(self):
        """initializing LyXMessage with keyword args should set the fields"""
        msg = LyXMessage(msg_type='LYXCMD', client='name', 
                         function='fun', data='arg')
        msg_string = 'LYXCMD:name:fun:arg\n'
        self.assertEqual(msg_string, str(msg))
    
    def test_init_with_string(self):
        """initializing LyXMessage with string should parse the string"""
        for msg_string in self.msg_strings:
            msg = LyXMessage(msg_string)
            self.assertEqual(msg_string+"\n", str(msg))
    
    def test_parse(self):
        """LyXMessage.parse(msg_str) should set the fields"""
        msg = LyXMessage()
        for msg_string in self.msg_strings:
            msg.parse(msg_string)
            self.assertEqual(msg_string+"\n", str(msg))
    
    def test_parse_strip_newlines(self):
        """LyXMessage.parse(msg_str) should strip newlines"""
        msg = LyXMessage()
        for msg_string in self.msg_strings:
            msg.parse(msg_string+"\n")
            self.assertEqual(msg_string+"\n", str(msg))
    
    def test_parse_with_wrong_type(self):
        """LyXMessage.parse(msg) with wrong msg_type should raise KeyError"""
        msg = LyXMessage()
        self.assertRaises(KeyError, msg.parse, "STRANGE:F12")
        
    def test_list_fields(self):
        """LyXMessage.list_fields should return ordered list of fields"""
        msg = LyXMessage(msg_type = "LYXCMD", client = "test", 
                         function = "open", data = "foo")
        self.assertEqual(["LYXCMD", "test", "open", "foo"],
                         msg.list_fields())

    def test_list_fields_missing_fields(self):
        """LyXMessage.list_fields should set missing fields to None"""
        msg = LyXMessage(msg_type = "NOTIFY", data = "foo")
        self.assertEqual(["NOTIFY", None, None, "foo"],
                         msg.list_fields())
    
    def test__str__(self):
        """str(LyXMessage) should return a string suitable for the lyxserver"""
        msg = LyXMessage(msg_type = "NOTIFY", data = "foo")
        self.assertEqual("NOTIFY:foo\n", str(msg))
    
    def test__eq__(self):
        """two LyXMessage object are equal, iff all fields are equal"""
        msg1 = LyXMessage("NOTIFY:F12")
        msg2 = LyXMessage("NOTIFY:F11")
        msg3 = LyXMessage(msg_type="NOTIFY", data="F12", client="test")
        self.assertEqual(msg1, msg1)
        self.assertNotEqual(msg1, msg2)
        self.assertNotEqual(msg1, msg3)
    
    def test__eq__wildcards(self):
        """Comparing LyXMessages uses special field value True as wildcard"""
        msg1 = LyXMessage("NOTIFY:F12")
        msg2 = LyXMessage("NOTIFY:F11")
        msg3 = LyXMessage(msg_type="NOTIFY", data=True)
        self.assertEqual(msg1, msg3)
        self.assertEqual(msg2, msg3)
        self.assertNotEqual(msg1, msg2)
    

class LyXErrorTests(TestCase):
    def test__init__(self):
        """lyxclient.LyXerror should work as an expeption"""
        try:
            raise LyXError, "test"
        except LyXError, value:
            pass
        self.assertEqual(str(value), "test")

class LyXClientTests(TestCase):
    """Test the lyxclient.LyXClient class"""
    
    def setUp(self):
        self.client = LyXClient()  # lyxclient object with defaults
        # self.client.write_lfun("file-open", filenames[0])
    def tearDown(self):
        # self.client.write_lfun("buffer-close")
        del(self.client)
    
    def test_readmessage(self):
        """LyXClient.readmessage should read one line and return as LyXMessage (or emty message)"""
        self.client.write_lfun("message", "hi", "lyx")
        msg = self.client.readmessage()
        self.assertEqual(msg.function, "message")
        
    def test_readmessage_filter(self):
        """LyXClient.readmessage should filter the messages according to arguments"""
        # target fun
        self.client.write_lfun("message", "hi", "lyx")
        # "noise"
        self.client.write_lfun("server-get-name")
        msg = self.client.readmessage(function="message")
        self.assertEqual(msg.function, "message")
        
    def test_readmessage_writeback(self):
        """LyXClient.readmessage should write back `writeback` filtered messages"""
        self.client.write_lfun("apropos", "file-open", "lyx")
        self.client.write_lfun("message", "hi", "lyx")
        msg = self.client.readmessage(function="message", writeback=3)
        self.assertEqual(msg.function, "message")
        # now get the earlier response
        msg2 = self.client.readmessage(function="apropos")
        self.assertEqual(msg2.function, "apropos")

    def test_readmessages(self):
        """LyXClient.readmessages should return array of messages"""
        self.client.write_lfun("message", "hi", "lyx")
        msg_list = self.client.readmessages()
        # self.assertEqual(len(msg_list), 2)
        self.assertEqual(type(msg_list), type([]))
    #
    def test_write_lfun(self):
        """LyXClient.write_lfun should convert function name and data into valid cmd_string"""
        self.client.read(timeout=0) # empty the outpipe
        self.client.write_lfun("message", "hi", "lyx")
        self.assertEqual(self.client.readmessage().msg_type, "INFO")
    #    
    def test__call__(self):
        """LyXClient(fun, *args) should call the lfun `fun` and return result"""
        self.client("file-open", filenames[0])
        # time.sleep(0.5) # give lyx some response time (adjust to your system)
        # self.assertEqual(self.client("server-get-name"), filenames[0])
        self.assert_(os.path.samefile(self.client("server-get-name"), filenames[0]))
        self.client("buffer-close")
    #
    def test__iter__(self):
        """LyXClient.__iter__ should return iterator (yielding messages)"""
        self.assertEqual(types.GeneratorType, type(self.client.__iter__()))
        self.client.read()  # empty the pipe
        for i in range(3):
            self.client.write_lfun("message", "hi", "lyx")
        for msg in self.client:
            self.assertEqual(msg.function, "message")
    
    # def test_listen(self):
    #     """LyXClient.listen should listen for notify-events and run actions bound to them"""
    #
    # This needs interactive input and special keybindings :-(
        

if __name__ == '__main__':    #run tests if called from command-line
    main()
