/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/crgc.c                  */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Sun Sep 13 11:58:32 1998                          */
/*    Last change :  Wed Nov  7 16:04:41 2001 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Rgc runtime (mostly port handling).                              */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h> 
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <sys/time.h>
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   ifdef HAVE_TERMIO
#      include <termio.h>
#   endif
#endif
#if( !defined( sony_news ) && \
     !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <unistd.h>
#endif
#include <sys/file.h>
#include <bigloo.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif

/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

#define RGC_DEBUG
#undef RGC_DEBUG

/*---------------------------------------------------------------------*/
/*    C importations                                                   */
/*---------------------------------------------------------------------*/
extern obj_t bigloo_case_sensitive;
extern obj_t string_to_bstring_len( char *, int );
extern obj_t string_to_keyword( char * );
extern int toupper( int );

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    bgl_fgetc ...                                                    */
/*    -------------------------------------------------------------    */
/*    We explicitly defines this function because on some system       */
/*    fgetc is a macro.                                                */
/*---------------------------------------------------------------------*/
int
bgl_fgetc( FILE *stream ) {
   return getc( stream );
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    bgl_feof ...                                                     */
/*    -------------------------------------------------------------    */
/*    We explicitly defines this function because on some system       */
/*    feof is a macro.                                                 */
/*---------------------------------------------------------------------*/
int
bgl_feof( FILE *stream ) {
   return feof( stream );
}


/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    rgc_enlarge_buffer ...                                           */
/*    -------------------------------------------------------------    */
/*    This function double the size of a port's buffer. An error is    */
/*    raised if there is not enough room for the allocation.           */
/*---------------------------------------------------------------------*/
static void
rgc_enlarge_buffer( obj_t port )
{
   long bufsize = INPUT_PORT( port ).bufsiz;
   char *buffer = (char *)GC_MALLOC_ATOMIC( 2 * bufsize );
   
#if defined( RGC_DEBUG )
   printf( "rgc_enlarge_buffer: bufsize: %d\n", bufsize );
#endif

   if( !buffer )
      C_FAILURE( "rgc_enlarge_buffer_port", "Can't enlarge buffer", port );
      
   memcpy( buffer, RGC_BUFFER( port ), bufsize );

   INPUT_PORT( port ).bufsiz = 2 * bufsize;
   RGC_BUFFER( port )        = (unsigned char *)buffer;
}
  
/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    fread_con ...                                                    */
/*    -------------------------------------------------------------    */
/*    This function implements a non blocking fread on the console.    */
/*---------------------------------------------------------------------*/
static long
fread_con( obj_t port, char *ptr, long size, long nmemb )
{
   long  num = size * nmemb;
   char *buf = ptr;
   int   c;
   int (*a_getc)() = INPUT_PORT( port ).sysread;
   FILE *stream = INPUT_PORT( port ).file;
   
   while( ((c = a_getc( stream )) != EOF) )
   {
      *buf++ = c;

      if( c == '\n' ) break;
      if( --num <= 0 ) break;
   }

   return (long)(buf - ptr);
}
     
/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_size_fill_con_buffer ...                                     */
/*---------------------------------------------------------------------*/
static bool_t
rgc_size_fill_con_buffer( obj_t port, int abufsize, int size )
{
   unsigned char *buffer = RGC_BUFFER( port );

#if defined( RGC_DEBUG )
   printf( "rgc_size_fill_console_buffer: abufsize: %d  size: %d\n", abufsize, size );
   printf( "buffer[%s]\n", buffer );
   assert( (abufsize + size) == INPUT_PORT( port ).bufsiz );
   assert( size > 0 );
#endif

   /* we start reading at ABUFSIZE - 1 because we have */
   /* to remove the '\0' sentinel that ends the buffer */
   abufsize += fread_con( port,
			  (char *)&buffer[ abufsize - 1 ],
 			  1,
			  size );
   buffer[ abufsize - 1 ] = '\0';
   
   INPUT_PORT( port ).abufsiz = abufsize;
   assert( INPUT_PORT( port ).abufsiz <= INPUT_PORT( port ).bufsiz );
   
   if( INPUT_PORT( port ).syseof( INPUT_PORT( port ).file ) )
#if defined( RGC_DEBUG )
   if( INPUT_PORT( port ).kindof == KINDOF_SOCKET )
      printf( "rgc_size_fill_console_buffer: EOF\n"),
#endif
      INPUT_PORT( port ).eof = 1;

   if( ferror( INPUT_PORT( port ).file ) )
      C_FAILURE( "rgc_size_fill_con_buffer",
		 "Error while reading on console",
		 BINT( ferror( INPUT_PORT( port ).file ) ) );
   
#if defined( RGC_DEBUG )
   printf( "FIN de fill: size: %d  asize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).bufsiz, INPUT_PORT( port ).abufsiz,
	   INPUT_PORT( port ).forward, 
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "buffer: [%s]\n", RGC_BUFFER( port ) );
#endif

   return (INPUT_PORT( port ).abufsiz > 0);
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_fill_console_buffer ...                                      */
/*---------------------------------------------------------------------*/
static bool_t
rgc_fill_console_buffer( obj_t port )
{
   long bufsize    = INPUT_PORT( port ).bufsiz;
   long abufsize   = INPUT_PORT( port ).abufsiz;
   long matchstart = INPUT_PORT( port ).matchstart;

#if defined( RGC_DEBUG )
   printf( "rgc_fill_console_buffer: bufsize: %d  abufsize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   bufsize, abufsize, INPUT_PORT( port ).forward,
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
#endif
   
   /* if the buffer is not full, we fill it */
   if( abufsize < bufsize )
      return rgc_size_fill_con_buffer( port, abufsize, bufsize - abufsize );
   else
   {
      if( matchstart > 0 )
      {
	 unsigned char *buffer = RGC_BUFFER( port );

	 assert( abufsize > 0 );
	 
	 /* we shift the buffer left and we fill the buffer */
	 strcpy( buffer, buffer + matchstart );
	 
	 abufsize                     -= matchstart;
	 INPUT_PORT( port ).matchstart = 0;
	 INPUT_PORT( port ).matchstop -= matchstart;
	 INPUT_PORT( port ).forward   -= matchstart;
	 INPUT_PORT( port ).lastchar   = RGC_BUFFER( port )[ matchstart - 1 ];
	 
	 return rgc_size_fill_con_buffer( port, abufsize, bufsize-abufsize );
      }
      else
      {
	 /* we current token is too large for the buffer */
	 /* we have to enlarge it.                       */
	 rgc_enlarge_buffer( port );
	 
	 return rgc_fill_console_buffer( port );
      }
   }
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_size_fill_file_buffer ...                                    */
/*---------------------------------------------------------------------*/
static bool_t
rgc_size_fill_file_buffer( obj_t port, int abufsize, int size )
{
   unsigned char *buffer = RGC_BUFFER( port );
   
#if defined( RGC_DEBUG )
   assert( abufsize >= 1 );
   assert( (abufsize + size) == INPUT_PORT( port ).bufsiz );

   printf( "rgc_size_fill_file_buffer: abufsize: %d  size: %d\n", abufsize, size );
   assert( size > 0 );
#endif

   /* we start reading at ABUFSIZE - 1 because we have */
   /* to remove the '\0' sentinel that ends the buffer */
   abufsize += INPUT_PORT( port ).sysread( &buffer[ abufsize - 1 ],
					   1,
					   size,
					   INPUT_PORT(port).file );
   
   INPUT_PORT( port ).abufsiz = abufsize;
   assert( INPUT_PORT( port ).abufsiz <= INPUT_PORT( port ).bufsiz );
   
   if( INPUT_PORT( port ).syseof( INPUT_PORT( port ).file ) )
      INPUT_PORT( port ).eof = 1;

   if( ferror( INPUT_PORT( port ).file ) )
      C_FAILURE( "rgc_size_fill_file_buffer",
		 "Error while reading on file",
		 BINT( ferror( INPUT_PORT( port ).file ) ) );
   

#if defined( RGC_DEBUG )
   printf( "FIN de fill: size: %d  asize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).bufsiz, INPUT_PORT( port ).abufsiz,
	   INPUT_PORT( port ).forward, 
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "buffer: [%s]\n", buffer );
#endif
   
   if( abufsize > 0 )
   { 
      buffer[ abufsize - 1 ] = '\0';

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    movemem ...                                                      */
/*---------------------------------------------------------------------*/
void movemem( char *dest, char *src, long n )
{
   while( n-- )
      *dest++ = *src++;
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_fill_file_buffer ...                                         */
/*---------------------------------------------------------------------*/
static bool_t
rgc_fill_file_buffer( obj_t port )
{
   long bufsize    = INPUT_PORT( port ).bufsiz;
   long abufsize   = INPUT_PORT( port ).abufsiz;
   long matchstart = INPUT_PORT( port ).matchstart;
   unsigned char *buffer = RGC_BUFFER( port );

#if defined( RGC_DEBUG )
   assert( (abufsize >= 0) && (abufsize <= bufsize) );
   
   printf( "rgc_fill_file_buffer: bufsize: %d  abufsize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   bufsize, abufsize, INPUT_PORT( port ).forward,
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
#endif

   if( matchstart > 0 )
   {
      long movesize = abufsize - matchstart;
	 
      /* we shift the buffer left and we fill the buffer */
      movemem( (char *)&buffer[ 0 ], (char *)&buffer[ matchstart ], movesize );
	 
      abufsize                     -= matchstart;
      INPUT_PORT( port ).matchstart = 0;
      INPUT_PORT( port ).matchstop -= matchstart;
      INPUT_PORT( port ).forward   -= matchstart;
      INPUT_PORT( port ).lastchar   = buffer[ matchstart - 1 ];
	 
      return rgc_size_fill_file_buffer( port, abufsize, bufsize - abufsize );
   }
   else
   {
      if( abufsize < bufsize )
	 return rgc_size_fill_file_buffer( port, abufsize, bufsize-abufsize );
      else
      {
	 /* we current token is too large for the buffer */
	 /* we have to enlarge it.                       */
	 rgc_enlarge_buffer( port );
	 
	 return rgc_fill_file_buffer( port );
      }
   }
}
      
/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_fill_buffer ...                                              */
/*---------------------------------------------------------------------*/
bool_t
rgc_fill_buffer( obj_t port )
{
#if defined( RGC_DEBUG ) 
   puts( "~~~~~ rgc_fill_buffer ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" );
   printf( "eof: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).eof,
	   INPUT_PORT( port ).matchstart,
	   INPUT_PORT( port ).matchstop );
#endif
   /* In every case, forward has to be unwinded */
   /* because forward has reached the sentinel  */
   INPUT_PORT( port ).forward--;
   
   /* an input port that has seen its eof       */
   /* cannot be filled anymore                  */
   if( INPUT_PORT( port ).eof )
      return 0;
   else
   {
#if defined( RGC_DEBUG )
      switch( (long)INPUT_PORT( port ).kindof ) {
	 case (long)KINDOF_FILE:
	    printf( "filling a FILE\n" );
	    break;
	 case (long)KINDOF_PIPE:
	    printf( "filling a PIPE\n" );
	    break;
	 case (long)KINDOF_PROCPIPE:
	    printf( "filling a PIPE\n" );
	    break;
	 case (long)KINDOF_CONSOLE:
	    printf( "filling a CONSOLE\n" );
	    break;
	 case (long)KINDOF_SOCKET:
	    printf( "filling a SOCKET\n" );
	    break;
	 case (long)KINDOF_CLOSED:
	    printf( "filling a CLOSED\n" );
	    break;
	 case (long)KINDOF_STRING:
	    printf( "filling a STRING\n" );
	    break;
      }
#endif

      switch( (long)INPUT_PORT( port ).kindof ) {
	 case (long)KINDOF_FILE:
	 case (long)KINDOF_PROCPIPE:
	 case (long)KINDOF_PIPE:
	    return rgc_fill_file_buffer( port );
	 case (long)KINDOF_CONSOLE:
	 case (long)KINDOF_SOCKET:
	    return rgc_fill_console_buffer( port );
	 default:
	    return 0;
      }
   }
}
   
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_substring ...                                         */
/*    -------------------------------------------------------------    */
/*    This function makes no bound checks because these tests have     */
/*    already been performed in the grammar.                           */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_substring( obj_t ip, long offset, long end )
{
   long start = INPUT_PORT( ip ).matchstart;
   long len   = end - offset;

#if defined( RGC_DEBUG )
   printf( "buffer-substring: start: %d  stop: %d  forward: %d  abufsiz: %d\n",
	   start, INPUT_PORT( ip ).matchstop,
	   INPUT_PORT( ip ).forward, INPUT_PORT( ip ).abufsiz );
#endif

   return string_to_bstring_len( (char *)&RGC_BUFFER( ip )[ start + offset ],
				 len );
}

/*---------------------------------------------------------------------*/
/*    CHEAT_BUFFER                                                     */
/*---------------------------------------------------------------------*/
#define CHEAT_BUFFER() \
   long stop  = INPUT_PORT( ip ).matchstop; \
   char bck; \
   bck = RGC_BUFFER( ip )[ stop ]; \
   RGC_BUFFER( ip )[ stop ] = '\0';

/*---------------------------------------------------------------------*/
/*    RESTORE_BUFFER                                                   */
/*---------------------------------------------------------------------*/
#define RESTORE_BUFFER() \
   RGC_BUFFER( ip )[ stop ] = bck;

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    rgc_buffer_fixnum ...                                            */
/*---------------------------------------------------------------------*/
long
rgc_buffer_fixnum( obj_t ip )
{
   long res;
   
   CHEAT_BUFFER();
   
   res = atol( &RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart ] );
   
   RESTORE_BUFFER();
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    double                                                           */
/*    rgc_buffer_flonum ...                                            */
/*---------------------------------------------------------------------*/
double
rgc_buffer_flonum( obj_t ip )
{
   double res;
   
   CHEAT_BUFFER();
  
   res = strtod( &RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart ], 0 );
   
   RESTORE_BUFFER();
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_symbol ...                                            */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_symbol( obj_t ip )
{
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_upcase_symbol ...                                     */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_upcase_symbol( obj_t ip )
{
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = toupper( *walk );

   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_downcase_symbol ...                                   */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_downcase_symbol( obj_t ip )
{
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = tolower( *walk );

   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_keyword ...                                           */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_keyword( obj_t ip )
{
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   key = string_to_keyword( (char *)aux );

   RESTORE_BUFFER();

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_downcase_keyword ...                                  */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_downcase_keyword( obj_t ip )
{
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = tolower( *walk );

   key = string_to_keyword( (char *)aux );

   RESTORE_BUFFER();

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_upcase_keyword ...                                    */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_upcase_keyword( obj_t ip )
{
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = toupper( *walk );

   key = string_to_keyword( (char *)aux );

   RESTORE_BUFFER();

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    rgc_buffer_unget_char ...                                        */
/*---------------------------------------------------------------------*/
int
rgc_buffer_unget_char( obj_t ip, int c )
{
   if( INPUT_PORT( ip ).matchstop > 0 )
      INPUT_PORT( ip ).matchstop--;
   else
   {
      RGC_BUFFER( ip )[ 0 ] = c;
      if( INPUT_PORT( ip ).abufsiz == 0 )
      {
	 INPUT_PORT( ip ).abufsiz = 1;
	 RGC_BUFFER( ip )[ 1 ] = '\0';
      }
   }

   return c;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the matchstart position located at the beginning of a line?   */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_bol_p( obj_t ip )
{
#if( defined( RGC_DEBUG ) )
   printf( "RGC_BUFFER_BOL_P: mstart: %d  [mstart]: %d  lastchar: %d  --> %d\n",
	   INPUT_PORT( ip ).matchstart, 
	   RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ],
	   INPUT_PORT( ip ).lastchar,
	   INPUT_PORT( ip ).lastchar == '\n' );
#endif
   
   if( INPUT_PORT( ip ).matchstart > 0 )
      return RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ] == '\n';
   else
      return INPUT_PORT( ip ).lastchar == '\n';
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Does the buffer contain, at its first non match position, a `\n' */
/*    character?                                                       */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_eol_p( obj_t ip )
{
   int c = RGC_BUFFER_GET_CHAR( ip );
   
#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOL_P: forward: %d %d", f, c );
#endif
   
   if( !c )
   {
      if( !RGC_BUFFER_EMPTY( ip ) )
      {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }

      if( INPUT_PORT( ip ).kindof == KINDOF_CONSOLE )
      {
#if( defined( RGC_DEBUG ) )   
	 puts( "  kindof == CONSOLE --> 1" );
#endif
	 return 1;
      }
      if( rgc_fill_buffer( ip ) )
	 return rgc_buffer_eol_p( ip );
      else
      {
#if( defined( RGC_DEBUG ) )   
	 puts( "   not rgc_fill_buffer --> 0" );
#endif
	 return 0;
      }
   }
   else
   {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      printf( "   --> %d\n", c == '\n' );
#endif
      return c == '\n';
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the match position at the beginning of the file?              */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_bof_p( obj_t ip )
{
   return INPUT_PORT( ip ).filepos == 1;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the input port at its end-of-file position?                   */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_eof_p( obj_t ip )
{
   int c = RGC_BUFFER_GET_CHAR( ip );

#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOF_P: forward: %d %d", f, c );
#endif
   
   if( !c )
   {
      if( !RGC_BUFFER_EMPTY( ip ) )
      {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }
      else
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   --> 1" );
#endif
	 return 1;
   }
   else
   {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      puts( "   not empty --> 0" );
#endif
      return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    file_charready ...                                               */
/*---------------------------------------------------------------------*/
static int
file_charready( FILE *f ) {
#if( BGL_HAVE_SELECT )
   fd_set readfds;
   struct timeval timeout;
   int fno = fileno( f ) + 1;

   FD_ZERO( &readfds );
   FD_SET( fno, &readfds );
   timeout.tv_sec = 0; timeout.tv_usec = 0;

   return select( fno, &readfds, NULL, NULL, &timeout );
#else
   return 0;
#endif
}

/*---------------------------------------------------------------------*/
/*    boot_t                                                           */
/*    bgl_charready ...                                                */
/*---------------------------------------------------------------------*/
bool_t
bgl_rgc_charready( obj_t port ) {
   switch( (long)INPUT_PORT( port ).kindof ) {
      case (long)KINDOF_CLOSED:
	 return 0;
	 
      case (long)KINDOF_STRING:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).abufsiz);
	 
      case (long)KINDOF_FILE:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).abufsiz)
	    || !bgl_feof( INPUT_PORT( port ).file );
	 
      case (long)KINDOF_PROCPIPE:
      case (long)KINDOF_PIPE:
      case (long)KINDOF_CONSOLE:
      case (long)KINDOF_SOCKET:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).abufsiz)
	    || file_charready( INPUT_PORT( port ).file );
	 
      default:
	 return 0;
   }
}

   
