/*=====================================================================*/
/*    .../prgm/project/bigloo/api/fthread/src/Posix/bglfmutex.c        */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Nov  3 07:58:16 2004                          */
/*    Last change :  Tue May  3 16:18:04 2005 (serrano)                */
/*    Copyright   :  2004-05 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    The Posix mutex implementation                                   */
/*=====================================================================*/
#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <string.h>

#define GC_PRIVATE_H
#include <gc.h>
#include <bglfthread.h>

/*---------------------------------------------------------------------*/
/*    Import                                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL void bgl_mutex_init_register( obj_t (*)( obj_t ) );
BGL_RUNTIME_DECL void bgl_mutex_lock_register( bool_t (*)( obj_t ) );
BGL_RUNTIME_DECL void bgl_mutex_unlock_register( bool_t (*)( obj_t ) );

/*---------------------------------------------------------------------*/
/*    Mutex symbols                                                    */
/*---------------------------------------------------------------------*/
static obj_t sym_not_owned = 0L;
static obj_t sym_abandoned = 0L;
static obj_t sym_not_abandoned = 0L;

static obj_t mutexes = BNIL;

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    symbols_init ...                                                 */
/*---------------------------------------------------------------------*/
static void
symbols_init() {
   if( !sym_not_owned ) {
      sym_not_owned = string_to_symbol( "not-owned" );
      sym_abandoned = string_to_symbol( "abandoned" );
      sym_not_abandoned = string_to_symbol( "not-abandoned" );
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglfth_mutex_state ...                                           */
/*---------------------------------------------------------------------*/
obj_t
bglfth_mutex_state( obj_t m ) {
   bglfmutex_t mut = BGLFTH_MUTEX_BGLPMUTEX( m );

   if( mut->locked ) {
      if( mut->thread ) {
	 return mut->thread->bglobj;
      }
      
      symbols_init();
      
      return sym_not_owned;
   } else {
      symbols_init();
      
      if( mut->thread ) 
	 return sym_abandoned;
      else 
	 return sym_not_abandoned;
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglfth_mutex_init ...                                            */
/*---------------------------------------------------------------------*/
obj_t
bglfth_mutex_init( obj_t m ) {
   bglfmutex_t mut = (bglfmutex_t)GC_MALLOC( sizeof( struct bglfmutex ) );

   mut->thread = 0L;
   mut->locked = 0;
   
   m->mutex_t.mutex = mut;
   pthread_mutex_init( &(mut->pmutex), 0L );
   
   return m;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_mutexes_unlock ...                                        */
/*---------------------------------------------------------------------*/
void
bglfth_mutexes_unlock( bglfthread_t thread ) {
   obj_t w = mutexes;

   while( PAIRP( w ) ) {
      obj_t m = CAR( w );
      bglfmutex_t mut = m->mutex_t.mutex;

      if( mut->thread == thread ) {
	 bglfth_mutex_unlock( m );
	 mut->thread = thread;
      }

      w = CDR( w );
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglfth_mutex_lock ...                                            */
/*---------------------------------------------------------------------*/
bool_t
bglfth_mutex_lock( obj_t m ) {
   bglfmutex_t mut = BGLFTH_MUTEX_BGLPMUTEX( m ); 
   bool_t res = !pthread_mutex_lock( &(mut->pmutex) );

   mut->locked = res;
   
   if( res ) {
      bglfthread_t cth = bglfth_current_thread();

      if( cth ) mut->thread = cth;
   }
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglfth_mutex_unlock ...                                          */
/*---------------------------------------------------------------------*/
bool_t
bglfth_mutex_unlock( obj_t m ) {
   bglfmutex_t mut = BGLFTH_MUTEX_BGLPMUTEX( m );
   bglfthread_t cth = bglfth_current_thread();
   bool_t res; 

   res = pthread_mutex_unlock( &(mut->pmutex) );

   if( !res ) {
      mut->thread = 0L;
      mut->locked = 0;
   }

   return 1;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_setup_mutex ...                                           */
/*---------------------------------------------------------------------*/
void
bglfth_setup_mutex() {
   bgl_mutex_init_register( &bglfth_mutex_init );
   bgl_mutex_lock_register( &bglfth_mutex_lock );
   bgl_mutex_unlock_register( &bglfth_mutex_unlock );
}
