/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2006  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-glib.h>

#include <gconf/gconf-client.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

static DBusGConnection *conn;

typedef enum {
	ICON_POLICY_NEVER,
	ICON_POLICY_ALWAYS,
	ICON_POLICY_PRESENT,
} EnumIconPolicy;

static int icon_policy = ICON_POLICY_PRESENT;

#define PREF_DIR 		"/apps/bluetooth-manager"
#define PREF_ICON_POLICY 	PREF_DIR "/icon_policy"

static GConfEnumStringPair icon_policy_enum_map [] = {
	{ ICON_POLICY_NEVER,	"never"		},
	{ ICON_POLICY_ALWAYS,	"always"	},
	{ ICON_POLICY_PRESENT,	"present"	},
	{ ICON_POLICY_PRESENT,	NULL		},
};

static GConfClient* gconf;

static GtkWidget *notebook;
static GtkWidget *button_never;
static GtkWidget *button_always;
static GtkWidget *button_present;

static void update_icon_policy(GtkWidget *button)
{
	GtkWidget *widget;
	const char *str;

	if (button == NULL) {
		switch (icon_policy) {
		case ICON_POLICY_NEVER:
			widget = button_never;
			break;
		case ICON_POLICY_ALWAYS:
			widget = button_always;
			break;
		case ICON_POLICY_PRESENT:
		default:
			widget = button_present;
			break;
		}

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);

		return;
	}

	if (button == button_always)
		icon_policy = ICON_POLICY_ALWAYS;
	else if (button == button_never)
		icon_policy = ICON_POLICY_NEVER;
	else if (button == button_present)
		icon_policy = ICON_POLICY_PRESENT;

	str = gconf_enum_to_string(icon_policy_enum_map, icon_policy);

	gconf_client_set_string(gconf, PREF_ICON_POLICY, str, NULL);
}

static void delete_callback(GtkWidget *window, GdkEvent *event,
						gpointer user_data)
{
	gtk_widget_destroy(GTK_WIDGET(window));

	gtk_main_quit();
}

static void close_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *window = user_data;

	gtk_widget_destroy(GTK_WIDGET(window));

	gtk_main_quit();
}

static void policy_callback(GtkWidget *button, gpointer user_data)
{
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) == TRUE)
		update_icon_policy(button);
}

static void create_general(void)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GSList *group = NULL;

	vbox = gtk_vbox_new(FALSE, 6);

	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, NULL);

	gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook), vbox, _("General"));

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), _("<b>Notification area</b>"));

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	button_never = gtk_radio_button_new_with_label(group,
						_("Never display icon"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button_never));

	gtk_box_pack_start(GTK_BOX(vbox), button_never, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button_never), "toggled",
					G_CALLBACK(policy_callback), NULL);

	button_present = gtk_radio_button_new_with_label(group,
				_("Only display when adapter present"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button_present));

	gtk_box_pack_start(GTK_BOX(vbox), button_present, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button_present), "toggled",
					G_CALLBACK(policy_callback), NULL);

	button_always = gtk_radio_button_new_with_label(group,
						_("Always display icon"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button_always));

	gtk_box_pack_start(GTK_BOX(vbox), button_always, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button_always), "toggled",
					G_CALLBACK(policy_callback), NULL);

	update_icon_policy(NULL);
}

static void create_window(void)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *buttonbox;
	GtkWidget *button;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_title(GTK_WINDOW(window), _("Bluetooth preferences"));

	gtk_window_set_icon_name(GTK_WINDOW(window), "stock_bluetooth");

	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

	gtk_window_set_default_size(GTK_WINDOW(window), 380, 440);

	g_signal_connect(G_OBJECT(window), "delete-event",
					G_CALLBACK(delete_callback), NULL);

	vbox = gtk_vbox_new(FALSE, 12);

	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	buttonbox = gtk_hbutton_box_new();

	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_END);

	gtk_box_pack_start(GTK_BOX(vbox), buttonbox, FALSE, FALSE, 0);

	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);

	g_signal_connect(G_OBJECT(button), "clicked",
					G_CALLBACK(close_callback), window);

	gtk_container_add(GTK_CONTAINER(buttonbox), button);

#if 0
	button = gtk_button_new_from_stock(GTK_STOCK_HELP);

	gtk_container_add(GTK_CONTAINER(buttonbox), button);

	gtk_button_box_set_child_secondary(GTK_BUTTON_BOX(buttonbox),
								button, TRUE);
#endif

	create_general();

	gtk_widget_show_all(window);
}

static GList *adapter_list = NULL;

struct adapter_data {
	char *path;
	int attached;
	gint page;
	GtkWidget *button_connect;
	GtkWidget *button_visible;
	GtkWidget *timeout_label;
	GtkWidget *timeout_scale;
	GtkWidget *entry;
	GtkWidget *combo;
	int name_changed;
};

static void mode_callback(GtkWidget *button, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const char *mode;
	DBusGProxy *object;
	gboolean sensitive;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) == FALSE)
		return;

	if (button == adapter->button_connect) {
		sensitive = FALSE;
		mode = "connectable";
	} else if (button == adapter->button_visible) {
		sensitive = TRUE;
		mode = "discoverable";
	} else
		return;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetMode", NULL,
			G_TYPE_STRING, mode, G_TYPE_INVALID, G_TYPE_INVALID);

	gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_label), sensitive);
	gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_scale), sensitive);
}

static void scale_callback(GtkWidget *scale, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const char *mode = "discoverable";
	gdouble value;
	guint32 timeout;
	DBusGProxy *object;

	value = gtk_range_get_value(GTK_RANGE(scale));

	if (value == 31)
		timeout = 0;
	else
		timeout = value * 60;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetDiscoverableTimeout", NULL,
			G_TYPE_UINT, timeout, G_TYPE_INVALID, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "SetMode", NULL,
			G_TYPE_STRING, mode, G_TYPE_INVALID, G_TYPE_INVALID);
}

static gchar *format_callback(GtkWidget *scale, gdouble value, gpointer user_data)
{
	if (value > 30)
		return g_strdup(_("never"));
	else if (value == 1)
		return g_strdup(_("1 minute"));
	else
		return g_strdup_printf(_("%g minutes"), value);
}

static void name_callback(GtkWidget *editable, gpointer user_data)
{
	struct adapter_data *adapter = user_data;

	adapter->name_changed = 1;
}

static gboolean focus_callback(GtkWidget *editable,
				GdkEventFocus *event, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const gchar *text;
	DBusGProxy *object;

	if (!adapter->name_changed)
		return FALSE;

	text = gtk_entry_get_text(GTK_ENTRY(editable));

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetName", NULL,
			G_TYPE_STRING, text, G_TYPE_INVALID, G_TYPE_INVALID);

	adapter->name_changed = 0;

	return FALSE;
}

static void class_callback(GtkWidget *combobox, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const char *minor;
	gint index;
	DBusGProxy *object;

	index = gtk_combo_box_get_active(GTK_COMBO_BOX(combobox));

	switch (index) {
	case 0:
		minor = "uncategorized";
		break;
	case 1:
		minor = "desktop";
		break;
	case 2:
		minor = "laptop";
		break;
	default:
		return;
	}

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetMinorClass", NULL,
			G_TYPE_STRING, minor, G_TYPE_INVALID, G_TYPE_INVALID);
}

static void create_adapter(struct adapter_data *adapter)
{
	DBusGProxy *object;
	const char *name = NULL, *mode = NULL, *major = NULL, *minor = NULL;
	const guint32 timeout;

	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *scale;
	GtkWidget *entry;
	GtkWidget *combobox;
	GSList *group = NULL;
	gdouble value;
	gint index;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "GetMode", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &mode, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetDiscoverableTimeout", NULL, G_TYPE_INVALID,
				G_TYPE_UINT, &timeout, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetMajorClass", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &major, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetMinorClass", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &minor, G_TYPE_INVALID);

	vbox = gtk_vbox_new(FALSE, 6);

	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);

	adapter->page = gtk_notebook_prepend_page(GTK_NOTEBOOK(notebook), vbox, NULL);

	gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook), vbox,
				name && strlen(name) > 0 ? name : _("Adapter"));

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), _("<b>Mode of operation</b>"));

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group,
					_("Other devices can connect"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	if (!strcmp(mode, "off"))
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	if (!strcmp(mode, "connectable"))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	adapter->button_connect = button;

	g_signal_connect(G_OBJECT(button), "toggled",
					G_CALLBACK(mode_callback), adapter);

	button = gtk_radio_button_new_with_label(group,
				_("Visible and connectable for other devices"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	if (!strcmp(mode, "off"))
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	if (!strcmp(mode, "discoverable"))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	adapter->button_visible = button;

	g_signal_connect(G_OBJECT(button), "toggled",
					G_CALLBACK(mode_callback), adapter);

	label = gtk_label_new(_("Make adapter invisible after:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	adapter->timeout_label = label;

	scale = gtk_hscale_new_with_range(1, 31, 1);

	gtk_scale_set_digits(GTK_SCALE(scale), 0);

	gtk_scale_set_value_pos(GTK_SCALE(scale), GTK_POS_BOTTOM);

	if (timeout == 0)
		value = 31;
	else if (timeout < 60)
		value = 1;
	else if (timeout > 60 * 30)
		value = 30;
	else
		value = timeout / 60;

	gtk_range_set_value(GTK_RANGE(scale), value);

	gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_DISCONTINUOUS);

	gtk_box_pack_start(GTK_BOX(vbox), scale, FALSE, FALSE, 0);

	adapter->timeout_scale = scale;

	g_signal_connect(G_OBJECT(scale), "value-changed",
					G_CALLBACK(scale_callback), adapter);

	g_signal_connect(G_OBJECT(scale), "format-value",
					G_CALLBACK(format_callback), NULL);

	if (strcmp(mode, "discoverable")) {
		gtk_widget_set_sensitive(GTK_WIDGET(label), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(scale), FALSE);
	}

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), _("\n<b>Adapter name</b>"));

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new();

	if (name != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), name);

	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

	adapter->entry = entry;

	g_signal_connect(G_OBJECT(entry), "changed",
					G_CALLBACK(name_callback), adapter);

	g_signal_connect(G_OBJECT(entry), "focus-out-event",
					G_CALLBACK(focus_callback), adapter);

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), _("\n<b>Class of device</b>"));

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	combobox = gtk_combo_box_new_text();

	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), _("Unspecified"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), _("Desktop workstation"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), _("Laptop computer"));

	if (major && minor && !strcmp(major, "computer")) {
		if (!strcmp(minor, "uncategorized"))
			index = 0;
		else if (!strcmp(minor, "desktop"))
			index = 1;
		else if (!strcmp(minor, "laptop"))
			index = 2;
		else
			index = -1;

		gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), index);
	} else
		gtk_widget_set_sensitive(GTK_WIDGET(combobox), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), combobox, FALSE, FALSE, 0);

	adapter->combo = combobox;

	g_signal_connect(G_OBJECT(combobox), "changed",
					G_CALLBACK(class_callback), adapter);

	gtk_widget_show_all(vbox);
}

static void adapter_free(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter_list = g_list_remove(adapter_list, adapter);

	g_free(adapter->path);
	g_free(adapter);
}

static void adapter_disable(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter->attached = 0;

	if (adapter->page >= 0)
		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), adapter->page);

	adapter->page = -1;
}

static gint adapter_compare(gconstpointer a, gconstpointer b)
{
	const struct adapter_data *adapter = a;
	const char *path = b;

	return strcmp(adapter->path, path);
}

static void mode_changed(DBusGProxy *object,
				const char *mode, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		GtkWidget *button = NULL;
		gboolean sensitive;

		if (!adapter->attached)
			return;

		sensitive = strcmp(mode, "off") ? TRUE : FALSE;
		
		gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_connect), sensitive);
		gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_visible), sensitive);

		if (!strcmp(mode, "connectable")) {
			sensitive = FALSE;
			button = adapter->button_connect;
		} else if (!strcmp(mode, "discoverable")) {
			sensitive = TRUE;
			button = adapter->button_visible;
		}

		gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_label), sensitive);
		gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_scale), sensitive);

		if (!button)
			return;

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	}
}

static void timeout_changed(DBusGProxy *object,
				const guint32 timeout, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		gdouble value;

		if (timeout == 0)
			value = 31;
		else if (timeout < 60)
			value = 1;
		else if (timeout > 60 * 30)
			value = 30;
		else
			value = timeout / 60;

		gtk_range_set_value(GTK_RANGE(adapter->timeout_scale), value);
	}
}

static void name_changed(DBusGProxy *object,
				const char *name, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		GtkWidget *child;

		if (!adapter->attached)
			return;

		child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook),
								adapter->page);

		gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook), child,
				name && strlen(name) > 0 ? name : _("Adapter"));

		gtk_entry_set_text(GTK_ENTRY(adapter->entry), name ? name : "");

		adapter->name_changed = 0;
	}
}

static void minor_changed(DBusGProxy *object,
				const char *minor, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		gint index;

		if (!adapter->attached)
			return;

		if (!strcmp(minor, "uncategorized"))
			index = 0;
		else if (!strcmp(minor, "desktop"))
			index = 1;
		else if (!strcmp(minor, "laptop"))
			index = 2;
		else
			index = -1;

		gtk_combo_box_set_active(GTK_COMBO_BOX(adapter->combo), index);
	}
}

static void add_adapter(const char *path)
{
	GList *list;
	DBusGProxy *object;
	struct adapter_data *adapter;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 1;

		create_adapter(adapter);

		return;
	}

        adapter = g_try_malloc0(sizeof(*adapter));
        if (!adapter)
		return;

	adapter->path = g_strdup(path);
	adapter->attached = 1;

	adapter_list = g_list_append(adapter_list, adapter);

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_add_signal(object, "ModeChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "ModeChanged",
				G_CALLBACK(mode_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "DiscoverableTimeoutChanged",
					G_TYPE_UINT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "DiscoverableTimeoutChanged",
				G_CALLBACK(timeout_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "NameChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "NameChanged",
				G_CALLBACK(name_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "MinorClassChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "MinorClassChanged",
				G_CALLBACK(minor_changed), NULL, NULL);

	create_adapter(adapter);
}

static void adapter_added(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	add_adapter(path);
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GList *list;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 0;

		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), adapter->page);

		adapter->page = -1;
	}
}

static int setup_manager(void)
{
	DBusGProxy *object;
	GError *error = NULL;
	const gchar **array = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");

	dbus_g_proxy_add_signal(object, "AdapterAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterAdded",
				G_CALLBACK(adapter_added), conn, NULL);

	dbus_g_proxy_add_signal(object, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterRemoved",
				G_CALLBACK(adapter_removed), conn, NULL);

	dbus_g_proxy_call(object, "ListAdapters", &error,
			G_TYPE_INVALID,	G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error == NULL) {
		while (*array) {
			add_adapter(*array);
			array++;
		}
	} else
		g_error_free(error);

	return 0;
}

static void name_owner_changed(DBusGProxy *object, const char *name,
			const char *prev, const char *new, gpointer user_data)
{
	if (!strcmp(name, "org.bluez") && *new == '\0')
		g_list_foreach(adapter_list, adapter_disable, NULL);
}

static int setup_dbus(void)
{
	DBusGProxy *object;

	object = dbus_g_proxy_new_for_name(conn, DBUS_SERVICE_DBUS,
					DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	dbus_g_proxy_add_signal(object, "NameOwnerChanged",
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "NameOwnerChanged",
				G_CALLBACK(name_owner_changed), NULL, NULL);

	return 0;
}

static void gconf_callback(GConfClient *client, guint cnxn_id,
					GConfEntry *entry, gpointer user_data)
{
	if (gconf_entry_get_value(entry) == NULL)
		return;

	if (strcmp(entry->key, PREF_ICON_POLICY) == 0) {
		char *str;

		str = gconf_client_get_string(client, entry->key, NULL);
		if (str) {
			gconf_string_to_enum(icon_policy_enum_map,
							str, &icon_policy);
			g_free(str);
		}

		update_icon_policy(NULL);
        }
}

int main(int argc, char *argv[])
{
	GError *error = NULL;
	char *str;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	gtk_init(&argc, &argv);

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr(_("Connecting to system bus failed: %s\n"),
							error->message);
		g_error_free(error);
		exit(EXIT_FAILURE);
	}

	gconf = gconf_client_get_default();

	str = gconf_client_get_string(gconf, PREF_ICON_POLICY, NULL);
	if (str) {
		gconf_string_to_enum(icon_policy_enum_map, str, &icon_policy);
		g_free(str);
	}

	gconf_client_add_dir(gconf, PREF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

	gconf_client_notify_add(gconf, PREF_DIR,
					gconf_callback, NULL, NULL, NULL);

	notebook = gtk_notebook_new();

	setup_dbus();

	setup_manager();

	create_window();

	gtk_main();

	g_object_unref(gconf);

	g_list_foreach(adapter_list, adapter_free, NULL);

	dbus_g_connection_unref(conn);

	return 0;
}
