/*  MCS
 *  Copyright (C) 2005-2006 M. Derezynski 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifndef HAVE_BMP

#include <mcs/config.h>
#if (1 != MCS_HAVE_XML)
#  error "This MCS installation was built without XML backend support!"
#endif

#endif

#ifndef MCS_BASE_H 
#define MCS_BASE_H 

#include <boost/variant.hpp>
#include <sigc++/sigc++.h>
#include <glibmm.h>

#include <mcs/types.h>
#include <mcs/key.h>
#include <mcs/subscriber.h>

#define MCS_CB_DEFAULT_SIGNATURE \
	    const ::std::string&    domain, \
	    const ::std::string&    key,    \
	    const ::Mcs::KeyVariant& value

namespace Mcs {

      class Mcs
      {
        public:

          enum VersionIgnore
          {
            VERSION_CHECK,	    //Checks the version specified in the root node and reports a mismatch
            VERSION_IGNORE	    //Ignores the version and just attempts to load the configuration 'as is'
          };

          enum Exceptions
          {
            PARSE_ERROR,
            NO_KEY,
          };

          Mcs	  (const std::string& xml_filename, const std::string& root_node_name, double version);
          ~Mcs	  ();
        
          void 
          register_domain (const std::string& domain);
            
          void 
          register_key    (const std::string& domain, //Must be registered
                           const std::string& key,
                           const KeyVariant&  key_default);
          void
          load (Mcs::VersionIgnore version_ignore = VERSION_CHECK); 

          bool
          domain_key_exist (const std::string& domain, const std::string& key);

          template <typename T> void 
          key_set (const std::string& domain,
                   const std::string& key,
                   T		              value) 
          {
            if (!domain_key_exist (domain, key))
              {
                g_warning ("The Key '%s' at Domain '%s' does not exist!", key.c_str(), domain.c_str());
                throw NO_KEY;
              }
            MKeys & keys = domains . find (domain)->second;
            Key & k = keys. find (key)->second;
            k. set_value<T>(value);
          }

          template <typename T> T 
          key_get	  (const std::string& domain,
                     const std::string& key)
          {
            if (!domain_key_exist (domain, key))
              {
                g_warning ("The Key '%s' at Domain '%s' does not exist!", key.c_str(), domain.c_str());
                throw NO_KEY;
              }
            return T (domains. find (domain)->second. find (key)->second);
          }
          
          void 
          key_unset	  (const std::string& domain,
               const std::string& key);

          void 
          subscribe	  (const std::string& name,   //Must be unique
                       const std::string& domain, //Must be registered 
                       const std::string& key,	//Must be registered,
                       SubscriberNotify   notify);  

          void 
          unsubscribe	  (const std::string& name,   //Must be unique
                         const std::string& domain, //Must be registered 
                         const std::string& key);	//Must be registered,

        private:

          typedef std::map<std::string /* Key Name */, Key>	  MKeys;
          typedef std::map<std::string /* Domain name */, MKeys>  MDomains;

          MDomains    domains;
          std::string xml_filename;
          std::string root_node_name;
          double      version;
      };
};

#endif //MCS_BASE_H 
