//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <glibmm.h>

#include <src/util_string.hh>
#include <src/util_file.hh>
#include <src/uri++.hh>

#include <src/vfs.hh>
#include <src/database.hh>

#include <src/x_library.hh>

namespace Bmp
{
  namespace VFS
  {
      class PluginContainerM3U
          : public Bmp::VFS::PluginContainerBase
      {
        public:

          virtual bool
          can_process (Glib::ustring const& uri)
          {
            Bmp::URI u (uri);
            return Bmp::Util::str_has_suffix_nocase (u.path.c_str(), "m3u"); 
          }

          virtual bool 
          handle_read	(Handle & handle,
                       VUri & list)
          {
            char **uris;
            int    n = 0;

            if (!handle.get_buffer()) throw ProcessingError ("Empty buffer"); 

            uris = g_uri_list_extract_uris ((const char*)handle.get_buffer());
            std::string pathname = Glib::path_get_dirname (Glib::filename_from_uri (handle.get_uri()));
          
            while (uris[n])
            {
              char *uri = uris[n];

              //Strip leading spaces/tabs
              while ((uri[0] == ' ') || (uri[0] == '\t') || (uri[0] == '<')) uri++;

              //Skip comments, extended information or null lenght uris
              if ((uri[0] == '#') || (uri[0] == '\0'))
                  {
                    n++;
                    continue;
                  }

              if (Glib::path_is_absolute (uri) && Glib::file_test (uri, Glib::FILE_TEST_EXISTS))
                {
                  try { list.push_back (Glib::filename_to_uri (uri)); } catch (...) {}
                }
              else		
                {
                  if (!g_ascii_strncasecmp ("http:", uri, 5))
                    {
                        list.push_back (uri);
                    }
                  else try
                    {
                      std::string filename = Glib::build_filename (pathname, uri);
                      if (Glib::file_test (filename, Glib::FILE_TEST_EXISTS))
                        {
                          list.push_back (Glib::filename_to_uri (filename));
                        }
                    }
                  catch (...) {}
                }
              n++;
            }

            g_strfreev (uris);
            return true;
          }

          virtual bool
          can_write ()
          {
            return true;
          }

          virtual bool
          handle_write  (Handle & handle, VUri const& uri_list)
          {
            std::stringstream out;
            std::string outstr;

            out << "#EXTM3U\n";

            for (VUri::const_iterator iter = uri_list . begin (); iter != uri_list . end (); ++iter)
              {
                  Bmp::Library::Track track;
                  try { library->get (*iter, track); } catch (Bmp::Library::Exception& cxe) { continue; }

                  if (!track.artist || !track.album || !track.title) continue;

                  out << "#EXTINF:" << track.artist.get() << "," << track.album.get() << " - " << track.title.get() << "\n";

                  Bmp::URI uri (*iter);
                  if (uri.get_protocol() != Bmp::URI::PROTOCOL_FILE)
                    { 		  
                      out << (*iter) << "\n";
                    }
                  else	    
                    {		      
                      try { out << Glib::filename_from_uri (*iter) << "\n"; } catch (...) {}
                    }
              }
            outstr = out.str(); 
            handle.set_buffer((const unsigned char *)strdup(outstr. c_str ()), strlen(outstr.c_str())+1);
            return true;
          }

          virtual Bmp::VFS::ExportData
          get_export_data ()
          {
            static Bmp::VFS::ExportData export_data ("M3U Playlist", "m3u"); 
            return export_data;
          }
      };
  }
}
  
extern "C" Bmp::VFS::PluginContainerBase* plugin_create ()
{
  return new Bmp::VFS::PluginContainerM3U;
}

extern "C" void plugin_delete (Bmp::VFS::PluginContainerM3U* p)
{
  delete p;
}
