//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <glibmm.h>

#include <src/database.hh>
#include <src/vfs.hh>

#include <src/util_string.hh>
#include <src/util_file.hh>
#include <src/uri++.hh>

#include <src/x_library.hh>
#include <src/x_vfs.hh>

#include <boost/algorithm/string.hpp>

using namespace Glib;

namespace Bmp
{
  namespace VFS
  {
      class PluginContainerMLQ
        : public Bmp::VFS::PluginContainerBase
      {
        public:

          virtual bool
          can_process (const ustring& uri)
          {
            return Bmp::Util::str_has_suffix_nocase (uri.c_str(), "mlq"); 
          }

          virtual bool 
          handle_read	(Handle & handle,
                       VUri & list)
          {
            if (!handle.get_buffer())
              throw ProcessingError ("Empty Buffer"); 

            typedef std::vector<std::string> StrV;
            StrV queries;

            using boost::algorithm::split;
            using boost::algorithm::is_any_of;

            std::string buffer = (const char*)handle.get_buffer();
            split (queries, buffer, is_any_of ("\n"));

            if (queries[0] != "#MLQ") return false;

            for (StrV::iterator sIter = queries.begin ()+1, sEnd = queries.end (); sIter != sEnd; ++sIter) 
              {
                if ((*sIter).empty()) continue;

                Bmp::URI u (*sIter);
                if (u.get_protocol() == Bmp::URI::PROTOCOL_QUERY)
                  {
                    VUri query_list;
                    Handle query_handle = Handle (ustring(*sIter));
                    vfs->read (query_handle, query_list, Bmp::VFS::CONTAINER);
                    std::copy (query_list.begin (), query_list.end (), std::back_inserter (list));
                  }
              }
            return true;
          }

          virtual bool
          can_write ()
          {
            return true;
          }

          virtual bool
          handle_write  (Handle & handle, VUri const& uri_list)
          {
            using namespace Bmp::DB;
            using namespace Bmp::Library;

            ustring outstr;

            std::string id_location = MetadatumId (DATUM_LOCATION); 
            VRows vector;

            for (VUri::const_iterator i  = uri_list.begin (); i != uri_list.end (); ++i)
              {
                VAttributes attrs;
                attrs.push_back (Attribute (DB::EXACT, id_location, *i)); 
                library->query (attrs, vector);
              }

            std::string id_artist = MetadatumId (DATUM_ARTIST); 
            std::string id_album  = MetadatumId (DATUM_ALBUM); 
            std::string id_title  = MetadatumId (DATUM_TITLE); 

            outstr.append ("#MLQ\n");

            for (VRows::iterator row = vector.begin (); row != vector.end (); ++row)
              {
                ustring uri;
                Row const& r = (*row);
          
                // FIXME: We should use Bmp::URI entirely to construct the query

                uri.append (Bmp::URI::get_protocol_scheme (Bmp::URI::PROTOCOL_QUERY));
                uri.append (":///?");
            
                Row::const_iterator rIter;
                rIter = r.find (id_artist);
                uri.append (id_artist);
                uri.append ("=");
                uri.append (boost::get<ustring>(rIter->second));
                
                uri.append ("&");

                rIter = r.find (id_album);
                uri.append (id_album);
                uri.append ("=");
                uri.append (boost::get<ustring>(rIter->second));

                uri.append ("&");

                rIter = r.find (id_title);
                uri.append (id_title);
                uri.append ("=");
                uri.append (boost::get<ustring>(rIter->second));

                Bmp::URI u (uri);
                u.escape ();
                outstr.append (ustring (u));
                outstr.append ("\n");
              }
            handle.set_buffer((const unsigned char *)strdup(outstr. c_str ()), strlen(outstr.c_str())+1);
            return true;
          }

          virtual Bmp::VFS::ExportData
          get_export_data ()
          {
            static Bmp::VFS::ExportData export_data ("MLQ Playlist", "mlq"); 
            return export_data;
          }
      };
  }
}
  
extern "C" Bmp::VFS::PluginContainerBase* plugin_create ()
{
  return new Bmp::VFS::PluginContainerMLQ;
}

extern "C" void plugin_delete (Bmp::VFS::PluginContainerMLQ* p)
{
  delete p;
}
