#ifndef BMP_PLAYBACK_SOURCE_HH
#define BMP_PLAYBACK_SOURCE_HH 

#include <glibmm.h>
#include <gtkmm.h>

#include <boost/optional.hpp>

namespace Bmp
{
    struct SimpleTrackInfo
    {
      boost::optional<Glib::ustring> artist;
      boost::optional<Glib::ustring> album;
      boost::optional<Glib::ustring> title;
      boost::optional<Glib::ustring> asin;
      boost::optional<unsigned int>  duration;
      Glib::RefPtr<Gdk::Pixbuf>      image;
      boost::optional<bool>          drawframe;

      SimpleTrackInfo () : duration (0), image (Glib::RefPtr<Gdk::Pixbuf>(0)) {}
    };

    enum PlaybackSourceID
    {
      SOURCE_NONE         = -1,
      SOURCE_ALBUMS       =  0,
      SOURCE_SHOUTCAST    =  1,
      SOURCE_ICECAST      =  2,
      SOURCE_CDDA         =  3,
      SOURCE_LASTFM       =  4,
      SOURCE_PODCASTS     =  5,
      SOURCE_PLAYLIST     =  6,
#ifdef HAVE_MOOD
      SOURCE_DOWNLOADS    =  7,
      N_SOURCES           =  8,
#else
      N_SOURCES           =  7,
#endif
    };

    class PlaybackSource
    {
      public:

#include "exception.hh"

      EXCEPTION(UnableToInitiatePlaybackError)
  
        enum Flags
        {
          F_NONE                = 0,
          F_HANDLE_STREAMINFO   = 1 << 0,
          F_PHONY_NEXT          = 1 << 1,
          F_PHONY_PREV          = 1 << 2,
        };

        enum Caps
        {
          NONE                  = 0,
          CAN_GO_NEXT           = 1 << 0,
          CAN_GO_PREV           = 1 << 1,
          CAN_PAUSE             = 1 << 2,
          CAN_PLAY              = 1 << 3,
          CAN_SEEK              = 1 << 4,
          CAN_RESTORE_CONTEXT   = 1 << 5, 
          CAN_PROVIDE_METADATA  = 1 << 6,
          PROVIDES_TIMING       = 1 << 7,
        };

        typedef sigc::signal<void, unsigned int> SignalPositionChanged;
        typedef sigc::signal<void, Caps> SignalCaps;
        typedef sigc::signal<void, Flags> SignalFlags;
        typedef sigc::signal<void, Glib::ustring, SimpleTrackInfo > SignalTrackInfo;
        typedef sigc::signal<void> SignalPlaybackRequest;
        typedef sigc::signal<void> SignalStopRequest;
        typedef sigc::signal<void, Glib::ustring> SignalErrorMessage;

        PlaybackSource (Caps caps = NONE, Flags flags = F_NONE) : m_caps (caps), m_flags (flags)
        {
        }

        virtual ~PlaybackSource () {}

        SignalErrorMessage&
        signal_error_message    () { return s_error_message_; }

        SignalPositionChanged&
        signal_position_changed () { return s_position_changed_; }

        SignalCaps&
        signal_caps             () { return s_caps_; }

        SignalFlags&
        signal_flags            () { return s_flags_; }

        SignalTrackInfo&
        signal_track_info       () { return s_track_info_; }

        SignalPlaybackRequest&
        signal_playback_request () { return s_playback_request_; }

        SignalStopRequest&
        signal_stop_request     () { return s_stop_request_; }

        virtual Glib::ustring
        get_uri () = 0; 
    
        virtual Glib::ustring
        get_type () = 0;

        virtual bool
        go_next () = 0;

        virtual bool
        go_prev () = 0;

        virtual void
        stop () = 0;
  
        virtual void
        play () = 0;

        virtual void
        play_requested () = 0;

        virtual void
        restore_context () = 0;

        void send_caps () const
        {
          s_caps_.emit (m_caps);
        }

        void send_flags () const
        {
          s_flags_.emit (m_flags);
        }

        virtual GHashTable*
        get_metadata () = 0;

      protected:

        SignalPositionChanged s_position_changed_;
        SignalCaps            s_caps_;
        SignalFlags           s_flags_;
        SignalTrackInfo       s_track_info_;
        SignalPlaybackRequest s_playback_request_;
        SignalStopRequest     s_stop_request_;
        SignalErrorMessage    s_error_message_;

        Caps                  m_caps;
        Flags                 m_flags;

    }; // PlaybackSource 
} // !Bmp:: 
  
#endif
