//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "main.hh"
#include "paths.hh"
#include "amazon.hh"

#include "popup.hh"
#include "ui_toolbox.hh"

#include <gtkmm.h>
#include <gdkmm.h>
#include <pangomm.h>
#include <cairomm/cairomm.h>
#include <boost/format.hpp>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <gdk/gdkx.h>
#include <cmath>
#include <sigc++/sigc++.h>
#include <iostream>

#include "x_play.hh"

namespace Bmp
{
  namespace
  {
    const int popup_width         = 380;
    const int popup_height        = 184;

    const int popup_animation_fps = 25;
    const int popup_animation_frame_period_ms = 1000 / popup_animation_fps;

    const double popup_full_alpha    = 1.0;
    const double popup_fade_in_time  = 0.15;
    const double popup_fade_out_time = 0.30;
    const double popup_hold_time     = 5.0;

    const double popup_total_time_no_fade = popup_hold_time;
    const double popup_total_time_fade    = popup_fade_in_time + popup_hold_time + popup_fade_out_time;

    const int arrow_width  = 26;
    const int arrow_height = 18;

    inline double
    cos_smooth (double x)
    {
      return (1.0 - std::cos (x * G_PI)) / 2.0;
    }

    double
    get_popup_alpha_at_time (double time,
                             bool   fading = true)
    {
      if (fading)
        {
          if (time >= 0.0 && popup_total_time_fade)
            {
              if (time < popup_fade_in_time)
                {
                  return popup_full_alpha * cos_smooth (time / popup_fade_in_time);
                }
              else if (time < popup_fade_in_time + popup_hold_time)
                {
                  return popup_full_alpha;
                }
              else
                {
                  time -= popup_fade_in_time + popup_hold_time;
                  return popup_full_alpha * cos_smooth (1.0 - time / popup_fade_out_time);
                }
            }
        }
      else
        {
          if (time >= 0.0 && time < popup_total_time_no_fade)
            return popup_full_alpha;
        }

      return 0.0;
    }

    inline double
    get_popup_end_time (bool fading = true)
    {
      return fading ? popup_total_time_fade : popup_total_time_no_fade;
    }

    inline double
    get_popup_time_offset (bool fading = true)
    {
      return fading ? popup_fade_in_time : 0.0;
    }

    inline double
    get_popup_disappear_start_time (bool fading = true)
    {
      return fading ? popup_fade_in_time + popup_hold_time : get_popup_end_time (fading);
    }

    void
    window_set_opacity (Glib::RefPtr<Gdk::Window> const& window,
                        double                           d)
    {
      const unsigned int opaque= 0xffffffff;
      const char * opacity_prop = "_NET_WM_WINDOW_OPACITY";

      unsigned int opacity = (unsigned int) (d * opaque);

      ::Display * dpy = gdk_x11_display_get_xdisplay (gdk_display_get_default());
      ::Window    win = GDK_WINDOW_XID (window->gobj());

      if (opacity == opaque)
        {
          XDeleteProperty (dpy, win, XInternAtom (dpy, opacity_prop, False));
        }
      else
        {
          XChangeProperty (dpy, win, XInternAtom (dpy, opacity_prop, False),
                           XA_CARDINAL, 32, PropModeReplace,
                           reinterpret_cast<unsigned char *> (&opacity), 1L);
          XSync (dpy, False);
        }
    }

  } // anonymous

  Popup::Popup (GtkWidget *                      widget,
                Gdk::Color const&                outline,
                Gdk::Color const&                inlay,
                Glib::RefPtr<Gdk::Pixbuf> const& image,
                Glib::ustring const&             text)

    : Gtk::Window     (Gtk::WINDOW_POPUP),
      m_widget        (widget),
      m_outline       (outline),
      m_inlay         (inlay),
      m_image         (image),
      m_source_icon   (Glib::RefPtr<Gdk::Pixbuf>(0)),
      m_text          (text),
      m_border        (true),
      m_time_offset   (0.0),
      m_has_alpha     (Util::has_alpha()),
      m_width         (popup_width),
      m_height        (popup_height),
      m_ax            (40),
      m_x             (0),
      m_y             (0),
      m_location      (ARROW_BOTTOM),
      m_position      (ARROW_POS_DEFAULT),
      m_tooltip_mode  (false)
  {
    if (m_has_alpha) set_colormap (Util::get_rgba_colormap());

    std::string family = get_pango_context()->get_font_description().get_family();

    m_layout = Pango::Layout::create (get_pango_context ());
    m_layout->set_ellipsize (Pango::ELLIPSIZE_END);
    m_layout->set_width (188 * PANGO_SCALE);

    PangoFontDescription *desc = pango_font_description_new ();
    pango_font_description_set_absolute_size (desc, 12 * PANGO_SCALE);
    pango_font_description_set_family (desc, family.c_str()); 
    pango_layout_set_font_description (m_layout->gobj(), desc);
    pango_font_description_free (desc);

    m_layout_l = Pango::Layout::create (get_pango_context()); 
    m_layout_r = Pango::Layout::create (get_pango_context()); 
    m_layout_l->set_justify ();
    m_layout_r->set_justify ();
    m_layout_l->set_alignment (Pango::ALIGN_LEFT);
    m_layout_r->set_alignment (Pango::ALIGN_RIGHT);
    m_layout_l->set_width (210 * PANGO_SCALE);
    m_layout_r->set_width (210 * PANGO_SCALE);

    desc = pango_font_description_new ();
    pango_font_description_set_absolute_size (desc, 12 * PANGO_SCALE);
    pango_font_description_set_family (desc, family.c_str()); 
    pango_layout_set_font_description (m_layout_l->gobj(), desc);
    pango_layout_set_font_description (m_layout_r->gobj(), desc);
    pango_font_description_free (desc);

    m_default_image = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT));

    add_events (Gdk::ALL_EVENTS_MASK);

    set_app_paintable (true);
    set_resizable (false);
    set_decorated (false);
    set_size_request (m_width, m_height);

    m_fade = Util::has_alpha();

    m_timer.stop ();
    m_timer.reset ();
  }

  void
  Popup::reposition ()
  {
    int x, y, width, height;
    acquire_widget_info (x, y, width, height);

    int new_x = x + width/2;
    int new_y = y - m_height;

    if (m_x != new_x || m_y != new_y)
      {
        m_x = new_x;
        m_y = new_y;

        m_location = ARROW_BOTTOM;
        m_position = ARROW_POS_DEFAULT;
        m_ax = 40;

        if (m_y < 0)
          {
            m_location = ARROW_TOP;
            m_y = y + height;
          }

        int screen_width = Gdk::Screen::get_default ()->get_width ();

        if (m_x + m_width > screen_width)
          {
            m_ax = m_width - 40;
            m_x = (screen_width - m_width) - (screen_width - m_x - 40);

            if (m_x + m_width > screen_width)
              {
                m_position = ARROW_POS_RIGHT;
                m_x -= (m_width - m_ax);
                m_ax  = 0;
              }
          }
        else
          {
            m_x -= 40;
            if (m_x < 0)
              {
                m_position = ARROW_POS_LEFT;
                m_ax  = 0;
                m_x  = 40 + m_x;
              }
          }

        move (m_x, m_y);

        update_mask ();
      }
  }

  void
  Popup::update_mask ()
  {
    if (!m_has_alpha && is_realized ())
      {
        Glib::RefPtr<Gdk::Bitmap> mask
          = Glib::RefPtr<Gdk::Bitmap>::cast_static (Gdk::Pixmap::create (Glib::RefPtr<Gdk::Drawable> (0), m_width, m_height, 1));

        Cairo::RefPtr<Cairo::Context> cr = Util::create_gdk_cairo_context (mask);

        int width, height;
        width = get_allocation ().get_width () - 2;
        height = get_allocation ().get_height ();

        draw_arrow_mask (cr, width, height);

        get_window ()->shape_combine_mask (mask, 0, 0);
      }
  }

  Popup::~Popup ()
  {
    // empty
  }

  bool
  Popup::on_button_press_event (GdkEventButton * event)
  {
    if (event->button == 1)
      disappear ();

    return false;
  }

  void
  Popup::draw_arrow (Cairo::RefPtr<Cairo::Context> & cr,
                     int                             w,
                     int                             h)
  {
    cr->save ();

    if (m_location == ARROW_TOP)
      {
        Cairo::Matrix matrix = { 1, 0, 0, -1, 0, h };
        cr->set_matrix (matrix);
      }

    cr->move_to (1, arrow_height);

    if (m_position == ARROW_POS_LEFT)
      {
        cr->rel_line_to (+w, 0);
        cr->rel_line_to (0, +(h-(2*arrow_height)));
        cr->rel_line_to (-(w-(arrow_width/2)), 0);
        cr->rel_line_to (-(arrow_width/2), +arrow_height);
        cr->rel_line_to (0, -(h-arrow_height));
      }
    else if (m_position == ARROW_POS_RIGHT)
      {
        cr->rel_line_to (+w, 0);
        cr->rel_line_to (0, +(h-arrow_height));
        cr->rel_line_to (-(arrow_width/2), -arrow_height);
        cr->rel_line_to (-(w-(arrow_width/2)), 0);
        cr->rel_line_to (0, -(h-(2*arrow_height)));
      }
    else
      {
        cr->rel_line_to (+w, 0);
        cr->rel_line_to (0, +(h-(2*arrow_height)));
        cr->rel_line_to (-((w-m_ax)-(arrow_width/2)), 0);
        cr->rel_line_to (-(arrow_width/2), +arrow_height);
        cr->rel_line_to (-(arrow_width/2), -arrow_height);
        cr->rel_line_to (-(m_ax-(arrow_width/2)), 0);
        cr->rel_line_to (0, -(h-(2*arrow_height)));
      }

    cr->restore ();
  }

  void
  Popup::draw_arrow_mask (Cairo::RefPtr<Cairo::Context> & cr,
                          int                             w,
                          int                             h)
  {
    cr->save ();

    // Clear the BG
    cr->set_operator (Cairo::OPERATOR_CLEAR);
    cr->set_source_rgba (1.0, 1.0, 1.0, 0.0);
    cr->paint ();

    cr->set_operator (Cairo::OPERATOR_SOURCE);
    cr->set_source_rgba (.0, .0, .0, 1.0);

    draw_arrow (cr, w, h);

    cr->fill ();

    cr->restore ();
  }

  void
  Popup::draw_arrow_outline (Cairo::RefPtr<Cairo::Context> & cr,
                             int                             w,
                             int                             h)
  {
    cr->save ();

    cr->set_line_width (1.5);
    cr->set_line_cap (Cairo::LINE_CAP_ROUND);
    cr->set_line_join (Cairo::LINE_JOIN_ROUND);
    cr->set_operator (Cairo::OPERATOR_SOURCE);

    Util::set_cairo_source_color (cr, m_inlay);
    draw_arrow (cr, w, h);

    cr->fill_preserve ();

    Util::set_cairo_source_color (cr, m_outline);
    cr->stroke();

    cr->restore ();
  }

  bool
  Popup::on_expose_event (GdkEventExpose * event)
  {
    Cairo::RefPtr<Cairo::Context> cr = Util::create_gdk_cairo_context (get_window ());

    int w, h;
    w = get_allocation ().get_width () - 2;
    h = get_allocation ().get_height ();

    // Clear the BG
    cr->set_operator (Cairo::OPERATOR_CLEAR);
    cr->set_source_rgba (1.0, 1.0, 1.0, 0.0);
    cr->paint ();

    draw_arrow_outline (cr, w, h);

    Util::set_cairo_source_color (cr, get_style()->get_bg (Gtk::STATE_SELECTED));
    cr->set_operator (Cairo::OPERATOR_SOURCE);
    cr->rectangle (3, arrow_height+2, 40-13-1, (h-(2*arrow_height))-4);
    cr->fill ();

    if (m_image)
      {
        get_window ()->draw_pixbuf (Glib::RefPtr<Gdk::GC> (0),
                                    m_image, 0, 0, 16, 28, -1, -1,
                                    Gdk::RGB_DITHER_MAX, 0, 0);
      }

    if (m_source_icon)
      {
        get_window ()->draw_pixbuf (Glib::RefPtr<Gdk::GC> (0),
                                    m_source_icon, 0, 0,
                                    380 - 10 - m_source_icon->get_width(), arrow_height + 10, 
                                    -1, -1,
                                    Gdk::RGB_DITHER_MAX, 0, 0);
      }

    if (!m_text.empty ())
      {
        cr->move_to (154, 28);
        cr->set_source_rgba (0., 0., 0., .9);
        cr->set_operator (Cairo::OPERATOR_ATOP);
        pango_cairo_show_layout (cr->cobj (), m_layout->gobj ());
      }

    if (m_border)
      {
        cr->set_line_width (2.0);
        cr->set_operator (Cairo::OPERATOR_ATOP);
        cr->set_source_rgba (0., 0., 0., 0.8);
        cr->rectangle (16, 28, 128, 128);
        cr->stroke ();
      }

    if (::play->property_status().get_value() != PLAYSTATUS_STOPPED)
    {
      double duration = ::play->property_length().get_value();
      double position = ::play->property_position().get_value();

      if (duration != 0.)
      {
        double d = position/duration; 

        cr->set_operator (Cairo::OPERATOR_ATOP);
        cr->set_source_rgba (0.82, 0.82, 0.82, 1.0);
        cr->rectangle (152, 134, 218 * d, 20);
        cr->fill ();
      
        cr->set_line_width (1.5);
        cr->set_operator (Cairo::OPERATOR_ATOP);
        cr->rectangle (152, 134, 218, 20);
        cr->set_source_rgba (0.8, 0.8, 0.8, 1.0);
        cr->stroke ();
      }

      static boost::format ftime ("%d:%02d");

      int pos_minutes = int(position) / 60;
      int pos_seconds = int(position) % 60;
      m_layout_l->set_text ((ftime % pos_minutes % pos_seconds).str());
      cr->set_operator( Cairo::OPERATOR_SOURCE );
      cr->move_to( 156, 136 ); 
      cr->set_source_rgba( 0., 0., 0., 1.);
      pango_cairo_show_layout (cr->cobj(), m_layout_l->gobj());

      if (duration != 0.)
      {
        int dur_minutes = int(duration) / 60;
        int dur_seconds = int(duration) % 60;
        m_layout_r->set_text ((ftime % dur_minutes % dur_seconds).str());
        cr->set_operator( Cairo::OPERATOR_SOURCE );
        cr->move_to( 156, 136 ); 
        cr->set_source_rgba( .4, .4, .4, 1.);
        pango_cairo_show_layout (cr->cobj(), m_layout_r->gobj());
      }
    }

    return false;
  }

  void
  Popup::tooltip_mode (bool mode, bool immediate)
  {
    if (G_UNLIKELY(!is_realized())) realize ();

    if (mode) 
    {
      m_update_connection.disconnect ();
      m_timer.stop ();
      m_timer.reset ();
      window_set_opacity (get_window (), 1.0);
      m_tooltip_mode = true;
      reposition ();
      Gtk::Window::show ();
    }
    else if (!mode)
    {
      if (m_tooltip_mode)
      {
        m_tooltip_mode = false;

        if (!immediate)
        {
          m_time_offset = 0.0;
          m_update_connection.disconnect ();
          m_timer.start ();
          m_update_connection = Glib::signal_timeout ().connect (sigc::mem_fun (this, &Bmp::Popup::update_frame),
                                                               popup_animation_frame_period_ms);
          disappear ();
        }
        else
        {
          window_set_opacity (get_window (), 1.0);
          hide ();
        }
      }
    }
  }

  void
  Popup::on_realize ()
  {
    Gtk::Window::on_realize ();

    window_set_opacity (get_window (), 0.0);

    update_mask ();
  }

  void
  Popup::on_show ()
  {
    reposition ();

    Gtk::Window::on_show ();
  }

  void
  Popup::on_map ()
  {
    Gtk::Window::on_map ();

    if (!m_tooltip_mode)
    {
      m_time_offset = 0.0;

      m_timer.start ();
      m_update_connection = Glib::signal_timeout ().connect (sigc::mem_fun (this, &Bmp::Popup::update_frame),
                                                           popup_animation_frame_period_ms);
    }
  }

  void
  Popup::on_unmap ()
  {
    if (!m_tooltip_mode)
    {
      m_update_connection.disconnect ();
      m_timer.stop ();
      m_timer.reset ();
    }

    Gtk::Window::on_unmap ();
  }

  void
  Popup::set_source_icon (Glib::RefPtr<Gdk::Pixbuf> icon)
  {
    m_source_icon = icon;
  }

  void
  Popup::set_info (SimpleTrackInfo const& sti)
  {
    using namespace Glib;

    m_text = Glib::ustring();
    m_text  .append ("<b>") 
            .append (sti.artist ? Glib::Markup::escape_text (sti.artist.get())  : "") 
            .append ("</b>\n") 
            .append ("<i>")
            .append (sti.album  ? Glib::Markup::escape_text (sti.album.get())   : "")
            .append ("</i>\n")
            .append (sti.title  ? Glib::Markup::escape_text (sti.title.get())   : "")
            .append ("\n");

    m_layout->set_markup (m_text);

    if (sti.asin && (sti.asin != m_asin))
      {
        try
          {
            RefPtr<Gdk::Pixbuf> cover;
            Amazon::fetch( sti.asin.get(), cover, true );
            m_image   = cover->scale_simple (128, 128, Gdk::INTERP_BILINEAR);
            m_border  = true;
            m_asin = sti.asin;
          }
        catch (Bmp::Amazon::Exception& cxe)
          {
            m_image   = m_default_image;
            m_border  = false;
            m_asin.reset();
          }
        catch (...)
          {
            m_image   = m_default_image;
            m_border  = false;
            m_asin.reset();
          }
      }
    else if (sti.image)
      {
        m_image = sti.image->scale_simple (128, 128, Gdk::INTERP_BILINEAR);

        if (sti.drawframe)
            m_border = sti.drawframe.get(); 
        else
            m_border = true;

        m_asin.reset();
      }
    else if (!sti.asin)
      {
        m_image   = m_default_image;
        m_border  = false;
        m_asin.reset();
      }

    queue_update ();
  }

  void
  Popup::queue_update ()
  {
    m_time_offset = get_popup_time_offset (m_fade);
    m_timer.reset ();

    if (is_mapped ())
      {
        reposition ();
        queue_draw ();
      }
  }

  void
  Popup::disappear ()
  {
    double time = m_timer.elapsed () + m_time_offset;
    double disappear_start_time = get_popup_disappear_start_time (m_fade);

    if (time < disappear_start_time)
      {
        m_timer.reset ();
        m_time_offset = disappear_start_time;
      }
  }

  void
  Popup::acquire_widget_info (int & x,
                              int & y,
                              int & width,
                              int & height)
  {
    gdk_flush ();
    while (gtk_events_pending ())
      gtk_main_iteration ();

    gdk_window_get_origin (m_widget->window, &x, &y);

    gdk_flush ();
    while (gtk_events_pending())
      gtk_main_iteration ();

    gdk_window_get_geometry (m_widget->window, NULL, NULL, &width, &height, NULL);
  }

  bool
  Popup::update_frame ()
  {
    double time;

    time = m_timer.elapsed () + m_time_offset;

    if (time < get_popup_end_time (m_fade))
      {
        double alpha = get_popup_alpha_at_time (time, m_fade);

        window_set_opacity (get_window (), alpha);

        return true;
      }
    else
      {
        if (!m_tooltip_mode) hide ();

        return false;
      }
  }

} // Bmp
