//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <iostream>
#include <sstream>

#include <glibmm.h>
#include <glibmm/markup.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>

#include <boost/format.hpp>
#include <sigc++/sigc++.h>

#include "neon++/request.hh"

#include "main.hh"
#include "xml.hh"
#include "util.hh"
#include "uri++.hh"

#include "streams-shoutcast.hh"

namespace
{
  const char* shoutcast_headers[] =
  {
    N_("Stream Name"),
    N_("Bitrate"),
    N_("Stream Genre")
  };
}

#define SHOUTCAST_HOST        "www.gwww.shoutcast.com"
#define SHOUTCAST_PATH        "/sbin/newxml.phtml"
#define SHOUTCAST_PATH_GENRE  "/sbin/newxml.phtml?genre="

namespace Bmp
{
  namespace Streams
  {
    namespace Shoutcast
    {
      Streams::Streams (BaseObjectType                        *cobject,
                        const Glib::RefPtr<Gnome::Glade::Xml> &xml)
            : Gtk::TreeView (cobject),
              m_streams (Gtk::ListStore::create (columns)),
              m_filtered (Gtk::TreeModelFilter::create (m_streams))
      {
        m_filtered->set_visible_func (sigc::mem_fun (this, &Bmp::Streams::Shoutcast::Streams::visible));

        for (unsigned int n = 0; n < G_N_ELEMENTS (shoutcast_headers); n++)
        {
          Gtk::CellRendererText *cell;
          cell = Gtk::manage (new Gtk::CellRendererText());
          append_column (_(shoutcast_headers[n]), *cell);
          get_column (n)->add_attribute (*cell, "text", n);
          get_column (n)->set_resizable (true);
          get_column (n)->signal_clicked ().connect
                    (sigc::bind (sigc::mem_fun (this, &Bmp::Streams::Shoutcast::Streams::column_clicked),n));
        }
        get_selection()->set_mode (Gtk::SELECTION_SINGLE);
        set_headers_clickable (true);
        set_model (m_filtered);
      }

      void
      Streams::column_clicked (int column)
      {
        Gtk::SortType sort_type, sort_type_new;
        int           sort_id;

        m_streams->get_sort_column_id (sort_id, sort_type);

        if ((sort_id >= 0) && (sort_id != column))
          {
            get_column (sort_id)->set_sort_indicator (false);
          }

        if (sort_id >= 0)
          sort_type_new = (sort_type == Gtk::SORT_ASCENDING) ? Gtk::SORT_DESCENDING : Gtk::SORT_ASCENDING;
        else
          sort_type_new = Gtk::SORT_ASCENDING;

        m_streams->set_sort_column_id (column, sort_type_new);
        get_column (column)->set_sort_indicator (true);
        get_column (column)->set_sort_order (sort_type_new);
      }

      bool
      Streams::visible (Gtk::TreeModel::iterator const& m_iter)
      {
        using namespace Glib;

        ustring name, genre, needle;

        if (!m_filter.length ()) return true;

        needle  = m_filter.casefold ();

        name  = (*m_iter)[columns.name];
        genre = (*m_iter)[columns.genre];
        name.casefold ();
        genre.casefold ();

        return ((Util::match_keys (name, needle)) || (Util::match_keys (genre, needle)));
      }

      void
      Streams::filter (Glib::ustring const& filter)
      {
        m_filter = filter;
        m_filtered->refilter ();
      }

      void 
      Streams::get_stream (Glib::ustring& title, Glib::ustring& uri)
      {
        using namespace Gtk;

        TreeModel::iterator m_iter = get_selection()->get_selected ();

        uri = (*m_iter)[columns.uri];
        title = (*m_iter)[columns.name];
      }

      void
      Streams::set_streams (StreamList const& list)
      {
        using namespace Gtk;

        m_streams->clear ();

        StreamList::const_iterator e = list.end();

        for (StreamList::const_iterator i = list.begin(); i != e; ++i)
        {
          StreamInfo const& info = *i;
          TreeModel::iterator m_iter = m_streams->append ();
          
          (*m_iter)[columns.name]     = info.name;
          (*m_iter)[columns.bitrate]  = info.bitrate;
          (*m_iter)[columns.genre]    = info.genre;
          (*m_iter)[columns.uri]      = info.uri;
        }
      }
    }
  }
}

namespace Bmp
{
  namespace Streams
  {
    namespace Shoutcast
    {
      void
      Genres::genre_cell_data (Gtk::CellRenderer                *cell,
                               Gtk::TreeModel::iterator const&   iter)
      {
        using namespace Glib;
        using namespace Gtk;

        CellRendererText *cell_text = reinterpret_cast<CellRendererText *>(cell);
        ustring genre = (*iter)[genres.name];

        if (g_hash_table_lookup (cache, genre.c_str()))
            cell_text->property_markup () = ((boost::format ("<b>%s</b>")) % std::string(Glib::Markup::escape_text(genre))).str();
        else
            cell_text->property_markup () = Glib::Markup::escape_text (genre);
      }

      void
      Genres::build_genre_list ()
      {
        using namespace Glib;
        using namespace Gtk;

        xmlDocPtr             doc;
        xmlXPathObjectPtr     xpathobj;
        xmlNodeSetPtr         nv;

        Neon::Request request (SHOUTCAST_HOST, SHOUTCAST_PATH); 

        try {
            request.dispatch (); 
          }
        catch (Neon::Request::Exception& cxe) 
          {
            return;
          }

        std::string data;
        request >> data;
        request.clear ();

        doc = xmlParseMemory ((const char*)(data.c_str()), data.size());
        if (!doc) return;

        xpathobj = xml_execute_xpath_expression (doc, BAD_CAST "//genre", NULL);
        nv = xpathobj->nodesetval;

        CellRendererText *cell;
        cell = manage ( new CellRendererText() );
        append_column (_("Genre"), *cell);
        get_column (0)->set_resizable (false);
        get_column (0)->set_sizing (TREE_VIEW_COLUMN_AUTOSIZE);

        m_genres = ListStore::create (genres);

        TreeModel::iterator m_iter;

        m_iter = m_genres->append ();
        (*m_iter)[genres.name] = "Top500";

        for (int n = 0; n < nv->nodeNr; n++)
          {
            xmlChar *prop = xmlGetProp (nv->nodeTab[n], BAD_CAST "name");
            m_iter = m_genres->append ();
            (*m_iter)[genres.name] = reinterpret_cast<char*>(prop);
            free (prop);
          }

        get_selection()->set_mode (SELECTION_BROWSE);
        get_selection()->select (TreeModel::Path("0"));
        get_selection()->signal_changed().connect(sigc::mem_fun (this, &Bmp::Streams::Shoutcast::Genres::refresh_wrap));
        get_column (0)->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::Streams::Shoutcast::Genres::genre_cell_data));

        xmlXPathFreeObject (xpathobj);
        xmlFreeDoc (doc);

        set_model (m_genres);

        while (gtk_events_pending ()) gtk_main_iteration();
      }

      void
      Genres::refresh_wrap ()
      {
        set_sensitive (false);
        refresh (false);
        set_sensitive (true);
      }

      void
      Genres::refresh (bool force)
      {
        using namespace Glib;
        using namespace Gtk;

        xmlDocPtr             doc;
        xmlXPathObjectPtr     xpathobj;
        xmlNodeSetPtr         nv;
        xmlNodePtr            node;

#define P_NAME      "name"
#define P_ID        "id"
#define P_BITRATE   "br"
#define P_GENRE     "genre"

        if (!get_selection()->count_selected_rows()) return;

        m_stream_list.clear ();
        s_start_.emit ();

        while (gtk_events_pending()) gtk_main_iteration();

        TreeModel::iterator iter = get_selection()->get_selected ();
        ustring genre = (*iter)[genres.name];
    
        if (g_hash_table_lookup (cache, genre.c_str()) && !force)
          {
            doc = static_cast<xmlDocPtr>(g_hash_table_lookup (cache, genre.c_str()));
          }
        else
          {
            if (force)
              {
                g_hash_table_remove (cache, genre.c_str());
              }

            std::string str (SHOUTCAST_PATH_GENRE);
            str.append (genre);
            
            Neon::Request request (SHOUTCAST_HOST, str); 
            std::string response;
            request >> response;
            request.clear ();

            doc = xmlParseMemory (response.c_str(), response.size()); 

            if (!doc)
              {
                s_stop_.emit ();
                return;
              }
            else
              {
                g_hash_table_insert (cache, strdup(genre.c_str()), gpointer(doc));
              }
          }

        xpathobj = xml_execute_xpath_expression (doc, BAD_CAST "//station", NULL);
        nv = xpathobj->nodesetval;

        for (int n = 0; n < nv->nodeNr; n++)
          {
            StreamInfo               info;
            xmlChar                 *prop;

            node = nv->nodeTab[n];
            prop = xmlGetProp (node, BAD_CAST P_ID);
            info.uri = Glib::ustring ("http://www.shoutcast.com/sbin/tunein-station.pls?id=").append(Glib::Markup::escape_text((char*)prop));
            free (prop);

            prop = xmlGetProp (node, BAD_CAST P_GENRE);
            info.genre = Glib::ustring(reinterpret_cast<char*>(prop));
            free (prop);

            prop = xmlGetProp (node, BAD_CAST P_NAME);
            info.name = Glib::ustring(reinterpret_cast<char*>(prop));
            free (prop);

            prop = xmlGetProp (node, BAD_CAST P_BITRATE);
            info.bitrate = atoi(reinterpret_cast<char*>(prop));
            free (prop);

            m_stream_list.push_back (info);

            while (gtk_events_pending()) gtk_main_iteration();
          }

        xmlXPathFreeObject (xpathobj);
        s_list_updated_.emit (m_stream_list);
        s_stop_.emit ();

        while (gtk_events_pending()) gtk_main_iteration();
      }

      Genres::Genres (BaseObjectType                        *cobject,
                      const Glib::RefPtr<Gnome::Glade::Xml> &xml)
          : Gtk::TreeView (cobject),
            cache (g_hash_table_new_full (g_str_hash,g_str_equal,GDestroyNotify(g_free),GDestroyNotify(xmlFreeDoc)))
      {
        build_genre_list ();
      }
    }
  }
}
