//  Bmp::URI (C) 2006 M. Derezynski 
//
//  Part of BMP (C) 2003-2006 BMP Project
//
//  ----
//  Based on:
//
//  GNet - Networking library
//  Copyright (C) 2000-2003  David Helder, David Bolcsfoldi, Eric Williams
//
//  ----
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  ----
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <glib.h>
#include <string>
#include <boost/algorithm/string.hpp>
#include <glibmm.h>
#include <iostream>

#include "debug.hh"
#include "uri++.hh"

#define USERINFO_ESCAPE_MASK	0x01
#define PATH_ESCAPE_MASK	0x02
#define QUERY_ESCAPE_MASK	0x04
#define FRAGMENT_ESCAPE_MASK	0x08

namespace 
{
   const char* protocols[] =
   {
	  "file",
	  "cdda",
	  "http", 
	  "ftp",
	  "mlq",
	  "track",
    "mms",
    "mmsu",
    "mmst",
    "lastfm"
  };

  char  *  field_unescape   (char * str);
  char  *  field_escape	    (char * str, guchar mask);

  guchar neednt_escape_table[] = 
  {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0e, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x0f, 0x00, 0x0c, 
    0x0e, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x00, 0x0f, 0x00, 0x00, 0x0f, 
    0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };

#define WITHIN_ESCAPE_TABLE_SIZE(_char) ((size_t)(_char) < (sizeof (neednt_escape_table)))
#define ISSPACE(C) (((C) >= 9 && (C) <= 13) || (C) == ' ')

  Glib::ustring dstr (char* v)
  {
      if (!v) return Glib::ustring();
      Glib::ustring val (v);
      g_free (v);
      return val;
  }

  char *
  field_escape (char * str, guchar mask)
  {
      int len = 0;
      unsigned int i = 0;
      bool must_escape = false;
      char* dst;
      gint j;

      if (!str)
        return 0;

      // Roughly calculate buffer size 
      len = 0;
      for (i = 0; str[i]; ++i)
      {
        if (WITHIN_ESCAPE_TABLE_SIZE (str[i]) && neednt_escape_table[(guint) str[i]] & mask)
        {
          len++;
        }
        else
        {
          len += 3;
          must_escape = true;
        }
      }

      // Don't escape if unnecessary
      if (must_escape == false)
          return str;

      // Allocate buffer
      dst = (char*) g_malloc (len + 1);

      // Copy 
      for (i = j = 0; str[i]; i++, j++)
      {
        // Unescaped character
        if (WITHIN_ESCAPE_TABLE_SIZE (str[i]) && neednt_escape_table[(guint) str[i]] & mask)
          {
            dst[j] = str[i];
          }

        // Escaped character
        else
          {
            dst[j] = '%';

              if (((str[i] & 0xf0) >> 4) < 10)
                  dst[j+1] = ((str[i] & 0xf0) >> 4) + '0';
              else
                  dst[j+1] = ((str[i] & 0xf0) >> 4) + 'a' - 10;

              if ((str[i] & 0x0f) < 10)
                  dst[j+2] = (str[i] & 0x0f) + '0';
              else
                  dst[j+2] = (str[i] & 0x0f) + 'a' - 10;

              j += 2;  // and j is incremented in loop too
          }
      }

      dst[j] = '\0';
      g_free (str);
      return dst;
    }

    char *
    field_unescape (char * _s)
    {
      char* s = strdup (_s);
      char* src;
      char* dst;

      for (src = dst = s; *src; ++src, ++dst)
        {
          if (src[0] == '%' && src[1] != '\0' && src[2] != '\0')
            {
              int high, low;

              if ('a' <= src[1] && src[1] <= 'f')
                  high = src[1] - 'a' + 10;
              else if ('A' <= src[1] && src[1] <= 'F')
                  high = src[1] - 'A' + 10;
              else if ('0' <= src[1] && src[1] <= '9')
                  high = src[1] - '0';
              else  // malformed
                  goto regular_copy;

              if ('a' <= src[2] && src[2] <= 'f')
                  low = src[2] - 'a' + 10;
              else if ('A' <= src[2] && src[2] <= 'F')
                  low = src[2] - 'A' + 10;
              else if ('0' <= src[2] && src[2] <= '9')
                  low = src[2] - '0';
              else  // malformed
                  goto regular_copy;

              *dst = (char)((high << 4) + low);
              src += 2;
            }
          else
            {
              regular_copy:
                *dst = *src;
            }
        }
      *dst = '\0';
      free (_s);
      return s;
    }
}

namespace Bmp
{
    bool
    URI::fragmentize (Glib::ustring const& _uri)
    {
      const char  * p;
      const char  * temp;
      char	      * uri;

      uri = strdup (_uri.c_str()); 

      //////// Skip initial whitespace
      p = uri;
      while (*p && ISSPACE ((int)*p)) ++p;
      if (!*p)	// Error if it's just a string of space
        return false;

      //////// Scheme
      temp = p;

      while (*p && *p != ':' && *p != '/' && *p != '?' && *p != '#') ++p;

      if (*p == ':')
      {
        scheme = dstr (g_strndup (temp, p - temp));
        ++p;
      }
      else
      /////// This char is NUL, /, ?, or # */
        p = temp;

      /////// Authority
      if (*p == '/' && p[1] == '/')
      {
        p += 2;

    		/////// Userinfo 
		    temp = p;

    		while (*p && *p != '@' && *p != '/' ) /* Look for @ or / */
		      ++p;

        if (*p == '@') /* Found userinfo */
		    {
          userinfo = dstr (g_strndup (temp, p - temp));
          ++p;
		    }
        else
		      p = temp;

        /////// Hostname
        // Check for no hostname at all (e.g. file://)  -- mderezynski
        if (*p == '/')
            goto path;

		    // Check for IPv6 canonical hostname in brackets
		    if (*p == '[')
		      {
            p++;  /* Skip [ */
			      temp = p;
			      while (*p && *p != ']') ++p;
			      if ((p - temp) == 0)
			        {
			          g_message ("Cond #1, %s", uri); 
			          goto error;
			        }
			      hostname = dstr (g_strndup (temp, p - temp));
			      if (*p)
			        p++;	// Skip ] (if there)
		      }
		    else
		      {
			      temp = p;
			      while (*p && *p != '/' && *p != '?' && *p != '#' && *p != ':') ++p;
			      if ((p - temp) == 0) 
			        {
			          g_message ("Cond #2, %s", uri); 
			          goto error;
			        }
			      hostname = dstr (g_strndup (temp, p - temp));
		      }

        /////// Port
		    if (*p == ':')
		      {
			      for (++p; isdigit ((int)*p); ++p)
			      port = port * 10 + (*p - '0');
		      }

	      }

        path:

	      // Path (we are liberal and won't check if it starts with /)
	      temp = p;
  
	      while (*p && *p != '?')
	        ++p;
	      if (p != temp)
	        path = dstr (g_strndup (temp, p - temp));

	      // Query
	      if (*p == '?')
	        {
		        temp = p + 1;
        		while (*p && *p != '#')
		         ++p;
        		query = dstr (g_strndup (temp, p - temp));
	        }

        // Fragment
      	if (*p == '#')
	        {
        		++p;
        		fragment = dstr (g_strdup (p));
          }

      	g_free (uri);
      	return true;

        error:
      	  return false;
    }

    URI::URI ()
        : port (0)
    {}

    URI::URI (Glib::ustring const &str, bool do_escape)
        : port (0)
    {
      debug ("uri++", "Constructing URI from '%s'", str.c_str());

      if (!fragmentize (str))
        throw ParseError(); 

      if (do_escape)
        escape ();

      if ((port == 0) && (get_protocol() == PROTOCOL_HTTP))
        port = 80;
    }

    void
    URI::escape ()
    {
      userinfo  = dstr (field_escape (strdup (userinfo.c_str()), USERINFO_ESCAPE_MASK));
      path      = dstr (field_escape (strdup (path.c_str()),     PATH_ESCAPE_MASK));
      query     = dstr (field_escape (strdup (query.c_str()),    QUERY_ESCAPE_MASK));
      fragment  = dstr (field_escape (strdup (fragment.c_str()), FRAGMENT_ESCAPE_MASK));
    }
  
    void
    URI::parse_query (Query& q)
    {
      using namespace boost;
      using namespace std;
      typedef vector < string > Fragments;
      Fragments fragments;
      split (fragments, query, is_any_of("&"));
    
      for (Fragments::const_iterator frag = fragments.begin (),
				     end  = fragments.end   (); frag != end; ++frag)
      {
        Fragments sub;
        split (sub, *frag, is_any_of("="));
        QElement element (std::make_pair (sub[0], sub[1]));
        q.insert (std::make_pair (sub[0], element)); 
      }
    }
  
    void
    URI::unescape ()
    {
      if (!userinfo.empty())
        userinfo = dstr (field_unescape (strdup (userinfo.c_str())));

      if (!path.empty())
        path = dstr (field_unescape (strdup (path.c_str())));

      if (!query.empty())
        query = dstr (field_unescape (strdup (query.c_str())));

      if (!userinfo.empty())
        fragment = dstr (field_unescape (strdup (fragment.c_str())));
    }

    std::string 
    URI::get_protocol_scheme (Bmp::URI::Protocol p)
    {
      return protocols[p];
    }

    Bmp::URI::Protocol
    URI::get_protocol ()
    {
      Bmp::URI::Protocol protocol = URI::PROTOCOL_UNKNOWN;

      for (unsigned int n = 0; n < G_N_ELEMENTS (protocols); ++n)
      {
        if (scheme == protocols[n])
        {
          protocol = Bmp::URI::Protocol(n);
          break;
        }
      }
      return protocol;
    }
  
    std::string
    URI::fullpath ()
    {
      std::string p (path);

      if (query.size())
      {
        p.append ("?");
        p.append (query);
      }

      if (fragment.size())
      {
        p.append ("#");
        p.append (fragment);
      }

      return p;
    }

    URI::operator Glib::ustring ()
    {
      GString* buffer = NULL;
      buffer = g_string_sized_new (16);

      if (!scheme.empty())
          g_string_sprintfa (buffer, "%s:", scheme.c_str());

      if ((!scheme.compare(protocols[PROTOCOL_FILE])) || 
          (!scheme.compare(protocols[PROTOCOL_CDDA])) || 
          (!scheme.compare(protocols[PROTOCOL_HTTP])) || 
          (!scheme.compare(protocols[PROTOCOL_LASTFM])) || 
          (!scheme.compare(protocols[PROTOCOL_QUERY])))
      {
          g_string_append (buffer, "//");
      }

      if (!userinfo.empty())
          {
            buffer = g_string_append   (buffer,  userinfo.c_str());
            buffer = g_string_append_c (buffer, '@');
          }

      // Add brackets around the hostname if it's IPv6
      if (!hostname.empty())
          {
        if (strchr (hostname.c_str(), ':') == NULL) 
            buffer = g_string_append (buffer, hostname.c_str()); 
        else
            g_string_sprintfa (buffer, "[%s]", hostname.c_str());
          }

      if (port)
          g_string_sprintfa (buffer, ":%d", port);

      if (!path.empty())
          {
        if (*(path.c_str()) == '/' ||
            !(!userinfo.empty() || !hostname.empty() || port))
            g_string_append (buffer, path.c_str());
        else
            g_string_sprintfa (buffer, "/%s", path.c_str());
          }

      if (!query.empty())
          g_string_sprintfa (buffer, "?%s", query.c_str());

      if (!fragment.empty())
          g_string_sprintfa (buffer, "#%s", fragment.c_str());

      // Free only GString not data contained, return the data instead
      Glib::ustring rvstd (buffer->str);
      g_string_free (buffer, TRUE); 
      return rvstd;
    }
}
