/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "eventlog.h"
#include "util.h"
#include "bnettime.h"
#include "account.h"


extern unsigned int account_get_numattr(t_account * account, char const * key)
{
    char const * temp;
    unsigned int val;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_numattr","got NULL account");
	return 0;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_get_numattr","got NULL key");
	return 0;
    }
    
    if (!(temp = account_get_strattr(account,key)))
	return 0;
    
    if (str_to_uint(temp,&val)<0)
    {
	eventlog(eventlog_level_error,"account_get_numattr","not a numeric string \"%s\" for key \"%s\"",temp,key);
	account_unget_strattr(temp);
	return 0;
    }
    account_unget_strattr(temp);
    
    return val;
}


extern int account_set_numattr(t_account * account, char const * key, unsigned int val)
{
    char temp[32]; /* should be more than enough room */
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_set_numattr","got NULL account");
	return -1;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_set_numattr","got NULL key");
	return -1;
    }
    
    sprintf(temp,"%u",val);
    return account_set_strattr(account,key,temp);
}


extern int account_get_boolattr(t_account * account, char const * key)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_boolattr","got NULL account");
	return -1;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_get_boolattr","got NULL key");
	return -1;
    }
    
    if (!(temp = account_get_strattr(account,key)))
	return -1;
    
    if (strcasecmp(temp,"true")==0 ||
        strcasecmp(temp,"yes")==0 ||
        strcasecmp(temp,"on")==0 ||
        strcmp(temp,"1")==0)
    {
	account_unget_strattr(temp);
	return 1;
    }
    if (strcasecmp(temp,"false")==0 ||
	strcasecmp(temp,"no")==0 ||
	strcasecmp(temp,"off")==0 ||
	strcmp(temp,"0")==0)
    {
	account_unget_strattr(temp);
	return 0;
    }
    
    account_unget_strattr(temp);
    eventlog(eventlog_level_error,"account_get_boolattr","bad boolean value \"%s\" for key \"%s\"",temp,key);
    return -1;
}


extern int account_set_boolattr(t_account * account, char const * key, int val)
{
    if (!account)
    {
	eventlog(eventlog_level_error,"account_set_boolattr","got NULL account");
	return -1;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_set_boolattr","got NULL key");
	return -1;
    }
    
    return account_set_strattr(account,key,val?"true":"false");
}


/****************************************************************/


extern char const * account_get_name(t_account * account)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_name","got NULL account");
	return NULL; /* FIXME: places assume this can't fail */
    }
    
    if (!(temp = account_get_strattr(account,"BNET\\acct\\username")))
	eventlog(eventlog_level_error,"account_get_name","account has no username");
    return temp;
}


extern void account_unget_name(char const * name)
{
    account_unget_strattr(name);
}


extern char const * account_get_pass(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\passhash1");
}


extern int account_set_pass(t_account * account, char const * passhash1)
{
    return account_set_strattr(account,"BNET\\acct\\passhash1",passhash1);
}


extern void account_unget_pass(char const * pass)
{
    account_unget_strattr(pass);
}


extern unsigned int account_get_uid(t_account * account)
{
    unsigned int temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_uid","got NULL account");
	return 0;
    }
    
    if ((temp = account_get_numattr(account,"BNET\\acct\\userid"))<1)
	eventlog(eventlog_level_error,"account_get_uid","account has no userid");
    return temp;
}


/****************************************************************/


extern int account_get_auth_admin(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\admin");
}


extern int account_get_auth_announce(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\announce");
}


extern int account_get_auth_botlogin(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\botlogin");
}


extern int account_get_auth_normallogin(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\normallogin");
}


extern int account_get_auth_operator(t_account * account, char const * channelname)
{
    char temp[256];
    
    if (!channelname)
	return account_get_boolattr(account,"BNET\\auth\\operator");
    
    sprintf(temp,"BNET\\auth\\operator\\%.100s",channelname);
    return account_get_boolattr(account,temp);
}


extern int account_get_auth_changepass(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\changepass");
}


extern int account_get_auth_changeprofile(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\changeprofile");
}


extern int account_get_auth_createnormalgame(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\createnormalgame");
}


extern int account_get_auth_joinnormalgame(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\joinnormalgame");
}


extern int account_get_auth_createladdergame(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\createladdergame");
}


extern int account_get_auth_joinladdergame(t_account * account)
{
    return account_get_boolattr(account,"BNET\\auth\\joinladdergame");
}




/****************************************************************/


extern char const * account_get_sex(t_account * account)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_sex","got NULL account");
	return NULL;
    }
    
    if (!(temp = account_get_strattr(account,"profile\\sex")))
	return "";
    return temp;
}


extern void account_unget_sex(char const * sex)
{
    account_unget_strattr(sex);
}


extern char const * account_get_age(t_account * account)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_age","got NULL account");
	return NULL;
    }
    
    if (!(temp = account_get_strattr(account,"profile\\age")))
	return "";
    return temp;
}


extern void account_unget_age(char const * age)
{
    account_unget_strattr(age);
}


extern char const * account_get_loc(t_account * account)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_loc","got NULL account");
	return NULL;
    }
    
    if (!(temp = account_get_strattr(account,"profile\\location")))
	return "";
    return temp;
}


extern void account_unget_loc(char const * loc)
{
    account_unget_strattr(loc);
}


extern char const * account_get_desc(t_account * account)
{
    char const * temp;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_desc","got NULL account");
	return NULL;
    }
    
    if (!(temp = account_get_strattr(account,"profile\\description")))
	return "";
    return temp;
}


extern void account_unget_desc(char const * desc)
{
    account_unget_strattr(desc);
}


/****************************************************************/


extern unsigned int account_get_fl_time(t_account * account)
{
    return account_get_numattr(account,"BNET\\acct\\firstlogin_time");
}


extern void account_set_fl_time(t_account * account, unsigned int t)
{
    account_set_numattr(account,"BNET\\acct\\firstlogin_time",t);
}


extern char const * account_get_fl_clientexe(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_clientexe");
}


extern void account_unget_fl_clientexe(char const * clientexe)
{
    account_unget_strattr(clientexe);
}


extern void account_set_fl_clientexe(t_account * account, char const * exefile)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_clientexe",exefile);
}


extern char const * account_get_fl_clientver(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_clientver");
}


extern void account_unget_fl_clientver(char const * clientver)
{
    account_unget_strattr(clientver);
}


extern void account_set_fl_clientver(t_account * account, char const * version)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_clientver",version);
}


extern char const * account_get_fl_clienttag(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_clienttag");
}


extern void account_unget_fl_clienttag(char const * clienttag)
{
    account_unget_strattr(clienttag);
}


extern void account_set_fl_clienttag(t_account * account, char const * clienttag)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_clienttag",clienttag);
}


extern unsigned int account_get_fl_connection(t_account * account)
{
    return account_get_numattr(account,"BNET\\acct\\firstlogin_connection");
}


extern void account_set_fl_connection(t_account * account, unsigned int connection)
{
    account_set_numattr(account,"BNET\\acct\\firstlogin_connection",connection);
}


extern char const * account_get_fl_host(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_host");
}


extern void account_unget_fl_host(char const * host)
{
    account_unget_strattr(host);
}


extern void account_set_fl_host(t_account * account, char const * host)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_host",host);
}


extern char const * account_get_fl_user(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_user");
}


extern void account_unget_fl_user(char const * user)
{
    account_unget_strattr(user);
}


extern void account_set_fl_user(t_account * account, char const * user)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_user",user);
}


extern char const * account_get_fl_owner(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_owner");
}


extern void account_unget_fl_owner(char const * owner)
{
    account_unget_strattr(owner);
}


extern void account_set_fl_owner(t_account * account, char const * owner)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_owner",owner);
}


extern char const * account_get_fl_cdkey(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\firstlogin_cdkey");
}


extern void account_unget_fl_cdkey(char const * cdkey)
{
    account_unget_strattr(cdkey);
}


extern void account_set_fl_cdkey(t_account * account, char const * cdkey)
{
    account_set_strattr(account,"BNET\\acct\\firstlogin_cdkey",cdkey);
}


/****************************************************************/


extern unsigned int account_get_ll_time(t_account * account)
{
    return account_get_numattr(account,"BNET\\acct\\lastlogin_time");
}


extern void account_set_ll_time(t_account * account, unsigned int t)
{
    account_set_numattr(account,"BNET\\acct\\lastlogin_time",t);
}


extern char const * account_get_ll_clientexe(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_clientexe");
}


extern void account_unget_ll_clientexe(char const * clientexe)
{
    account_unget_strattr(clientexe);
}


extern void account_set_ll_clientexe(t_account * account, char const * exefile)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_clientexe",exefile);
}


extern char const * account_get_ll_clientver(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_clientver");
}


extern void account_unget_ll_clientver(char const * clientver)
{
    account_unget_strattr(clientver);
}


extern void account_set_ll_clientver(t_account * account, char const * version)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_clientver",version);
}


extern char const * account_get_ll_clienttag(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_clienttag");
}


extern void account_unget_ll_clienttag(char const * clienttag)
{
    account_unget_strattr(clienttag);
}


extern void account_set_ll_clienttag(t_account * account, char const * clienttag)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_clienttag",clienttag);
}


extern unsigned int account_get_ll_connection(t_account * account)
{
    return account_get_numattr(account,"BNET\\acct\\lastlogin_connection");
}


extern void account_set_ll_connection(t_account * account, unsigned int connection)
{
    account_set_numattr(account,"BNET\\acct\\lastlogin_connection",connection);
}


extern char const * account_get_ll_host(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_host");
}


extern void account_unget_ll_host(char const * host)
{
    account_unget_strattr(host);
}


extern void account_set_ll_host(t_account * account, char const * host)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_host",host);
}


extern char const * account_get_ll_user(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_user");
}


extern void account_unget_ll_user(char const * user)
{
    account_unget_strattr(user);
}


extern void account_set_ll_user(t_account * account, char const * user)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_user",user);
}


extern char const * account_get_ll_owner(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_owner");
}


extern void account_unget_ll_owner(char const * owner)
{
    account_unget_strattr(owner);
}


extern void account_set_ll_owner(t_account * account, char const * owner)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_owner",owner);
}


extern char const * account_get_ll_cdkey(t_account * account)
{
    return account_get_strattr(account,"BNET\\acct\\lastlogin_cdkey");
}


extern void account_unget_ll_cdkey(char const * cdkey)
{
    account_unget_strattr(cdkey);
}


extern void account_set_ll_cdkey(t_account * account, char const * cdkey)
{
    account_set_strattr(account,"BNET\\acct\\lastlogin_cdkey",cdkey);
}


/****************************************************************/


extern unsigned int account_get_normal_wins(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_wins","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\0\\wins",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_normal_wins(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_inc_normal_wins","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\wins",clienttag);
    account_set_numattr(account,key,account_get_normal_wins(account,clienttag)+1);
}


extern unsigned int account_get_normal_losses(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_losses","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\0\\losses",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_normal_losses(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_inc_normal_losses","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\losses",clienttag);
    account_set_numattr(account,key,account_get_normal_losses(account,clienttag)+1);
}


extern unsigned int account_get_normal_draws(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_draws","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\0\\draws",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_normal_draws(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_inc_normal_draws","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\draws",clienttag);
    account_set_numattr(account,key,account_get_normal_draws(account,clienttag)+1);
}


extern unsigned int account_get_normal_disconnects(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_disconnects","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\0\\disconnects",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_normal_disconnects(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_inc_normal_disconnects","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\disconnects",clienttag);
    account_set_numattr(account,key,account_get_normal_disconnects(account,clienttag)+1);
}


extern void account_set_normal_last_time(t_account * account, char const * clienttag, t_bnettime t)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_last_time","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\last game",clienttag);
    account_set_strattr(account,key,bnettime_get_str(t));
}


extern void account_set_normal_last_result(t_account * account, char const * clienttag, char const * result)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_last_result","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\0\\last game result",clienttag);
    account_set_strattr(account,key,result);
}


/****************************************************************/


extern unsigned int account_get_ladder_active_wins(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_wins","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active wins",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_wins(t_account * account, char const * clienttag, unsigned int wins)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_wins","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active wins",clienttag);
    account_set_numattr(account,key,wins);
}


extern unsigned int account_get_ladder_active_losses(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_losses","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active losses",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_losses(t_account * account, char const * clienttag, unsigned int losses)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_losses","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active losses",clienttag);
    account_set_numattr(account,key,losses);
}


extern unsigned int account_get_ladder_active_draws(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_draws","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active draws",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_draws(t_account * account, char const * clienttag, unsigned int draws)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_draws","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active draws",clienttag);
    account_set_numattr(account,key,draws);
}


extern unsigned int account_get_ladder_active_disconnects(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_disconnects","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active disconnects",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_disconnects(t_account * account, char const * clienttag, unsigned int disconnects)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_disconnects","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active disconnects",clienttag);
    account_set_numattr(account,key,disconnects);
}


extern unsigned int account_get_ladder_active_rating(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_rating","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active rating",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_rating(t_account * account, char const * clienttag, unsigned int rating)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_rating","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active rating",clienttag);
    account_set_numattr(account,key,rating);
}


extern unsigned int account_get_ladder_active_rank(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_wins","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\active rank",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_active_rank(t_account * account, char const * clienttag, unsigned int rank)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_rank","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active rank",clienttag);
    account_set_numattr(account,key,rank);
}


extern char const * account_get_ladder_active_last_time(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_active_last_time","got bad clienttag");
	return NULL;
    }
    sprintf(key,"Record\\%s\\1\\active last game",clienttag);
    return account_get_strattr(account,key);
}


extern void account_set_ladder_active_last_time(t_account * account, char const * clienttag, t_bnettime t)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_active_last_time","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\active last game",clienttag);
    account_set_strattr(account,key,bnettime_get_str(t));
}


/****************************************************************/


extern unsigned int account_get_ladder_wins(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_wins","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\wins",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_ladder_wins(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_inc_ladder_wins","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\wins",clienttag);
    account_set_numattr(account,key,account_get_ladder_wins(account,clienttag)+1);
}


extern unsigned int account_get_ladder_losses(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_losses","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\losses",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_ladder_losses(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
       eventlog(eventlog_level_error,"account_inc_ladder_losses","got bad clienttag");
       return;
    }
    sprintf(key,"Record\\%s\\1\\losses",clienttag);
    account_set_numattr(account,key,account_get_ladder_losses(account,clienttag)+1);
}


extern unsigned int account_get_ladder_draws(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_draws","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\draws",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_ladder_draws(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
       eventlog(eventlog_level_error,"account_inc_ladder_draws","got bad clienttag");
       return;
    }
    sprintf(key,"Record\\%s\\1\\draws",clienttag);
    account_set_numattr(account,key,account_get_ladder_draws(account,clienttag)+1);
}


extern unsigned int account_get_ladder_disconnects(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_disconnects","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\disconnects",clienttag);
    return account_get_numattr(account,key);
}


extern void account_inc_ladder_disconnects(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
       eventlog(eventlog_level_error,"account_inc_ladder_disconnects","got bad clienttag");
       return;
    }
    sprintf(key,"Record\\%s\\1\\disconnects",clienttag);
    account_set_numattr(account,key,account_get_ladder_disconnects(account,clienttag)+1);
}


extern unsigned int account_get_ladder_rating(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_rating","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\rating",clienttag);
    return account_get_numattr(account,key);
}


extern void account_adjust_ladder_rating(t_account * account, int delta, char const * clienttag)
{
    char         key[256];
    unsigned int oldrating;
    unsigned int newrating;
    
    if (!clienttag || strlen(clienttag)!=4)
    {
       eventlog(eventlog_level_error,"account_adjust_ladder_rating","got bad clienttag");
       return;
    }
    sprintf(key,"Record\\%s\\1\\rating",clienttag);
    /* don't allow rating to go below 1 */
    oldrating = account_get_ladder_rating(account,clienttag);
    if (delta<0 && oldrating<=abs(delta))
        newrating = 1;
    else
        newrating = oldrating+delta;
    account_set_numattr(account,key,newrating);
    
    if (newrating>account_get_ladder_high_rating(account,clienttag))
    {
	sprintf(key,"Record\\%s\\1\\high rating",clienttag);
	account_set_numattr(account,key,newrating);
    }
}


extern unsigned int account_get_ladder_rank(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_rank","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\rank",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_rank(t_account * account, unsigned int rank, char const * clienttag)
{
    char         key[256];
    unsigned int oldrank;
    
    if (!clienttag || strlen(clienttag)!=4)
    {
       eventlog(eventlog_level_error,"account_set_ladder_rank","got bad clienttag");
       return;
    }
    if (rank==0)
        eventlog(eventlog_level_warn,"account_set_ladder_rank","setting rank to zero?");
    sprintf(key,"Record\\%s\\1\\rank",clienttag);
    account_set_numattr(account,key,rank);
    
    oldrank = account_get_ladder_high_rank(account,clienttag);
    if (oldrank==0 || rank<oldrank)
    {
	sprintf(key,"Record\\%s\\1\\high rank",clienttag);
	account_set_numattr(account,key,rank);
    }
}


extern unsigned int account_get_ladder_high_rating(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_high_rating","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\high rating",clienttag);
    return account_get_numattr(account,key);
}


extern unsigned int account_get_ladder_high_rank(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_high_rank","got bad clienttag");
	return 0;
    }
    sprintf(key,"Record\\%s\\1\\high_rank",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_ladder_last_time(t_account * account, char const * clienttag, t_bnettime t)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_last_time","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\last game",clienttag);
    account_set_strattr(account,key,bnettime_get_str(t));
}


extern char const * account_get_ladder_last_time(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_ladder_last_time","got bad clienttag");
	return NULL;
    }
    sprintf(key,"Record\\%s\\1\\last game",clienttag);
    return account_get_strattr(account,key);
}


extern void account_set_ladder_last_result(t_account * account, char const * clienttag, char const * result)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_ladder_last_result","got bad clienttag");
	return;
    }
    sprintf(key,"Record\\%s\\1\\last game result",clienttag);
    account_set_strattr(account,key,result);
}


/****************************************************************/


extern unsigned int account_get_normal_level(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_level","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\level",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_level(t_account * account, char const * clienttag, unsigned int level)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_level","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\level",clienttag);
    account_set_numattr(account,key,level);
}


extern unsigned int account_get_normal_class(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_class","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\class",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_class(t_account * account, char const * clienttag, unsigned int class)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_class","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\class",clienttag);
    account_set_numattr(account,key,class);
}


extern unsigned int account_get_normal_diablo_kills(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_diablo_kills","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\diablo kills",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_diablo_kills(t_account * account, char const * clienttag, unsigned int diablo_kills)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_diablo_kills","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\diablo kills",clienttag);
    account_set_numattr(account,key,diablo_kills);
}


extern unsigned int account_get_normal_strength(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_strength","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\strength",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_strength(t_account * account, char const * clienttag, unsigned int strength)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_strength","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\strength",clienttag);
    account_set_numattr(account,key,strength);
}


extern unsigned int account_get_normal_magic(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_magic","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\magic",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_magic(t_account * account, char const * clienttag, unsigned int magic)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_magic","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\magic",clienttag);
    account_set_numattr(account,key,magic);
}


extern unsigned int account_get_normal_dexterity(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_dexterity","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\dexterity",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_dexterity(t_account * account, char const * clienttag, unsigned int dexterity)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_dexterity","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\dexterity",clienttag);
    account_set_numattr(account,key,dexterity);
}


extern unsigned int account_get_normal_vitality(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_vitality","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\vitality",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_vitality(t_account * account, char const * clienttag, unsigned int vitality)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_vitality","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\vitality",clienttag);
    account_set_numattr(account,key,vitality);
}


extern unsigned int account_get_normal_gold(t_account * account, char const * clienttag)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_get_normal_gold","got bad clienttag");
	return 0;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\gold",clienttag);
    return account_get_numattr(account,key);
}


extern void account_set_normal_gold(t_account * account, char const * clienttag, unsigned int gold)
{
    char key[256];
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"account_set_normal_gold","got bad clienttag");
	return;
    }
    sprintf(key,"BNET\\Record\\%s\\0\\gold",clienttag);
    account_set_numattr(account,key,gold);
}
