/*
 * Copyright (C) 1998,1999  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define SERVER_INTERNAL_ACCESS
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <sys/types.h>
#include <signal.h>
#include "compat/signal.h"
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include <netdb.h>
#include "packet.h"
#include "init_protocol.h"
#include "udp_protocol.h"
#include "connection.h"
#include "hexdump.h"
#include "eventlog.h"
#include "message.h"
#include "queue.h"
#include "bnetd.h"
#include "network.h"
#include "prefs.h"
#include "account.h"
#include "tracker.h"
#include "list.h"
#include "adbanner.h"
#include "timer.h"
#include "addr.h"
#include "server.h"


static void quit_sig_handle(int unused);
static void restart_sig_handle(int unused);
static void save_sig_handle(int unused);
static int init_connection(t_connection * c, int udpsock);


static time_t starttime;
static volatile time_t sigexittime=0;
static volatile int do_restart=0;
static volatile int do_save=0;


static void quit_sig_handle(int unused)
{
    if (sigexittime)
	sigexittime -= prefs_get_shutdown_decr();
    else
	sigexittime = time(NULL)+(time_t)prefs_get_shutdown_delay(); /* in five minutes */
}


static void restart_sig_handle(int unused)
{
    do_restart = 1;
}


static void save_sig_handle(int unused)
{
    do_save = 1;
}


static int init_connection(t_connection * c, int udpsock)
{
    int  addlen;
    char connect_class;
    
    /* determine connection class by first character sent by client */
#ifdef HAVE_RECV
    addlen = recv(conn_get_socket(c),&connect_class,sizeof(char),0);
#else
    addlen = recvfrom(conn_get_socket(c),&connect_class,sizeof(char),0,NULL,NULL);
#endif
    
    if (addlen<0 && (errno==EINTR || errno==EAGAIN || errno==EWOULDBLOCK))
	return 0;
    
    /* error occured or connection lost */
    if (addlen<1)
    {
	eventlog(eventlog_level_error,"init_connection","[%d] could not get connection class (closing connection) (recvfrom: %s)",conn_get_socket(c),strerror(errno));
	return -1;
    }
    
    switch (connect_class)
    {
    case CONNECT_CLASS_NORMAL:
	eventlog(eventlog_level_info,"init_connection","[%d] client initiated normal connection",conn_get_socket(c));
	conn_set_state(c,conn_state_connected);
	conn_set_class(c,conn_class_normal);
	
	{
	    struct sockaddr_in caddr;
	    unsigned int       tries,successes;
	    
	    memset(&caddr,0,sizeof(caddr));
	    caddr.sin_family = AF_INET;
	    caddr.sin_port = htons((unsigned short)prefs_get_testport());
	    caddr.sin_addr.s_addr = htonl(conn_get_addr(c));
	    /* send three in case one gets dropped on the way */
	    for (tries=successes=0; successes!=3 && tries<5; tries++)
	    {
	        if (sendto(udpsock,BNET_UDPTEST,BNET_UDPTEST_SIZE,0,(struct sockaddr *)&caddr,sizeof(caddr))!=BNET_UDPTEST_SIZE)
	            eventlog(eventlog_level_error,"init_connection","[%d] failed to send UDPTEST to port %u (attempt %u) (sendto: %s)",conn_get_socket(c),prefs_get_testport(),tries+1,strerror(errno));
		else
		    successes++;
	    }
	}
	
	break;
	
    case CONNECT_CLASS_FILE:
	eventlog(eventlog_level_info,"init_connection","[%d] client initiated file download connection",conn_get_socket(c));
	conn_set_state(c,conn_state_connected);
	conn_set_class(c,conn_class_file);
	
	break;
	
    case CONNECT_CLASS_BOT:
	eventlog(eventlog_level_info,"init_connection","[%d] client initiated chat bot connection",conn_get_socket(c));
	conn_set_state(c,conn_state_connected);
	conn_set_class(c,conn_class_bot);
	
	{
	    char const * const temp="\r\nEnter your account name and password.\r\n\r\nUsername: ";
	    t_packet *         rpacket;
	    
	    if (!(rpacket = packet_create(packet_class_raw)))
		eventlog(eventlog_level_error,"init_connection","could not create rpacket");
	    else
	    {
		packet_append_data(rpacket,temp,strlen(temp));
		queue_push_packet(conn_get_out_queue(c),rpacket);
		packet_del_ref(rpacket);
	    }
	}
	
	break;
	
    default:
	eventlog(eventlog_level_error,"init_connection","[%d] client initiated unknown connection class 0x%02hx (length %d) (closing connection)",conn_get_socket(c),(unsigned short)connect_class,addlen);
	return -1;
    }
    
    return 0;
}    


extern unsigned int server_get_uptime(void)
{
    return (unsigned int)(time(NULL)-starttime);
}


extern int server_process(void)
{
    char                   tempa[32];
    struct timeval         tv;
    t_addrlist *           laddrs;
    t_addr *               curr_laddr;
    t_addr_data            laddr_data;
    t_laddr_info *         laddr_info;
    fd_set                 rfds, wfds;
    int                    highest_fd;
    time_t                 curr_exittime, prev_exittime, prev_savetime, track_time, now;
    unsigned int           syncdelta;
    t_connection *         c;
    t_connection *         nc;
    t_list const * const * save;
    int                    csocket;
    int                    ssocket;
    int                    usocket;
    sigset_t               block_set;
    sigset_t               save_set;
    int                    do_track;
    unsigned int           count;
    
    /* setup the tracking mechanism */
    if (prefs_get_track())
    {
	tracker_set_servers(prefs_get_trackserv_addrs());
	track_time = time(NULL)-prefs_get_track();
	do_track = 1;
    }
    else
	do_track = 0;
    
    syncdelta = prefs_get_user_sync_timer();
    
    if (!(laddrs = addrlist_create(prefs_get_bnetdserv_addrs(),INADDR_ANY,BNETD_SERV_PORT)))
    {
	eventlog(eventlog_level_error,"server_process","could not create server address list from \"%s\"",prefs_get_bnetdserv_addrs());
	goto error_track;
    }
    
    for (curr_laddr=addrlist_get_first(laddrs,&save); curr_laddr; curr_laddr = addrlist_get_next(&save))
    {
	if (!(laddr_info = malloc(sizeof(t_laddr_info))))
	{
	    eventlog(eventlog_level_error,"server_process","could not create a listening socket (socket: %s)",strerror(errno));
	    goto error_listen;
	}
	if ((ssocket = socket(PF_INET,SOCK_STREAM,0))<0)
	{
	    eventlog(eventlog_level_error,"server_process","could not create a listening socket (socket: %s)",strerror(errno));
	    pfree(laddr_info,sizeof(t_laddr_info));
	    goto error_listen;
	}
	if ((usocket = socket(PF_INET,SOCK_DGRAM,0))<0)
	{
	    eventlog(eventlog_level_error,"server_process","could not create UDP socket (socket: %s)",strerror(errno));
	    close(ssocket);
	    pfree(laddr_info,sizeof(t_laddr_info));
	    goto error_listen;
	}
	
	laddr_info->ssocket = ssocket;
	laddr_info->usocket = usocket;
	laddr_data.p = laddr_info;
	if (addr_set_data(curr_laddr,laddr_data)<0)
	{
	    eventlog(eventlog_level_error,"server_process","could not set address data");
	    close(usocket);
	    close(ssocket);
	    pfree(laddr_info,sizeof(t_laddr_info));
	    goto error_listen;
	}
	
	{
	    int val;
	    
	    val = 1;
	    if (setsockopt(ssocket,SOL_SOCKET,SO_REUSEADDR,&val,sizeof(int))<0)
		eventlog(eventlog_level_error,"server_process","could not set socket option SO_REUSEADDR (setsockopt: %s)",strerror(errno));
	    
	    val = 1;
	    if (setsockopt(usocket,SOL_SOCKET,SO_REUSEADDR,&val,sizeof(int))<0)
		eventlog(eventlog_level_error,"server_process","could not set socket option SO_REUSEADDR (setsockopt: %s)",strerror(errno));
	    /* not a fatal error... */
	}
	
	{
	    struct sockaddr_in saddr;
	    
	    memset(&saddr,0,sizeof(saddr));
	    saddr.sin_family = AF_INET;
	    saddr.sin_port = htons(addr_get_port(curr_laddr));
	    saddr.sin_addr.s_addr = htonl(addr_get_ip(curr_laddr));
	    if (bind(ssocket,(struct sockaddr *)&saddr,sizeof(saddr))<0)
	    {
		if (addr_get_addr_str(curr_laddr,tempa,sizeof(tempa))<0)
		    strcpy(tempa,"x.x.x.x:x");
		eventlog(eventlog_level_error,"server_process","could bind to address %s TCP (bind: %s)",tempa,strerror(errno));
		goto error_listen;
	    }
	    
	    memset(&saddr,0,sizeof(saddr));
	    saddr.sin_family = AF_INET;
	    saddr.sin_port = htons(addr_get_port(curr_laddr));
	    saddr.sin_addr.s_addr = htonl(addr_get_ip(curr_laddr));
	    if (bind(usocket,(struct sockaddr *)&saddr,sizeof(saddr))<0)
	    {
		if (addr_get_addr_str(curr_laddr,tempa,sizeof(tempa))<0)
		    strcpy(tempa,"x.x.x.x:x");
		eventlog(eventlog_level_error,"server_process","could bind socket to address %s UDP (bind: %s)",tempa,strerror(errno));
		goto error_listen;
	    }
	}
	
	/* tell socket to listen for connections */
	if (listen(ssocket,LISTEN_QUEUE)<0)
	{
	    eventlog(eventlog_level_error,"server_process","could not listen (listen: %s)",strerror(errno));
	    goto error_listen;
	}
	
	if (addr_get_addr_str(curr_laddr,tempa,sizeof(tempa))<0)
	    strcpy(tempa,"x.x.x.x:x");
	eventlog(eventlog_level_info,"server_process","listening on %s TCP",tempa);
	
	if (fcntl(ssocket,F_SETFL,O_NONBLOCK)<0)
	    eventlog(eventlog_level_error,"server_process","could not set TCP listen socket to non-blocking mode (fcntl: %s)",strerror(errno));
	if (fcntl(usocket,F_SETFL,O_NONBLOCK)<0)
	    eventlog(eventlog_level_error,"server_process","could not set UDP socket to non-blocking mode (fcntl: %s)",strerror(errno));
    }
    
    /* setup signal handlers */
    prev_exittime = sigexittime;
    
    if (sigemptyset(&save_set)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not initialize signal set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    if (sigemptyset(&block_set)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not initialize signal set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    if (sigaddset(&block_set,SIGINT)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not add signal to set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    if (sigaddset(&block_set,SIGHUP)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not add signal to set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    if (sigaddset(&block_set,SIGTERM)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not add signal to set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    if (sigaddset(&block_set,SIGUSR1)<0)
    {
	eventlog(eventlog_level_error,"server_process","could not add signal to set (sigemptyset: %s)",strerror(errno));
	goto error_listen;
    }
    
    {
	struct sigaction quit_action;
	struct sigaction restart_action;
	struct sigaction save_action;

	quit_action.sa_handler = quit_sig_handle;
	if (sigemptyset(&quit_action.sa_mask)<0)
	    eventlog(eventlog_level_error,"server_process","could not initialize signal set (sigemptyset: %s)",strerror(errno));
	quit_action.sa_flags = SA_RESTART;
	
	restart_action.sa_handler = restart_sig_handle;
	if (sigemptyset(&restart_action.sa_mask)<0)
	    eventlog(eventlog_level_error,"server_process","could not initialize signal set (sigemptyset: %s)",strerror(errno));
	restart_action.sa_flags = SA_RESTART;
	
	save_action.sa_handler = save_sig_handle;
	if (sigemptyset(&save_action.sa_mask)<0)
	    eventlog(eventlog_level_error,"server_process","could not initialize signal set (sigemptyset: %s)",strerror(errno));
	save_action.sa_flags = SA_RESTART;

	if (sigaction(SIGINT,&quit_action,NULL)<0) /* control-c */
	    eventlog(eventlog_level_error,"server_process","could not set SIGINT signal handler (sigaction: %s)",strerror(errno));
	if (sigaction(SIGHUP,&restart_action,NULL)<0)
	    eventlog(eventlog_level_error,"server_process","could not set SIGHUP signal handler (sigaction: %s)",strerror(errno));
	if (sigaction(SIGTERM,&quit_action,NULL)<0)
	    eventlog(eventlog_level_error,"server_process","could not set SIGTERM signal handler (sigaction: %s)",strerror(errno));
	if (sigaction(SIGUSR1,&save_action,NULL)<0)
	    eventlog(eventlog_level_error,"server_process","could not set SIGUSR1 signal handler (sigaction: %s)",strerror(errno));
    }
    
    starttime=prev_savetime = time(NULL);
    count = 0;
    
    for (;;)
    {
	if (sigprocmask(SIG_SETMASK,&save_set,NULL)<0)
	    eventlog(eventlog_level_error,"server_process","could not unblock signals");
	/* receive signals here */
	if (sigprocmask(SIG_SETMASK,&block_set,NULL)<0)
	    eventlog(eventlog_level_error,"server_process","could not block signals");
	
	now = time(NULL);
	
	curr_exittime = sigexittime;
	if (curr_exittime && (curr_exittime<=now || connlist_get_length()<1))
	{
	    eventlog(eventlog_level_info,"server_process","the server is shutting down NOW! (%d connections left)",connlist_get_length());
	    break;
	}
	if (prev_exittime!=curr_exittime)
	{
	    char text[MAX_MESSAGE_LEN];
	    
	    sprintf(text,"The server will shut down in %02d:%02d minutes (%d connections left).",((int)(curr_exittime-now))/60,((int)(curr_exittime-now))%60,connlist_get_length());
	    message_send_all(MT_ERROR,NULL,text);
	    eventlog(eventlog_level_info,"server_process","the server will shut down in %02d:%02d minutes (%d connections left)",((int)(curr_exittime-now))/60,((int)(curr_exittime-now))%60,connlist_get_length());
	}
	prev_exittime = curr_exittime;
	
	if (syncdelta && prev_savetime+(time_t)syncdelta<=now)
	{
	    accountlist_save(prefs_get_user_sync_timer());
	    prev_savetime = now;
	}
	
	if (do_track && track_time+(time_t)prefs_get_track()<=now)
	{
	    track_time = now;
	    tracker_send_report(laddrs);
	}
	
	if (do_save)
	{
	    eventlog(eventlog_level_info,"server_process","saving accounts due to signal");
	    accountlist_save(0);
	    do_save = 0;
	}
	
	if (do_restart)
	{
	    eventlog(eventlog_level_info,"server_process","reading configuration files");
	    if (preffile)
	    {
        	if (prefs_load(preffile)<0)
		    eventlog(eventlog_level_error,"server_process","could not parse configuration file");
	    }
	    else
		if (prefs_load(BNETD_DEFAULT_CONF_FILE)<0)
		    eventlog(eventlog_level_error,"server_process","using default configuration");
	    
	    if (eventlog_open(prefs_get_logfile())<0)
		eventlog(eventlog_level_error,"server_process","could not use the file \"%s\" for the eventlog",prefs_get_logfile());
	    
	    /* FIXME: load new network settings */
	    /* FIXME: load new ad banners */
	    
	    accountlist_load_default(); /* FIXME: free old one */
	    
	    /* FIXME: reload channel list */
	    
	    if (adbannerlist_unload()<0)
		eventlog(eventlog_level_error,"server_process","could not unload old adbanner list");
	    if (adbannerlist_load(prefs_get_adfile())<0)
		eventlog(eventlog_level_error,"server_process","could not new adbanner list");
		
	    if (prefs_get_track())
	    {
		tracker_set_servers(prefs_get_trackserv_addrs());
		track_time = time(NULL)-prefs_get_track();
		do_track = 1;
	    }
	    else
		do_track = 0;
	    
	    syncdelta = prefs_get_user_sync_timer();
	    
	    eventlog(eventlog_level_info,"server_process","done reconfiguring");
	    
	    do_restart = 0;
	}
	
	if (++count>=(1000000/BNETD_POLL_INTERVAL)) /* only check once a second */
	{
	    timerlist_check_timers(now);
	    count = 0;
	}
	
	/* loop over all connections to create the sets for select() */
	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	highest_fd = 0;
	
	for (curr_laddr=addrlist_get_first(laddrs,&save); curr_laddr; curr_laddr = addrlist_get_next(&save))
	{
	    laddr_data = addr_get_data(curr_laddr);
	    laddr_info = laddr_data.p;
	    
	    ssocket = laddr_info->ssocket;
	    usocket = laddr_info->usocket;
            
	    FD_SET(ssocket,&rfds);
	    FD_SET(usocket,&rfds);
            
	    if (ssocket>highest_fd)
        	highest_fd = ssocket;
	    if (usocket>highest_fd)
        	highest_fd = usocket;
	}
	
	for (c=connlist_get_first(&save); c; c=connlist_get_next(&save))
	{
            csocket = conn_get_socket(c);
	    
	    if (queue_get_length((t_queue const * const *)conn_get_out_queue(c))>0)
		FD_SET(csocket,&wfds); /* pending output, also check for writeability */
	    FD_SET(csocket,&rfds);
            
	    if (csocket>highest_fd)
        	highest_fd = csocket;
	}
	
	/* always set the select() timeout */
	tv.tv_sec  = 0;
	tv.tv_usec = BNETD_POLL_INTERVAL;
	
	/* find which sockets need servicing */
	switch (select(highest_fd+1,&rfds,&wfds,NULL,&tv))
	{
	case -1: /* error */
	    if (errno!=EINTR)
	        eventlog(eventlog_level_error,"server_process","select failed (select: %s)",strerror(errno));
	case 0: /* timeout and no sockets ready */
	    continue;
	}
	
	/* check for incoming connection */
	if (!curr_exittime) /* don't allow connections while exiting... */
	    for (curr_laddr=addrlist_get_first(laddrs,&save); curr_laddr; curr_laddr = addrlist_get_next(&save))
	    {
		laddr_data = addr_get_data(curr_laddr);
		laddr_info = laddr_data.p;
		
		ssocket = laddr_info->ssocket;
		usocket = laddr_info->usocket;
		
		if (FD_ISSET(ssocket,&rfds))
		{
		    struct sockaddr_in caddr;
		    unsigned int       caddr_len;
		    
		    if (addr_get_addr_str(curr_laddr,tempa,sizeof(tempa))<0)
			strcpy(tempa,"x.x.x.x:x");
		    
		    /* accept the connection */
		    caddr_len = sizeof(caddr);
		    if ((csocket = accept(ssocket,(struct sockaddr *)&caddr,&caddr_len))<0)
		    {
			if (errno==EWOULDBLOCK || errno==ECONNABORTED) /* BSD, POSIX error for aborted connections, SYSV often uses EAGAIN */
			    eventlog(eventlog_level_error,"server_process","client aborted connection on %s (accept: %s)",strerror(errno),tempa);
			else /* EAGAIN can mean out of resources _or_ connection aborted */
			    if (errno!=EINTR)
				eventlog(eventlog_level_error,"server_process","could not accept new connection on %s (accept: %s)",strerror(errno),tempa);
		    }
		    else
		    {
			eventlog(eventlog_level_info,"server_process","[%d] accepted connection from %s:%hu on %s",csocket,inet_ntoa(caddr.sin_addr),ntohs(caddr.sin_port),tempa);
			if (fcntl(csocket,F_SETFL,O_NONBLOCK)<0)
			{
			    eventlog(eventlog_level_error,"server_process","[%d] could not set TCP socket to non-blocking mode (closing connection) (fcntl: %s)",csocket,strerror(errno));
			    close(csocket);
			}
			else
			    if (!(c = conn_create(csocket,ntohl(caddr.sin_addr.s_addr),ntohs(caddr.sin_port))))
			    {
				eventlog(eventlog_level_error,"server_process","[%d] unable to create new connection (closing connection)",csocket);
				close(csocket);
			    }
		    }
		}
		
		if (FD_ISSET(usocket,&rfds))
		{
		    t_packet *         upacket;
		    struct sockaddr_in fromaddr;
		    int                fromlen;
		    int                len;
		    
		    if (!(upacket = packet_create(packet_class_raw)))
			eventlog(eventlog_level_error,"server_process","could not allocate raw packet for input");
		    else
		    {
			fromlen = sizeof(fromaddr);
			if ((len = recvfrom(usocket,packet_get_raw_data_build(upacket,0),MAX_PACKET_SIZE,0,(struct sockaddr *)&fromaddr,&fromlen))<0)
			{
			    if (errno!=EINTR && errno!=EAGAIN && errno!=EWOULDBLOCK)
				eventlog(eventlog_level_error,"server_process","could not recv UDP datagram (recvfrom: %s)",strerror(errno));
			}
			else
			{
			    if (fromaddr.sin_family!=AF_INET)
				eventlog(eventlog_level_error,"server_process","got UDP datagram with bad address family %d",fromaddr.sin_family);
			    else
			    {
				eventlog(eventlog_level_debug,"server_process","got UDP datagram len=%d from=%s:%hu",len,inet_ntoa(fromaddr.sin_addr),ntohs(fromaddr.sin_port));
				
				packet_set_size(upacket,len);
				
				if (hexstrm)
				{
				    if (addr_get_addr_str(curr_laddr,tempa,sizeof(tempa))<0)
					strcpy(tempa,"x.x.x.x:x");
				    fprintf(hexstrm,"%d: recv class=UDP from=%s:%hu to=%s length=%d\n",
					    usocket,
					    inet_ntoa(fromaddr.sin_addr),
					    ntohs(fromaddr.sin_port),
					    tempa,
					    len);
				    hexdump(hexstrm,packet_get_raw_data(upacket,0),len);
				}
				
				/* FIXME: enqueue */
			    }
			}
			packet_del_ref(upacket);
		    }
		}
	    }
	
	/* search connections for sockets that need service */
	for (c=connlist_get_first(&save); c; c=nc)
	{
	    unsigned int currsize;
	    t_packet *   packet;
	    
	    nc = connlist_get_next(&save); /* in case we destroy c, get the next now */
	    
	    if (conn_get_state(c)==conn_state_destroy)
	    {
		conn_destroy(c);
		continue;
	    }
            csocket = conn_get_socket(c);
	    
	    if (FD_ISSET(csocket,&rfds))
	    {
		if (conn_get_state(c)==conn_state_initial)
		{
		    if (init_connection(c,usocket)<0)
		    {
			conn_destroy(c);
			continue;
		    }
		}
		else
		{
		    currsize = conn_get_in_size(c);
		    
		    if (!*conn_get_in_queue(c))
		    {
			switch (conn_get_class(c))
			{
			case conn_class_normal:
			    if (!(packet = packet_create(packet_class_normal)))
			    {
				eventlog(eventlog_level_error,"server_process","could not allocate normal packet for input");
				continue;
			    }
			    break;
			case conn_class_file:
			    if (!(packet = packet_create(packet_class_file)))
			    {
				eventlog(eventlog_level_error,"server_process","could not allocate file packet for input");
				continue;
			    }
			    break;
			case conn_class_bot:
			    if (!(packet = packet_create(packet_class_raw)))
			    {
				eventlog(eventlog_level_error,"server_process","could not allocate raw packet for input");
				continue;
			    }
			    packet_set_size(packet,1); /* start by only reading one char */
			    break;
			default:
			    eventlog(eventlog_level_error,"server_process","[%d] connection has bad class (closing connection)",conn_get_socket(c));
			    conn_destroy(c);
			    continue;
			}
			queue_push_packet(conn_get_in_queue(c),packet);
			packet_del_ref(packet);
			if (!*conn_get_in_queue(c))
			    continue; /* push failed */
			currsize = 0;
		    }
		    
		    packet = queue_peek_packet((t_queue const * const *)conn_get_in_queue(c)); /* avoid warning */
		    switch (net_recv_packet(csocket,packet,&currsize))
		    {
		    case -1:
			conn_destroy(c);
			continue;
			
		    case 0: /* still working on it */
			conn_set_in_size(c,currsize);
			break;
			
		    case 1: /* done reading */
			if (conn_get_class(c)==conn_class_bot &&
			    currsize<MAX_PACKET_SIZE) /* if we overflow, we can't wait for the end of the line.
							 handle_packet() should take care of it */
			{
			    char const * const temp=packet_get_raw_data_const(packet,0);
			    
			    if (temp[currsize-1]!='\r' && temp[currsize-1]!='\n')
			    {
				conn_set_in_size(c,currsize);
				packet_set_size(packet,currsize+1);
			        break; /* no end of line, get another char */
			    }
			    /* got a complete line... fall through */
			}
			
			packet = queue_pull_packet(conn_get_in_queue(c));
			
			if (hexstrm)
			{
			    fprintf(hexstrm,"%d: recv class=%s[0x%04hx] type=%s[0x%04hx] length=%hu\n",
				    csocket,
				    packet_get_class_str(packet),packet_get_class(packet),
				    packet_get_type_str(packet,packet_dir_from_client),packet_get_type(packet),
				    packet_get_size(packet));
			    hexdump(hexstrm,packet_get_raw_data_const(packet,0),packet_get_size(packet));
			}
			
			if (conn_get_class(c)==conn_class_bot) /* NUL terminate the line to make life easier */
			{
			    char * const temp=packet_get_raw_data(packet,0);
			    
			    if (temp[currsize-1]=='\r' || temp[currsize-1]=='\n')
				temp[currsize-1] = '\0'; /* have to do it here instead of above so everything
							    is intact for the hexdump */
			}
			
			if (handle_packet(c,packet)<0)
			{
			    packet_del_ref(packet);
			    conn_destroy(c);
			    continue;
			}
			
			packet_del_ref(packet);
			conn_set_in_size(c,0);
		    }
		}
	    }
	    
	    if (FD_ISSET(csocket,&wfds))
	    {
		currsize = conn_get_out_size(c);
		switch (net_send_packet(csocket,queue_peek_packet((t_queue const * const *)conn_get_out_queue(c)),&currsize)) /* avoid warning */
		{
		case -1:
		    conn_destroy(c);
		    continue;
		    
		case 0: /* still working on it */
		    conn_set_out_size(c,currsize);
		    break;
		    
		case 1: /* done sending */
		    packet = queue_pull_packet(conn_get_out_queue(c));
		    
		    if (hexstrm)
		    {
			fprintf(hexstrm,"%d: send class=%s[0x%04hx] type=%s[0x%04hx] length=%hu\n",
				csocket,
				packet_get_class_str(packet),packet_get_class(packet),
				packet_get_type_str(packet,packet_dir_from_server),packet_get_type(packet),
				packet_get_size(packet));
		        hexdump(hexstrm,packet_get_raw_data(packet,0),packet_get_size(packet));
		    }
		    
		    packet_del_ref(packet);
		    conn_set_out_size(c,0);
		}
	    }
	}
    }
    
    for (c=connlist_get_first(&save); c; c=nc)
    {
	nc = connlist_get_next(&save); /* since we will destroy c, get the next now */
        conn_destroy(c);
    }
    
    for (curr_laddr=addrlist_get_first(laddrs,&save); curr_laddr; curr_laddr = addrlist_get_next(&save))
    {
	laddr_data = addr_get_data(curr_laddr);
	laddr_info = laddr_data.p;
	
	if (laddr_info)
	{
	    close(laddr_info->usocket);
	    close(laddr_info->ssocket);
	    pfree(laddr_info,sizeof(t_laddr_info));
	}
    }
    addrlist_destroy(&laddrs);
    
    tracker_set_servers(NULL);
    
    return 0;
    
    /************************************************************************/
    
    /* error cleanup */
    
    error_listen:
	for (curr_laddr=addrlist_get_first(laddrs,&save); curr_laddr; curr_laddr = addrlist_get_next(&save))
	{
	    laddr_data = addr_get_data(curr_laddr);
	    laddr_info = laddr_data.p;
	    
	    if (laddr_info)
	    {
		close(laddr_info->usocket);
		close(laddr_info->ssocket);
		pfree(laddr_info,sizeof(t_laddr_info));
	    }
	}
	addrlist_destroy(&laddrs);
	
    error_track:
	tracker_set_servers(NULL);
    
    return -1;
}
