/* BOGL - Ben's Own Graphics Library.
   This file is by Edmund GRIMLEY EVANS <edmundo@rano.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA. */

/*
 * This implements a simple text console whose capatilities are
 * described by the terminfo source in "bterm.ti".
 */

#include "bogl.h"
#include "bogl-term.h"

#define MAX_CCHARS 5

struct bogl_term *bogl_term_new(struct bogl_font *font)
{
  struct bogl_term *term;
  int i;

  term = malloc(sizeof(struct bogl_term));
  if (!term)
    return 0;

  term->font = font;
  term->xbase = term->ybase = 0;
  term->xstep = bogl_font_glyph(font, ' ', 0);
  term->ystep = bogl_font_height(font);
  if (term->xstep <= 0 || term->ystep <= 0) {
    free(term);
    return 0;
  }

  term->xsize = bogl_xres / term->xstep;
  term->ysize = bogl_yres / term->ystep;
  term->xpos = 0, term->ypos = 0;
  term->fg = term->def_fg = 0;
  term->bg = term->def_bg = 7;
  term->reverse_mode = 0;
  term->state = 0;
  term->cur_visible = 1;
  mbrtowc(0, 0, 0, &term->ps);

  term->screen = malloc(term->xsize * term->ysize * sizeof(wchar_t));
  term->screenfg = malloc(term->xsize * term->ysize * sizeof(int));
  term->screenbg = malloc(term->xsize * term->ysize * sizeof(int));
  term->screenul = malloc(term->xsize * term->ysize * sizeof(int));
  term->cchars = malloc(term->xsize * term->ysize * sizeof(wchar_t *));
  if (!term->screen || !term->screenfg || !term->screenbg || !term->screenul || !term->cchars) {
    free(term->screen);
    free(term->screenfg);
    free(term->screenbg);
    free(term->screenul);
    free(term->cchars);
    free(term);
    return 0;
  }
  for (i = 0; i < term->xsize * term->ysize; i++) {
    term->screen[i] = ' ';
    term->screenfg[i] = term->def_fg;
    term->screenbg[i] = term->def_bg;
    term->screenul[i] = 0;
    term->cchars[i] = 0;
  }
  term->yorig = 0;

  return term;
}

#define XPOS(x) (term->xbase + term->xstep * (x))
#define YPOS(y) (term->ybase + term->ystep * (y))
#define SCR(x, y) \
((x) + (((y) + term->yorig) % term->ysize) * term->xsize)

static void
cursor_down (struct bogl_term *term)
{
    int i;

    if (term->ypos < term->ysize - 1) {
        ++term->ypos;
        return;
    }

    ++term->yorig;
    for (i = 0; i < term->xsize; i++)
    {
        int p = SCR(i, term->ypos);

        term->screen[p] = ' ';
        term->screenfg[p] = term->fg;
        term->screenbg[p] = term->bg;
        term->screenul[p] = 0;
        free (term->cchars[p]);
        term->cchars[p] = 0;
    }

    /* If we had a bogl_move or bogl_scroll we could use it here. */
    bogl_term_redraw (term);
}

static void
put_char (struct bogl_term *term, int x, int y, wchar_t wc, wchar_t *cchars, int fg, int bg, int ul)
{
    char buf[MB_LEN_MAX];
    int j, k, r, w;

    wctomb(0, 0);
    if ((k = wctomb(buf, wc)) == -1)
        return;

    if (bogl_in_font (term->font, wc))
    {
        bogl_text (XPOS(x), YPOS(y), buf, k, fg, bg, ul, term->font);

        if (cchars)
            for (j = 0; j < MAX_CCHARS && cchars[j]; j++)
            {
                wctomb(0, 0);
                if ((k = wctomb(buf, cchars[j])) != -1)
                bogl_text(XPOS(x), YPOS(y), buf, k, fg, -1, ul, term->font);
            }
    }
    else
    {
        /* repeat the default char w times */
        w = wcwidth(wc);
        for (r = 0; r < w; r++)
        bogl_text (XPOS(x + r), YPOS(y), buf, k, fg, bg, ul, term->font);
    }
}

static void
show_cursor (struct bogl_term *term, int show)
{
    int i, x, fg, bg;

    if ((x = term->xpos) == term->xsize)
        x = term->xsize - 1;

    i = SCR(x, term->ypos);

    while (!term->screen[i])
        --i, --x;

    if (term->screen[i])
    {
        if (show)
            fg = term->screenbg[i], bg = term->screenfg[i];
        else
            fg = term->screenfg[i], bg = term->screenbg[i];

        put_char (term, x, term->ypos, term->screen[i], term->cchars[i], fg, bg, term->screenul[i]);
    }
}

static void
clear_left (struct bogl_term *term)
{
    int j, i = SCR(term->xpos, term->ypos);
    if (!term->screen[i])
    {
        for (j = i - 1; !term->screen[j]; j--)
            term->screen[j] = ' ';

        term->screen[j] = ' ';

        bogl_clear (XPOS(term->xpos + j - i), YPOS(term->ypos), XPOS(term->xpos), YPOS(term->ypos + 1), term->screenbg[j]);
    }
}

static void
clear_right (struct bogl_term *term)
{
  int j, i = SCR(term->xpos, term->ypos);

  for (j = 0; term->xpos + j < term->xsize && !term->screen[i + j]; j++)
    term->screen[i + j] = ' ';
  if (j)
    bogl_clear(XPOS(term->xpos), YPOS(term->ypos), XPOS(term->xpos + j), YPOS(term->ypos + 1), term->screenbg[i]);
}

void
bogl_term_out (struct bogl_term *term, char *s, int n)
{
    wchar_t wc;
    int i, j, k, kk, w, txp;
    char buf[MB_LEN_MAX];

    if (term->cur_visible)
        show_cursor (term, 0);

    for (; (k = mbrtowc (&wc, s, n, &term->ps)) >= 0; s += k, n -= k)
    {
        if (!k)
            k = 1;
#if 0
        {
            char buf[20];

            sprintf (buf, " %d", wc);
            write (2, buf, strlen (buf));
        }
#endif
        txp = term->xp;
        term->xp = -1;

        if (wc == 0)            /* 0 has a special meaning in term->screen[] */
            continue;

        if (wc == 8)
        {                       /* cub1=^H */
            if (term->xpos)
                --term->xpos;
            term->state = 0;
            continue;
        }

        if (wc == 9)
        {                       /* ht=^I */
            /* I'm not sure whether this way of going over the right margin
               is correct, so I don't declare this capability in terminfo. */
            term->xpos = (term->xpos / 8) * 8 + 8;
            if (term->xpos >= term->xsize)
            {
                term->xpos = 0;
                cursor_down (term);
            }
            term->state = 0;
            continue;
        }

        if (wc == 10)
        {                       /* ind=^J */
            cursor_down (term);
            term->state = 0;
            continue;
        }

        if (wc == 13)
        {                       /* cr=^M */
            term->xpos = 0;
            term->state = 0;
            continue;
        }
        if (wc == 27)
        {                       /* ESC = \E */
            term->state = -1;
            continue;
        }

        if (term->state == -1)
        {
            if (wc == '[')
            {
                term->state = 1;
                term->arg[0] = 0;
                continue;
            }
            term->state = 0;
            continue;
        }

        if (term->state > 0)
        {
            if ('0' <= wc && wc <= '9')
            {
                if (term->state > sizeof (term->arg) / sizeof (int))
                    continue;

                term->arg[term->state - 1] *= 10;
                term->arg[term->state - 1] += wc - '0';
                continue;
            }
            if (wc == ';')
            {
                if (term->state > sizeof (term->arg) / sizeof (int))
                    continue;
                if (term->state < sizeof (term->arg) / sizeof (int))
                    term->arg[term->state] = 0;

                ++term->state;
                continue;
            }

            if (wc == '?')
            {
                if (term->state == 1 && term->arg[0] == 0)
                    ++term->state, term->arg[0] = -1, term->arg[1] = 0;
                else
                    term->state = 0;
                continue;
            }

            if (wc == 'H')
            {                   /* home=\E[H, cup=\E[%i%p1%d;%p2%dH */
                if (term->state < 3)
                {
                    if (term->state == 2)
                        if (term->arg[1] <= term->xsize)
                            term->xpos = term->arg[1] ? term->arg[1] - 1 : 0;
                    if (term->arg[0] <= term->ysize)
                        term->ypos = term->arg[0] ? term->arg[0] - 1 : 0;
                }
                term->state = 0;
                continue;
            }

            if (wc == 'J')
            {                   /* clear=\E[H\E[2J */
                if (term->arg[0])
                {
                    bogl_clear (XPOS (0), YPOS (0),
                                XPOS (term->xsize), YPOS (term->ysize),
                                term->bg);
                    for (i = 0; i < term->xsize * term->ysize; i++)
                    {
                        term->screen[i] = ' ';
                        term->screenfg[i] = term->fg;
                        term->screenbg[i] = term->bg;
                        term->screenul[i] = 0;
                        free (term->cchars[i]);
                        term->cchars[i] = 0;
                    }
                }
                term->state = 0;
                continue;
            }

            if (wc == 'K')
            {                   /* el=\E[K */
                if (term->state == 1 && !term->arg[0])
                {
                    bogl_clear (XPOS (term->xpos), YPOS (term->ypos),
                                XPOS (term->xsize), YPOS (term->ypos + 1),
                                term->bg);
                    clear_left (term);
                    for (i = SCR (term->xpos, term->ypos); i < SCR (term->xsize, term->ypos); i++)
                    {
                        term->screen[i] = ' ';
                        term->screenfg[i] = term->fg;
                        term->screenbg[i] = term->bg;
                        term->screenul[i] = 0;
                        free (term->cchars[i]);
                        term->cchars[i] = 0;
                    }
                }
                term->state = 0;
                continue;
            }

            if (wc == 'h')
            {                   /* cnorm=\E[?25h */
                if (term->arg[0] == -1 && term->arg[1] == 25)
                    term->cur_visible = 1;
                term->state = 0;
                continue;
            }

            if (wc == 'l')
            {                   /* civis=\E[?25l */
                if (term->arg[0] == -1 && term->arg[1] == 25)
                    term->cur_visible = 0;

                term->state = 0;
                continue;
            }

            if (wc == 'm')
            {                   /* setab=\E[4%p1%dm, setaf=\E[3%p1%dm */
                if (term->arg[0] == 4 || term->arg[0] == 24)
                    term->ul = term->arg[0] == 4;
                else if (30 <= term->arg[0] && term->arg[0] < 38)
		{
		    if(!term->reverse_mode)
			term->fg = term->arg[0] - 30;
		    else
			term->bg = term->arg[0] - 30;
		}
                else if (40 <= term->arg[0] && term->arg[0] < 48)
                {
		    if(!term->reverse_mode)
			term->bg = term->arg[0] - 40;
		    else
			term->fg = term->arg[0] - 40;
                }
                else if (term->arg[0] == 39)
                {
		    if(!term->reverse_mode)
			term->fg = term->def_fg;
		    else
			term->bg = term->def_fg;
                }
                else if (term->arg[0] == 49)
                {
		    if(!term->reverse_mode)
			term->bg = term->def_bg;
		    else
			term->fg = term->def_fg;
                }
		else if (term->arg[0] == 7)
		{
		    int tempcolor;
		    
		    term->reverse_mode = 1;
		    /* Set color here for safety */
		    tempcolor = term->fg;
		    term->fg = term->bg;
		    term->bg = tempcolor;
		}
		else if (!term->arg[0] || term->arg[0] == 0)
		{
		    term->reverse_mode = 0;
		    term->fg = term->def_fg;
		    term->bg = term->def_bg;
		}
                term->state = 0;
                continue;
            }

            term->state = 0;
            continue;
        }

        if ((w = wcwidth (wc)) < 0)
            continue;

        wctomb (0, 0);
        kk = wctomb (buf, wc);
        if (kk == -1)           /* impossible */
            continue;
#if 0
        {
            write (2, "p", 1);
        }
#endif
        if (w > 0)
        {
            if (w <= term->xsize)
            {

                if (term->xpos + w > term->xsize)
                {
                    bogl_clear (XPOS (term->xpos), YPOS (term->ypos), XPOS (term->xsize), YPOS (term->ypos + 1), term->bg);

                    clear_left (term);

                    for (i = SCR (term->xpos, term->ypos); i < SCR (term->xsize, term->ypos); i++)
                    {
                        term->screen[i] = ' ';
                        term->screenfg[i] = term->fg;
                        term->screenbg[i] = term->bg;
                        term->screenul[i] = 0;
                        free (term->cchars[i]);
                        term->cchars[i] = NULL;
                    }

                    term->xpos = 0;
                    cursor_down (term);
                }

                clear_left (term);
                i = SCR (term->xpos, term->ypos);
                term->screen[i] = wc;
                term->screenfg[i] = term->fg;
                term->screenbg[i] = term->bg;
                term->screenul[i] = term->ul;
                free (term->cchars[i]);
                term->cchars[i] = NULL;

                for (j = 1; j < w; j++)
                {
                    term->screen[i + j] = 0;
                    term->screenfg[i + j] = term->fg;
                    term->screenbg[i + j] = term->bg;
                    term->screenul[i + j] = 0;
                }

                if (bogl_in_font (term->font, wc))
                {
                    bogl_text (XPOS (term->xpos), YPOS (term->ypos), buf, kk, term->fg, term->bg, term->ul, term->font);
                    term->xp = term->xpos, term->yp = term->ypos;
                    term->xpos += w;
                }
                else
                {
                    /* repeat the default char w times */
                    int r;

                    for (r = 0; r < w; r++)
                    {
                        bogl_text (XPOS (term->xpos), YPOS (term->ypos), buf, kk, term->fg, term->bg, term->ul, term->font);
                        term->xp = term->xpos, term->yp = term->ypos;
                        ++term->xpos;
                    }
                }

                clear_right (term);
            }
        }
        else
        {                       /* w == 0 */
            if (txp >= 0)
            {
                term->xp = txp;
                bogl_text (XPOS (term->xp), YPOS (term->yp), buf, kk, term->fg, -1, term->ul, term->font);
            }
            else
            {
                clear_left (term);
                bogl_text (XPOS (term->xpos), YPOS (term->ypos), buf, kk, term->fg, term->bg, term->ul, term->font);
                term->xp = term->xpos, term->yp = term->ypos;
                term->xpos += 1;
                clear_right (term);
            }
        }
    }

    if (term->cur_visible)
        show_cursor (term, 1);
}

void
bogl_term_redraw (struct bogl_term *term)
{
    int x, y, i;

    bogl_clear(0, YPOS(term->ysize), bogl_xres, bogl_yres, 0);
    bogl_clear(XPOS(term->xsize), 0, bogl_xres, YPOS(term->ysize), 0);
    for (y = 0; y < term->ysize; y++)
        for (x = 0; x < term->xsize; x++)
        {
            i = SCR(x, y);
            if (term->screen[i])
                put_char(term, x, y, term->screen[i], term->cchars[i], term->screenfg[i], term->screenbg[i], term->screenul[i]);
        }
    
    if (term->cur_visible)
        show_cursor(term, 1);
}
